//////////////////////////////////////////////////////////////////////
// GT.cpp - Game Trigger
// -------------------------------------------------------------------
//////////////////////////////////////////////////////////////////////
#include "External/D2Hackit.h"
#include "External/D2Client.h"
#include <string.h>
#include <time.h>

//////////////////////////////////////////////////////////////////////
// Version History
//
// Ver.  Date       Details
// -------------------------------------------------------------------
// 0.10  07-Feb-02  Initial code
//                  GT go to town when low in HP
//                  GT quit game when really low in HP
//                  How GT react to hostile player?
//                    - tp and unhostile
//                    - quit game
//                  Go to town thingie copy from PvPBuddy
// 0.20  28-Feb-02  First beta release
//                  - Support 10 HP level trigger
//                  - Support 5 MP level trigger
//                  - New player join game trigger (Hardcode response)
//                  - Hostiled trigger
//                  - allow loot response
// 0.25  11-Mar-02  Remove debug message 
//                  - Party invite trigger (hard code response)
//                  - Fixed info command
// 0.30  29-Mar-02  Fixed MP triggered incorrectly
//                  Add block quit game packet 
//////////////////////////////////////////////////////////////////////

void LoadConfig(void);
void OverheadMsg(char *);
void ChatMsg(char *);

//////////////////////////////////////////////////////////////////////
// CLIENTINFO
//////////////////////////////////////////////////////////////////////
DWORD   ModuleVersion=MAKELONG(0, 30);
char    ModuleAuthor[]="Maulei";
char    ModuleWebsite[]="jonathan.makes.it";
char    ModuleDescription[]="Game Trigger";
char    ModuleEmail[]="jonathanwu@hongkong.com";

//////////////////////////////////////////////////////////////////////
// Function Definition
//////////////////////////////////////////////////////////////////////
DWORD WINAPI ThreadProc(LPVOID lpParameter);
BOOL PRIVATE OnGameCommandSet(char** argv, int argc);
BOOL PRIVATE OnGameCommandReset(char** argv, int argc);
BOOL PRIVATE OnGameCommandDefault(char** argv, int argc);
BOOL PRIVATE OnGameCommandLoot(char** argv, int argc);
BOOL PRIVATE OnGameCommandMaxHP(char** argv, int argc);
BOOL PRIVATE OnGameCommandMaxMP(char** argv, int argc);
BOOL PRIVATE OnGameCommandInfo(char** argv, int argc);
BOOL PRIVATE OnGameCommandList(char** argv, int argc);
BOOL PRIVATE OnGameCommandTest(char** argv, int argc);
BOOL PRIVATE OnGameCommandTime(char** argv, int argc);
BOOL PRIVATE OnGameCommandStart(char** argv, int argc);
BOOL PRIVATE OnGameCommandStop(char** argv, int argc);
void Quit(void);
void ShowMax(void);
int addPlayerInfo(BYTE* aPacket, DWORD aLen);
int removePlayerInfo(BYTE* aPacket, DWORD aLen);

// array size constants
#define MAX_PLAYER_IN_GAME 8

//------------------------------------------------//
// TYPEDEF
//------------------------------------------------//
typedef struct playeringame_t
{
	bool		InUse;
	DWORD		PlayerID;
	char		PlayerName[16];
} PLAYERINGAME, *PPLAYERINGAME;

PLAYERINGAME playerInGame[MAX_PLAYER_IN_GAME];	// store player name & id in current game

//////////////////////////////////////////////////////////////////////
// ClientCore.cpp stuff
//////////////////////////////////////////////////////////////////////
char ConfigPath[1024];
FUNCTIONENTRYPOINTS	*server;
THISGAMESTRUCT *thisgame;

long cur, max, percent;
int CurrentHP, CurrentMP, MaxHP, MaxMP, HPpercent, MPpercent, LastHPTrigger, LastMPTrigger;
BYTE PlayerID;

// Timer to check last update time
time_t Current, LastHPTriggerTime, LastMPTriggerTime;
double interval;
int elapsed, Duration;

// Game Trigger Definition
enum {
	HP0 = 0, HP10, HP20, HP30, HP40, HP50, HP60, HP70, HP80, HP90,
	MP0, MP20, MP40, MP60, MP80,
	Party, NewPlayer, Hostile,
	TotalTriggers
};

char *TriggerName[TotalTriggers+1] = {
	"HP0%", "HP10%", "HP20%", "HP30%", "HP40%", "HP50%", "HP60%", "HP70%", "HP80%", "HP90%",
	"MP0%", "MP20%", "MP40%", "MP60%", "MP80%",
	"Party", "NewPlayer", "Hostile", "All HP/MP Triggers Allowed"
};

char Trigger[TotalTriggers][256];
BOOL Triggered[TotalTriggers], HPUpdate, BlockPacket66, InTrigger;
int StateInParty=0;

// For writing INI file
char _Character[64], _Key[64], _Value[256];
char Msg1[256];

typedef struct thread_data_t
{
	// Thread ID
	HANDLE   Thread;

	// TRUE if in a game
	BOOL  Active;

    // For getting info :)
    THISGAMESTRUCT*   thisgame;

} THREAD_DATA;

THREAD_DATA td;

//////////////////////////////////////////////////////////////////////
// Module Commands
//////////////////////////////////////////////////////////////////////
MODULECOMMANDSTRUCT ModuleCommands[]=
{
	{
		"help",  OnGameCommandHelp,
		"help c0 List commands available in this module.\n"
		"<command> help c0 Shows detailed help for <command> in this module."
	},
	{
		"reset", OnGameCommandReset,
		"resset c0 Clear all game triggers\n"
	},
	{
		"set", OnGameCommandSet,
		"set c0 Setup Game Trigger\n"
	},
	{
		"loot", OnGameCommandLoot,
		"loot c0 Allow all players in game to loot you\n"
	},
	{
		"default", OnGameCommandDefault,
		"default c0 Use default game trigger\n"
	},
	{
		"maxhp", OnGameCommandMaxHP,
		"maxhp c0 Setup Max HP\n"
	},
	{
		"maxmp", OnGameCommandMaxMP,
		"maxmp c0 Setup Max MP\n"
	},
	{
		"time", OnGameCommandTime,
		"time c0 Setup HP/MP trigger duration\n"
	},
	{
		"list", OnGameCommandList,
		"list c0 List Game Triggers\n"
	},
	{
		"info", OnGameCommandInfo,
		"info c0 Show Game Trigger Info\n"
	},
	{
		"test", OnGameCommandTest,
		"test c0 Test Game Triggers\n"
	},
	{
		"start", OnGameCommandStart,
		"start c0 Start or reset Game Trigger\n"
	},
	{
		"stop",  OnGameCommandStop,
		"stop c0 Temporarily stops GT\n"
	},
	{NULL}
};

BOOL EXPORT OnClientStart()
{
	FILE *file;

	// Make sure ini file exists
	sprintf(Msg1, "%s\\%s", ConfigPath, "GT.ini");
	file = fopen(Msg1, "a+");
	fclose(file);

	return TRUE;
}

void EXPORT OnGameJoin(THISGAMESTRUCT* thisgame)
{
	td.thisgame=thisgame;
	LoadConfig();

	// Reset Flags
	HPUpdate = FALSE;
	StateInParty = 0;

	// Allow all HP/MP Triggers
	LastHPTrigger = TotalTriggers;
	LastMPTrigger = TotalTriggers;

	// Reset Timer;
	time(&LastMPTriggerTime);
	time(&LastHPTriggerTime);
	time(&Current);

	for (int i = 0; i < MAX_PLAYER_IN_GAME; i++)
		playerInGame[i].InUse = false;

	// List defined triggers
	server->GameCommandLine("gt list");

	// Start thread
	td.Active=TRUE;
	DWORD dummy=0;
    td.Thread = CreateThread(NULL,0,ThreadProc,&td,0,&dummy);
}

void EXPORT OnGameLeave(THISGAMESTRUCT* thisgame)
{
    td.Active=FALSE;

    // Wait for thread to die
    if (td.Thread)
    {
		server->GamePrintVerbose("Game Trigger: Waiting max 2 seconds for worker thread to exit.");
        WaitForSingleObject(td.Thread, 2000);
        server->GamePrintVerbose("Game Trigger: OK!");
    }
}

//////////////////////////////////////////////////////////////////////
// OnGameCommandStart
//////////////////////////////////////////////////////////////////////
BOOL PRIVATE OnGameCommandStart(char** argv, int argc)
{
    if (argc!=2) return FALSE;

    OnGameLeave(0);
    OnGameJoin(td.thisgame);
    return TRUE;
}

//////////////////////////////////////////////////////////////////////
// OnGameCommandStop
//////////////////////////////////////////////////////////////////////
BOOL PRIVATE OnGameCommandStop(char** argv, int argc)
{
    OnGameLeave(0);
    return TRUE;
}

BOOL PRIVATE OnGameCommandMaxHP(char** argv, int argc)
{
    if (argc == 3)
    {
        MaxHP = atoi(argv[2]);
		sprintf(_Character, "%s", td.thisgame->player->PlayerName);
		sprintf(_Key, "Max HP");
		sprintf(_Value, "%d", MaxHP);
		server->SetHackProfileString("gt", _Character, _Key, _Value);
	}

	if (MaxHP)
		ShowMax();
	else
		OverheadMsg("HP Triggers OFF!");

    return TRUE;
}

BOOL PRIVATE OnGameCommandTime(char** argv, int argc)
{
    if (argc == 3)
    {
        Duration = atoi(argv[2]);
		sprintf(_Character, "%s", td.thisgame->player->PlayerName);
		sprintf(_Key, "Duration");
		sprintf(_Value, "%d", Duration);
		server->SetHackProfileString("gt", _Character, _Key, _Value);
		sprintf(Msg1, "HP/MP trigger last for c1%dc0 sec.", Duration);
		server->GamePrintInfo(Msg1);
		return TRUE;
	}

	server->GamePrintError(".gt time <duration>");
	return TRUE;
}

BOOL PRIVATE OnGameCommandMaxMP(char** argv, int argc)
{
    if (argc == 3)
	{
		MaxMP = atoi(argv[2]);
		sprintf(_Character, "%s", td.thisgame->player->PlayerName);
		sprintf(_Key, "Max MP");
		sprintf(_Value, "%d", MaxMP);
		server->SetHackProfileString("gt", _Character, _Key, _Value);
	}

	if (MaxMP)
		ShowMax();
	else
		OverheadMsg("MP Triggers OFF!");

    return TRUE;
}

void ShowMax(void)
{
	sprintf(Msg1, "Max HP:%d Max MP:%d", MaxHP, MaxMP);
	OverheadMsg(Msg1);
}

BOOL PRIVATE OnGameCommandInfo(char** argv, int argc)
{
	sprintf(Msg1, "Max HP:%d", MaxHP);
	server->GamePrintInfo(Msg1);
	sprintf(Msg1, "Max MP:%d", MaxMP);
	server->GamePrintInfo(Msg1);
	sprintf(Msg1, "HP/MP Trigger Duration:%d secs", Duration);
	server->GamePrintInfo(Msg1);
	return TRUE;
}

BOOL PRIVATE OnGameCommandList(char** argv, int argc)
{
	int i;

	if (argc!=2 && argc !=3) return FALSE;

	server->GamePrintInfo("Defined game triggers:");
	for (i=0; i<TotalTriggers; i++)
	{
		if (Trigger[i][0] != 0x00)
		{
			sprintf(Msg1, "c3%sc0 -> c2%s", TriggerName[i], Trigger[i]);
			server->GamePrintInfo(Msg1);
		}
		else
		{
			sprintf(Msg1, "c3%sc0 -> c1None", TriggerName[i]);
			if (argc == 3) server->GamePrintInfo(Msg1);
		}
	}
	return TRUE;
}

BOOL PRIVATE OnGameCommandTest(char** argv, int argc)
{
	int i;
	BOOL ValidTriggerName;
	int ChosenTrigger, result;

	if (argc !=3) return FALSE;

	ValidTriggerName = FALSE;
	for (i=0; i<TotalTriggers; i++)
	{
		result = _stricmp(argv[2], TriggerName[i]);
		if (result == 0)
		{
			ValidTriggerName = TRUE;
			ChosenTrigger = i;
		}
	}

	if (!ValidTriggerName)
	{
		sprintf(Msg1, "Game trigger 'c1%sc0' is not defined.", argv[2]);
		server->GamePrintError(Msg1);
		return TRUE;
	}

	server->GameCommandLine(Trigger[ChosenTrigger]);

	return TRUE;
}

BOOL PRIVATE OnGameCommandReset(char** argv, int argc)
{
	for (int i=0; i<TotalTriggers; i++)
		strcpy(Trigger[i], "");

	server->GamePrintInfo("Game triggers cleared.");

	return TRUE;
}

BOOL PRIVATE OnGameCommandDefault(char** argv, int argc)
{
	int i;

	// Clear defined triggers
	for (i=0; i<TotalTriggers; i++)
		strcpy(Trigger[i], "");

	// Setup default triggers
	server->GameCommandLine("gt set hp0%  say WTF! I'm killed! =(;gt loot");
	server->GameCommandLine("gt set hp10% say Wow that was close!  I'll be back!;send 66");
	server->GameCommandLine("gt set hp20% say Ouch! Back to town first!;tr tp go");
	server->GameCommandLine("gt set hp30% tr potion r");
	server->GameCommandLine("gt set hp50% tr potion h");
	server->GameCommandLine("gt set mp0%  tr potion m");
	server->GameCommandLine("gt set mp20% tr potion m");
	server->GameCommandLine("gt set hostile say WTF! I hate PK!;overhead Back to town!;tr tp go");
	server->GameCommandLine("gt set newplayer x");
	server->GameCommandLine("gt set party x");
	server->GameCommandLine("gt time 5");

	return TRUE;
}

BOOL PRIVATE OnGameCommandLoot(char** argv, int argc)
{
	BYTE pPacket[] = {0x5d, 0x01, 0x01, 0xFF, 0x00, 0x00, 0x00};
	for (int i=0; i < MAX_PLAYER_IN_GAME; i++)
	{
		if (playerInGame[i].InUse == true)
		{
			pPacket[3] = i;
			server->GameSendPacketToServer(pPacket, 7);
		}
	}
	server->GamePrintInfo("Loot is enabled for all players");

	return TRUE;
}

BOOL PRIVATE OnGameCommandSet(char** argv, int argc)
{
	BOOL ValidTriggerName;
	int ChosenTrigger, result;

	// If command line is shorter than 3 tokens, something is wrong
	if (argc < 3) return FALSE;

	char t[256];	// Should be enough...

	ValidTriggerName = FALSE;
	for (int i=0; i<TotalTriggers; i++)
	{
		result = _stricmp(argv[2], TriggerName[i]);
		if (result == 0)
		{
			ValidTriggerName = TRUE;
			ChosenTrigger = i;
		}
	}

	if (!ValidTriggerName)
	{
		sprintf(Msg1, "Game trigger '%s' is not defined.", argv[2]);
		server->GamePrintError(Msg1);
		return TRUE;
	}

	// Clear trigger
	strcpy(Trigger[ChosenTrigger], "");

	if (argc > 3)
	{
		strcpy(t,argv[3]);
		strcat(t, " ");

		for (int i=4; i!=argc; i++)
		{
			strcat(t, argv[i]);
			strcat(t, " ");
		}
		strcpy(Trigger[ChosenTrigger],t);

		// Force these two triggers to use default action 'description'
		if (ChosenTrigger == Party)
			strcpy(Trigger[ChosenTrigger], "<Join party>");
		if (ChosenTrigger == NewPlayer)
			strcpy(Trigger[ChosenTrigger], "<Invite to party>");
	}

	sprintf(Msg1, "Game Trigger [c3%sc0] -> ", TriggerName[ChosenTrigger]);

	if (Trigger[ChosenTrigger][0] != 0x00)
		sprintf(Msg1, "%s %s", Msg1, Trigger[ChosenTrigger]);
	else
		strcat(Msg1, "c1No action");

	// Replace ';' with '|' to support Multi-command line
	for (i=0; i<strlen(Trigger[ChosenTrigger]); i++)
	{
		if (Trigger[ChosenTrigger][i] == ';')
			Trigger[ChosenTrigger][i] = '|';
	}


	// Write to INI file
	sprintf(_Character, "%s", td.thisgame->player->PlayerName);
	sprintf(_Key, "%s", TriggerName[ChosenTrigger]);
	sprintf(_Value, "%s", Trigger[ChosenTrigger]);
	server->SetHackProfileString("gt", _Character, _Key, _Value);

	server->GamePrintVerbose(Msg1);
	return TRUE;

}

DWORD EXPORT OnGamePacketBeforeReceived(BYTE* aPacket, DWORD aLen)
{
	// World interaction
	if (aPacket[0] == 0x13 && BlockPacket66)
	{
		server->GamePrintInfo("Packet 66 unblocked.");
		BlockPacket66 = FALSE;
		return aLen;
	}

	// Capture Hostile Packet
	if (aPacket[0] == 0x8c && aPacket[5] == td.thisgame->player->PlayerID && aPacket[9] & 0x08)
	{
		Triggered[Hostile] = TRUE;
		return aLen;
	}

	// HP, MP
	if (aPacket[0] == 0x95)
	{
		CurrentHP = *((WORD *)(aPacket + 1)) & 0x0FFF;
		CurrentMP = (*((WORD *)(aPacket + 3)) & 0x0FFF) << 1;
		HPUpdate = TRUE;
		sprintf(Msg1, "HP:%d/%d (%d%%) MP:%d/%d (%d%%)", CurrentHP, MaxHP, HPpercent, CurrentMP, MaxMP, MPpercent);
		//server->GamePrintInfo(Msg1);
		if (Trigger[HP0][0] == 'x') OverheadMsg(Msg1);
	}

	//------------------------------------------------//
	// Packet 5a 07 party - from wanderer
	///////////////////////
	//Automaticly join any party invite
	if ( (aPacket[0] == 0x5a) &&
		 (aPacket[1] == 0x07) &&
		 (aPacket[2] == 0x02) &&
		 (aPacket[7] == 0x05) &&
		 StateInParty == 0 )
	{
		if (Trigger[Party][0] != 0x00)
		{
			BYTE pPacket[] = {0x5e, 0x08, 0, 0, 0, 0};
			pPacket[2] = aPacket[3];
			server->GameSendPacketToServer(pPacket, 6);
			StateInParty = 1;
		}
		return aLen;
	}

	//------------------------------------------------//
	// Packet 59 - Player info update
	///////////////////////
	// New Player encountered - store information
	if(aPacket[0] == 0x59)
	{
		int result = addPlayerInfo(aPacket, aLen);
		return aLen;
	}

	//-----------------------------------------------//
	// Packet 8b - Thanks to mike1572 for packet info
	/////////////////////
	// Invite new player to join party
	if (aPacket[0] == 0x8b && aPacket[5] == 0x00)
	{
		if (Trigger[NewPlayer][0] != 0x00)
		{
			BYTE pPacket[] = {0x5e, 0x06, 0, 0, 0, 0};
			pPacket[2] = aPacket[3];
			pPacket[3] = aPacket[4];
			pPacket[4] = aPacket[5];
			pPacket[5] = aPacket[6];
			server->GameSendPacketToServer(pPacket, 6);
			return aLen;
		}
	}

	//------------------------------------------------//
	// Packet 5a 03 - player leaves game notification
	///////////////////////
	// player leave
	if ( (aPacket[0] == 0x5a) && (aPacket[1] == 0x03) )
	{
		int result = removePlayerInfo(aPacket, aLen);
		return aLen;
	}

    return aLen;
}

DWORD EXPORT OnGamePacketBeforeSent(BYTE* aPacket, DWORD aLen)
{
	if (InTrigger && aPacket[0] == 0x66)
	{
		if (BlockPacket66)
		{
			server->GamePrintInfo("Packet 66 blocked, interact with any NPC to unblock.");
			return 0;
		}
		BlockPacket66 = TRUE;
	}
	return aLen;
}

DWORD WINAPI ThreadProc(LPVOID lpParameter)
{
	int i;

	THREAD_DATA *td=(THREAD_DATA*)lpParameter;

	while (td->Active)
    {
		InTrigger = TRUE;
		// Process HP trigger
		if  (MaxHP && HPUpdate)
		{
			time(&Current);
			interval = difftime(Current, LastHPTriggerTime);
    		elapsed = interval;

			// If HP trigger expired, allow any HP trigger
			if (elapsed +1 > Duration)
				LastHPTrigger = TotalTriggers;

			// Calculate HP percentage
			cur = CurrentHP;
			max = MaxHP;
			HPpercent = cur * 100 / max;

			// Debug message
			sprintf(Msg1, "HP %d/%d(%d%%) LastHPTrigger:%s LastMPTrigger:%s Elapsed:%d Duration:%d", CurrentHP, MaxHP, HPpercent, TriggerName[LastHPTrigger], TriggerName[LastMPTrigger], elapsed, Duration);
			if (Trigger[HP0][0] == 'x') OverheadMsg(Msg1);

			for (i=HP0; i<=HP90; i++)
			{
				if (HPpercent < i * 10 && i < LastHPTrigger)
				{
					if (Trigger[i][0] != 0x00)
					{
						server->GameCommandLine(Trigger[i]);
						sprintf(Msg1, "%s triggered", TriggerName[i]);
						server->GamePrintInfo(Msg1);
						time(&LastHPTriggerTime);
						LastHPTrigger = i;
						i = TotalTriggers;
					}
				}
			}
		}

		// Process MP trigger
		if  (MaxMP && HPUpdate)
		{
			time(&Current);
			interval = difftime(Current, LastMPTriggerTime);
    		elapsed = interval;

			// If MP trigger expired, allow any MP trigger
			if (elapsed +1 > Duration)
				LastMPTrigger = TotalTriggers;

			// Calculate MP percentage
			cur = CurrentMP;
			max = MaxMP;
			MPpercent = cur * 100 / max;

			// Debug message
			sprintf(Msg1, "MP %d%% LastMPTrigger:%s", MPpercent, TriggerName[LastMPTrigger]);
			if (Trigger[MP0][0] == 'x') OverheadMsg(Msg1);

			for (i=MP0; i<=MP80; i++)
			{
				if (MPpercent < (i-MP0+1) * 20 && i < LastMPTrigger)
				{
					if (Trigger[i][0] != 0x00)
					{
						server->GameCommandLine(Trigger[i]);
						sprintf(Msg1, "%s triggered", TriggerName[i]);
						server->GamePrintInfo(Msg1);
						time(&LastMPTriggerTime);
						LastMPTrigger = i;
						i = TotalTriggers;
					}
				}
			}
		}

		// Process Misc triggers
		for (i=Hostile; i<TotalTriggers; i++)
		{
			if (Triggered[i])
			{
				if (Trigger[i][0] != 0x00)
				{
					server->GameCommandLine(Trigger[i]);
				}
				Triggered[i] = FALSE;
			}
		}
		InTrigger = FALSE;
		Sleep(250);

		if (!td->Active)
		{
			td->Thread=0;
			return 0;
		}
	}

	td->Thread=0;
	return 0;
}

void LoadConfig()
{
	char *retval;
	int i;

	// Character specific Config
	sprintf(_Character, "%s", td.thisgame->player->PlayerName);

	for (i=0; i<TotalTriggers; i++)
    {
		Triggered[i] = FALSE;
		sprintf(_Key, "%s", TriggerName[i]);
		retval = server->GetHackProfileString("gt", _Character, _Key);
		if (strlen(retval))
			strcpy(Trigger[i], retval);
		else
		{
			strcpy(Trigger[i], "");
			sprintf(_Value, "");
			server->SetHackProfileString("gt", _Character, _Key, _Value);
		}
	}

	// Max HP
	sprintf(_Key, "%s", "Max HP");
	retval = server->GetHackProfileString("gt", _Character, _Key);

	if (strlen(retval))
		MaxHP = atoi(retval);
	else
	{
		MaxHP = 0;
		sprintf(_Value, "%d", 0);
		server->SetHackProfileString("gt", _Character, _Key, _Value);
	}

	// Max MP
	sprintf(_Key, "%s", "Max MP");
	retval = server->GetHackProfileString("gt", _Character, _Key);

	if (strlen(retval))
		MaxMP = atoi(retval);
	else
	{
		MaxMP = 0;
		sprintf(_Value, "%d", 0);
		server->SetHackProfileString("gt", _Character, _Key, _Value);
	}

	// HP/MP Trigger Duration
	sprintf(_Key, "%s", "Duration");
	retval = server->GetHackProfileString("gt", _Character, _Key);

	if (strlen(retval))
		Duration = atoi(retval);
	else
	{
		Duration = 5;
		sprintf(_Value, "%d", 5);
		server->SetHackProfileString("gt", _Character, _Key, _Value);
	}
}

//////////////////////////////////////////////////////////////////////
// addPlayerInfo
// -------------------------------------------------------------------
// store player name and id
//////////////////////////////////////////////////////////////////////
int addPlayerInfo(BYTE* aPacket, DWORD aLen)
{
	int i = 0;
	int freeIndex = -1;
	for(i = 0; i < MAX_PLAYER_IN_GAME; i++)
	{
		// does player already exist in table?
		if (playerInGame[i].InUse == true &&
			strcmpi(playerInGame[i].PlayerName, (char*) aPacket+6) == 0)
			return i;
		else if(playerInGame[i].InUse == false && freeIndex == -1)
			freeIndex = i; // locate a free slot
	}

	if (freeIndex < 0) // more than 8 player impossible, but just in case
		return (-1) ;

	playerInGame[freeIndex].PlayerID = aPacket[1];
	strcpy(playerInGame[freeIndex].PlayerName, (const char*) aPacket+6);
	playerInGame[freeIndex].InUse = true;

	sprintf(Msg1,"Player %d, %s", playerInGame[freeIndex].PlayerID,
									  playerInGame[freeIndex].PlayerName);
	server->GamePrintString(Msg1);
	return freeIndex;
}

//////////////////////////////////////////////////////////////////////
// removePlayerInfo
// -------------------------------------------------------------------
// remove by setting the InUse flag to false
//////////////////////////////////////////////////////////////////////
int removePlayerInfo(BYTE* aPacket, DWORD aLen)
{
	int i = 0;

	for(i = 0; i < MAX_PLAYER_IN_GAME; i++)
	{
		// find player in table
		if (playerInGame[i].InUse == true &&
			strcmpi(playerInGame[i].PlayerName, (char*) aPacket+8) == 0)
		{
			playerInGame[i].InUse = false;
			return i;
		}
	}
	// shouldn't get here
	return (-1);
}

void OverheadMsg(char *Msg)
{
	BYTE aPacket[512];

	sprintf((char*)aPacket, "%c%c%c%s%c%c%c", 0x14, 0x00, 0x00, Msg, 0,0,0);
	server->GameSendPacketToServer(aPacket, strlen(Msg)+6);
}

void ChatMsg(char *Msg)
{
	BYTE aPacket[512];

	sprintf((char*)aPacket, "%c%c%c%s%c%c%c", 0x15, 0x01, 0x00, Msg, 0,0,0);
	server->GameSendPacketToServer(aPacket, strlen(Msg)+6);
}

//////////////////////////////////////////////////////////////////////
// stuff below here is from ClientCore.cpp
//////////////////////////////////////////////////////////////////////
// Dll entry/exit
//////////////////////////////////////////////////////////////////////
BOOL APIENTRY DllMain(HANDLE hModule, DWORD  ul_reason_for_call, LPVOID lpReserved)
{
	BOOL hResult = TRUE;
	char *t;
    switch (ul_reason_for_call)
	{
		case DLL_PROCESS_ATTACH:

			// Create server struct
			server = new FUNCTIONENTRYPOINTS;

			// Bind exported functions from server
			HMODULE hModule;
			hModule = (HMODULE)GetModuleHandle("D2HackIt");

			// Macros make this look a lot nicer :)
			BIND_TO_SERVER(GamePrintInfo);
			BIND_TO_SERVER(GamePrintError);
			BIND_TO_SERVER(GamePrintVerbose);
			BIND_TO_SERVER(GamePrintString);
			BIND_TO_SERVER(GameCommandLine);
			BIND_TO_SERVER(GameSendPacketToServer);
			BIND_TO_SERVER(GameSendPacketToGame);
			BIND_TO_SERVER(GetFingerprint);
			BIND_TO_SERVER(Intercept);
			BIND_TO_SERVER(GetHackProfileString);
			BIND_TO_SERVER(SetHackProfileString);
			BIND_TO_SERVER(GetThisgameStruct);

			// Get plugin path
			t = ConfigPath;
			if (GetModuleFileName((HINSTANCE)hModule, t, _MAX_PATH)) {
				int p=strlen(ConfigPath);
				while (p) {
						if (ConfigPath[p] == '\\')
							{ ConfigPath[p] = 0; p=0;}
						else
					p--;
				}
			}
			// initiate client
			// hResult = OnClientStart();
			break;

		case DLL_PROCESS_DETACH:
			// kill client

			// hResult = OnClientStop();

			delete server;
			break;
    }
    return hResult;
}

//////////////////////////////////////////////////////////////////////
// Stubfunctions for 'property get' functions.
//////////////////////////////////////////////////////////////////////
LPCSTR	EXPORT GetModuleAuthor()		{return ModuleAuthor;}
LPCSTR	EXPORT GetModuleWebsite()		{return ModuleWebsite;}
DWORD	EXPORT GetModuleVersion()		{return ModuleVersion;}
LPCSTR	EXPORT GetModuleEmail()			{return ModuleEmail;}
LPCSTR	EXPORT GetModuleDescription()	{return ModuleDescription;}

//////////////////////////////////////////////////////////////////////
// OnClientCommandLine
// -------------------------------------------------------------------
// The modules own extension of the command line interface. Any custom
// commands you add are parsed here.
//
// Return value should be TRUE, but it is not used at this
// time.
//
// Arguments when we get here:
// argv[0]			Name of module
// argv[1]			Name of command (If supplied)
// argv[2 ... n]	The rest
//
// Syntax in the game: .<module> <arguments>
//////////////////////////////////////////////////////////////////////
BOOL EXPORT OnGameCommandLine(char* argv[], int argc)
{
	// Check if user supplied anything at all, if not assume help...
	if (argc==1)
		argv[argc++]="help";


	MODULECOMMANDSTRUCT* mcs=ModuleCommands;

	while (mcs->szName) {
		if (!stricmp(mcs->szName, argv[1]))
			break;
		mcs++;
	}

	char *p,*t,*msg,*fMsg;
	fMsg=new char[256];
	//
	// Is this a built-in function ?
	if (mcs->szName) {
		//
		// If functions returns false, show usage help
		if (!mcs->pFunc(argv, argc)) {
			t=new char[strlen(mcs->szUsage)+1];
			server->GamePrintInfo("Usage:");
			sprintf((char*)t, "%s", mcs->szUsage);
			if (strlen((char*)t))
			{
				msg=p=t;
				while (*p != 0) {
					if (*p == '\n')
					{
						*(p++) = 0;
						sprintf(fMsg, "c4.%s %s", argv[0], msg);
						server->GamePrintInfo((char*)fMsg);
					if (*p != 0)
						msg = p;
					} else
						p++;
					}
				sprintf(fMsg, "c4.%s %s", argv[0], msg);
				server->GamePrintInfo((char*)fMsg);
			}
			delete t;
		}
	}
	else {
	// Unknown command, show catch-all help phraze.
	t=new char[128];
	sprintf(t, "Unknown command c4'%s %s'c0 - try c4'.%s help'c0 to get help.",
		argv[0], argv[1], argv[0]);
	server->GamePrintError(t);
	delete t;
	}
	delete fMsg;
	return TRUE;
}

//////////////////////////////////////////////////////////////////////
// OnGameCommandHelp
// -------------------------------------------------------------------
// Our default help function.
//
// Syntax in the game: .<module> <arguments>
//////////////////////////////////////////////////////////////////////
BOOL PRIVATE OnGameCommandHelp(char** argv, int argc)
{
	// If command line is longer than 2, show usage for 'help'
	if (argc>2) return FALSE;

	char t[1024];
	sprintf(t, "Available commands for %s:", argv[0]);
	server->GamePrintInfo(t);

	// Loop through ModuleCommands[] and print their names
	for (int i=0; ModuleCommands[i].szName != NULL; i++)
	{
		sprintf(t, "c4.%s %s", argv[0], ModuleCommands[i].szName);
		server->GamePrintInfo(t);
	}

	sprintf(t, "For help on a specific command, type c4.%s <command> help", argv[0]);
	server->GamePrintInfo(t);
	return TRUE;
}