#define _WIN32
#include <windows.h>
#include <windowsx.h>
#include <ddraw.h>

long FAR PASCAL WindowProc( HWND hWnd, UINT message,
                            WPARAM wParam, LPARAM lParam );

// *********************************************************************
// Variables globales
// *********************************************************************

HWND hwnd;                             // Handle de la ventana principal
HINSTANCE hInstance;
char ClassName[] = "ClaseDirectX";     // Nombre de la clase de ventana
char Titulo[] = "Prueba Direct-X";     // Titulo de la ventana

// - Variables globales para DirectDraw --------------------------------

// defines utilizados para definir la resolucion de la pantalla
// ejemplos validos son: 320x200, 640x480, 800x600, etc.

#define XSIZE 640
#define YSIZE 480

LPDIRECTDRAW         lpDD;           // Objeto DirectDraw
LPDIRECTDRAWSURFACE  lpDDSPrimary;   // Superficie primaria
LPDIRECTDRAWSURFACE  lpDDSBack;      // Back-Buffer
LPDIRECTDRAWSURFACE  lpDDSSprite;    // Aqui guardaremos los sprites
LPDIRECTDRAWSURFACE  lpDDSFondo;     // y aqui el fondo de "estrellas"
DDSURFACEDESC        ddsd;
DDSCAPS              ddscaps;
HRESULT              ddrval;

// Esta variable nos indica si Directdraw esta activo o no.
// En la funcion DirectDrawInit() la ponemos a TRUE si no ha ocurrido
// ningun error

BOOL                 bActive;

int XSize, YSize;  // Variables utilizadas para la resolucion de la pantalla

// - Variables Globales para escribir el FrameRate ---------------------

int FrameRate;           // FPS: frames por segundo
int FrameCount;
int FrameCount0;
DWORD FrameTime;
DWORD FrameTime0;

// - Variables globales para el movimiento del pacman ------------------

#define DIR_RIGHT 0
#define DIR_LEFT  1
#define DIR_DOWN  2
#define DIR_UP    3

int XPos = 0, YPos = 50;
int XInc = 1, YInc = 0;
char Speed = 1, Dir = DIR_RIGHT;

// *********************************************************************
// Rutinas para DirectDraw
// *********************************************************************

void DirectDrawEnd();
BOOLEAN CleanupAndExit( char *err );

/* --------------------------------------------------------------------------
   char DirectDrawInit( HWND hwnd )
   ------------------------
   Inicializa DirectDraw.
   Devuelve:
      TRUE: no problem
      FALSE: ha ocurrido algun error. Antes de salir, libera todos los
             recursos de DirectDraw que habia creado ya, y presenta un
             mensaje al usuario ( ver funcion CleanupAndExit() )
   -------------------------------------------------------------------------- */

BOOLEAN DirectDrawInit( HWND hwnd )
{
   ddrval = DirectDrawCreate( NULL, &lpDD, NULL );
   if( ddrval != DD_OK ) return CleanupAndExit("DirectDrawCreate Failed!");

   ddrval = lpDD->SetCooperativeLevel( hwnd,
                              DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN );
   if( ddrval != DD_OK ) return CleanupAndExit("SetCooperativeLevel Failed");

   XSize = XSIZE;
   YSize = YSIZE;
   ddrval = lpDD->SetDisplayMode( XSize, YSize, 8 );
   if( ddrval != DD_OK ) return CleanupAndExit("SetDisplayMode Failed!");

   memset( &ddsd, 0, sizeof(ddsd) );  // Clear all fields in structure
   ddsd.dwSize = sizeof( ddsd );      // Set flags in structure
   ddsd.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
   ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_FLIP |
                         DDSCAPS_COMPLEX;
   ddsd.dwBackBufferCount = 1;
   //Create Surface
   ddrval = lpDD->CreateSurface( &ddsd, &lpDDSPrimary, NULL );
   if( ddrval != DD_OK ) return CleanupAndExit("CreateSurface FrontBuffer Failed!");

   // create back buffer
   ddscaps.dwCaps = DDSCAPS_BACKBUFFER;
   ddrval = lpDDSPrimary->GetAttachedSurface( &ddscaps, &lpDDSBack );
   if( ddrval != DD_OK ) return CleanupAndExit("GetAttachedDurface Failed!");


   // create off-screen surface
   ddsd.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
   ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
   ddsd.dwWidth = 38;
   ddsd.dwHeight = 43;
   ddrval = lpDD->CreateSurface( &ddsd, &lpDDSSprite, NULL );
   if( ddrval != DD_OK ) return CleanupAndExit("CreateSurface lpDDSSprite Failed!");

   // create off-screen surface
   ddsd.dwHeight = YSize;
   ddsd.dwWidth = XSize;
   ddrval = lpDD->CreateSurface( &ddsd, &lpDDSFondo, NULL );
   if( ddrval != DD_OK ) return CleanupAndExit("CreateSurface lpDDSFondo Failed!");

   bActive = TRUE;
   return TRUE;
}

/* --------------------------------------------------------------------------
   void DirectDrawEnd(void)
   ------------------------
   Destruye todos los objetos DirectDraw
   ------------------------------------------------------------------------- */

void DirectDrawEnd(void)
{
   if(lpDD != NULL)
   {
      if( lpDDSFondo != NULL ) { lpDDSFondo->Release(); lpDDSFondo = NULL; }
      if( lpDDSSprite != NULL ) { lpDDSSprite->Release(); lpDDSSprite = NULL; }
      if(lpDDSPrimary != NULL) { lpDDSPrimary->Release(); lpDDSPrimary = NULL; }

      lpDD->Release();
      lpDD = NULL;
   }
   bActive = FALSE;
}

/* --------------------------------------------------------------------------
   BOOLEAN CleanUpAndExit()
   ------------------------
   Llamada por DirectDrawInit() cuando ha ocurrido algun error.
   Destruye todos los objetos directDraw, y muestra un mensaje
   de error al usuario.
   ------------------------------------------------------------------------- */

BOOLEAN CleanupAndExit( char *err )
{
   DirectDrawEnd();
   MessageBox( hwnd, err, "Direct-X Error", MB_OK );
   return FALSE;
}

// * Fin de las funciones de DirectDraw *************************************

/* --------------------------------------------------------------------------
   void SetupFondo()
   -----------------
   Dibuja el fondo en la superficie lpDDSFondo.
   En este caso lo pone de color negro, y dibuja unas cuantas "estrellas"
   que comprobemos que el pacman no borra el fondo cuando pasa.
   ------------------------------------------------------------------------- */

void SetupFondo()
{
   unsigned char *pointer;
   unsigned char *p;
   int i, j, l;
   HDC hdc;
   char s[100];

   while (lpDDSFondo->Lock(NULL, &ddsd, 0, NULL) == DDERR_WASSTILLDRAWING);
   pointer = (unsigned char *) ddsd.lpSurface;

   // Esto se puede hacer tambien con un Blt, utilizando DDBLT_COLORFILL
   p = pointer;
   for(i=0; i<XSize; i++)
      for(j=0; j<YSize; j++) *p++ = 0;

   // Ahora ponemos unas cuantas "estrellas" en el fondo, para comprobar
   // que no se borra el fondo cuando pasa el pacman

   // *(pointer + random(XSize) + random(YSize)*XSize ) = (unsigned char ) random(256);
   for(i = 0; i < 500; i++)
   {
      p = pointer;
      for(j=random(YSize); j; j--) p += XSize;
      p += random(XSize );
      *p = (unsigned char ) random(256);
   }
   lpDDSFondo->Unlock(ddsd.lpSurface);

   // Y un poco de texto en la parte de abajo a la izquierda
   lpDDSFondo->GetDC( &hdc );
   l = wsprintf(s, "Modo: %dx%d - +/- Velocidad - ESC salir" , XSize, YSize);
   SetBkMode( hdc, TRANSPARENT );
   SetTextColor(hdc, RGB(0,0,255));
   SetTextAlign( hdc, TA_LEFT | TA_BOTTOM );
   TextOut( hdc, 0, YSize, s, l);
   lpDDSFondo->ReleaseDC( hdc );
}

/* --------------------------------------------------------------------------
   char LoadPacmanBitmap()
   -----------------------
   Lee el recurso bitmap PACMAN_BMP, y lo dibuja en la superficie lpDDSSprite
   ------------------------------------------------------------------------- */

char LoadPacmanBitmap()
{
   HBITMAP hbm, bmOld;
   HDC hdcImage, hdc;
   BITMAP bm;

   hbm = LoadBitmap( hInstance, "PACMAN_BMP");
   if( hbm == NULL ) return 1;
   hdcImage = CreateCompatibleDC(NULL);
   bmOld = (HBITMAP) SelectObject(hdcImage, hbm);

   // get size of the bitmap
   GetObject(hbm, sizeof(bm), &bm);    // get size of bitmap

   // get size of surface.
   ddsd.dwSize = sizeof(ddsd);
   ddsd.dwFlags = DDSD_HEIGHT | DDSD_WIDTH;
   lpDDSSprite->GetSurfaceDesc(&ddsd);

   if( lpDDSSprite->GetDC(&hdc) == DD_OK )
   {
       BitBlt( hdc, 0, 0, bm.bmWidth, bm.bmHeight, hdcImage, 0, 0, SRCCOPY);
       lpDDSSprite->ReleaseDC(hdc);
   }
   SelectObject( hdcImage, bmOld );
   DeleteDC(hdcImage);

   DDCOLORKEY ddck;
   ddck.dwColorSpaceLowValue  = 0;
   ddck.dwColorSpaceHighValue = ddck.dwColorSpaceLowValue;
   lpDDSSprite->SetColorKey(DDCKEY_SRCBLT, &ddck);

   return 0;
}

/* --------------------------------------------------------------------------
   void BlitPacman( )
   ------------------
   Copia el pacman de lpDDSSprite a lpDDSBack.
   ------------------------------------------------------------------------- */

#define PACMAN_ANCHO 12    // ancho y alto del pacman (ver fichero .bmp )
#define PACMAN_ALTO  10

void BlitPacman( )
{
   static char frame = 0;
   static DWORD LastFrameTick = 0, t;
   RECT rc;

   t = GetTickCount();
   if ( t - LastFrameTick > 150 )
   {
      LastFrameTick = t;
      frame++;
      if( frame >= 3 ) frame = 0;
   }
   rc.left = frame * (PACMAN_ANCHO+1);
   rc.top = Dir*(PACMAN_ALTO+1);
   rc.right = rc.left + PACMAN_ANCHO;
   rc.bottom = rc.top + PACMAN_ALTO;

   ddrval = lpDDSBack->BltFast(XPos, YPos, lpDDSSprite, &rc, DDBLTFAST_SRCCOLORKEY);

   if( XPos+XInc >= XSize-PACMAN_ANCHO){ XInc = 0; XPos = XSize - PACMAN_ANCHO; }
      else if( XPos+XInc <= 0 )  { XInc = 0; XPos = 0; }
   if( YPos+YInc >= YSize-PACMAN_ALTO) { YInc = 0; YPos = YSize - PACMAN_ALTO; }
      else if( YPos+YInc <= 0 )  { YInc = 0; YPos = 0; }
   XPos += XInc;
   YPos += YInc;
}

/* --------------------------------------------------------------------------
   void CopyFondo2Back()
   ---------------------
   Copia el Fondo en el BackBuffer
   ------------------------------------------------------------------------- */

void CopyFondo2Back()
{
   RECT rc = {0,0,XSize,YSize};
   while( 1 )
   {
      ddrval= lpDDSBack->BltFast(0, 0, lpDDSFondo, &rc, FALSE);
      if( ddrval == DD_OK ) { break; }
      if( ddrval != DDERR_WASSTILLDRAWING )
      {
         return;
      }
   }
}

/* --------------------------------------------------------------------------
   void WriteInfo()
   ----------------
   Escribe en el backbuffer los fps y la velocidad del pacman
   ------------------------------------------------------------------------- */

void WriteInfo()
{
   HDC hdc;
   char s[50];
   int l;

   lpDDSBack->GetDC( &hdc );
   l = wsprintf(s, "FPS: %05d - Velocidad Pacman: %2d", FrameRate, Speed );
   SetBkMode( hdc, TRANSPARENT );
   SetTextColor(hdc, RGB(255,0,0));
   TextOut( hdc, 0, 0, s, l);
   lpDDSBack->ReleaseDC( hdc );
}

/* --------------------------------------------------------------------------
   void Flip()
   -----------
   Intercambia el backbuffer y el frontbuffer.
   ------------------------------------------------------------------------- */

void Flip()
{
   while(1)
   {
       HRESULT ddrval;
       ddrval = lpDDSPrimary->Flip(NULL, 0);
       if(ddrval == DD_OK) break;
       if(ddrval == DDERR_SURFACELOST)
       {
           ddrval = lpDDSPrimary->Restore();
           if(ddrval != DD_OK)
           {
               break;
           }
       }
       if(ddrval != DDERR_WASSTILLDRAWING)
       {
           break;
       }
   }
   FrameCount++;
   FrameTime = timeGetTime();
   if (FrameTime - FrameTime0 > 1000)
   {
      FrameRate = (FrameCount - FrameCount0) * 1000 / (FrameTime - FrameTime0);
      FrameTime0 = FrameTime;
      FrameCount0 = FrameCount;
   }
}

/* --------------------------------------------------------------------------
   void UpdateFrame()
   ------------------

   ------------------------------------------------------------------------- */

void UpdateFrame()
{
   if( bActive )
   {
      CopyFondo2Back();
      WriteInfo();
      BlitPacman( );

      Flip();
   }
}

// *********************************************************************

long FAR PASCAL WindowProc( HWND hwnd, UINT message,
                            WPARAM wParam, LPARAM lParam )
{
   switch (message)
   {
      case WM_ACTIVATEAPP:
         FrameTime0 = FrameTime = timeGetTime();
         FrameCount0 = FrameCount = 0;
         FrameRate = 0;
         break;

      case WM_CREATE:
         break;

      case WM_SETCURSOR:
         SetCursor(NULL);
         break;

      case WM_TIMER:
         break;

      case WM_KEYDOWN:
         switch( wParam )
         {
            case VK_ESCAPE: SendMessage(hwnd, WM_DESTROY,0,0); break;

            case VK_LEFT: Dir = DIR_LEFT; XInc = -Speed; YInc = 0; break;
            case VK_RIGHT: Dir = DIR_RIGHT; XInc = Speed; YInc = 0; break;
            case VK_UP: Dir = DIR_UP; XInc = 0; YInc = -Speed; break;
            case VK_DOWN: Dir = DIR_DOWN; XInc = 0; YInc = Speed; break;

            case VK_ADD: if( Speed < 10 ) Speed++;
                         if( XInc > 0 ) XInc = Speed;
                            else if(XInc < 0) XInc = -Speed;
                         if( YInc > 0 ) YInc = Speed;
                            else if(YInc < 0) YInc = -Speed;
                         break;

            case VK_SUBTRACT: if (Speed > 1 ) Speed --;
                         if( XInc > 0 ) XInc = Speed;
                            else if(XInc < 0) XInc = -Speed;
                         if( YInc > 0 ) YInc = Speed;
                            else if(YInc < 0) YInc = -Speed;
                         break;
         }
			break;

   	case WM_DESTROY:
	      DirectDrawEnd(); //Destruye objetos DDraw
         PostQuitMessage( 0 );
         break;

   	default: return DefWindowProc(hwnd, message, wParam, lParam);
	}
   return 0;
}

// --------------------------------------------------------------------------

#pragma 	argsused
int PASCAL WinMain( HINSTANCE xhInstance, HINSTANCE hPrevInstance,
                    LPSTR lpszCmdLine, int nCmdShow )
{
   MSG msg;
   WNDCLASS wc;

   if (hPrevInstance) return -1; // Retorna si hay instancia previa

   hInstance = xhInstance;  // set global var

   // Definir la clase de ventana
   wc.style = CS_HREDRAW | CS_VREDRAW ;
   wc.lpfnWndProc = WindowProc;           //Funcion que maneja los mensajes
   wc.cbClsExtra = 0;
   wc.cbWndExtra = 0;
   wc.hInstance = hInstance;
   wc.hIcon = LoadIcon( hInstance, IDI_APPLICATION );
   wc.hCursor = LoadCursor( NULL, IDC_ARROW );
   wc.hbrBackground = NULL;
   wc.lpszMenuName = NULL;
   wc.lpszClassName = ClassName;

   RegisterClass( &wc );
   hwnd = CreateWindowEx(
		  0,                               // Estilo Extra ventana
		  ClassName,                       // Nombre de la clase
		  Titulo,                          // Titulo ventana
		  WS_POPUP,                        // Estilo de la ventana
		  0,                               // Posicion X
		  0,                               // Posicion Y
		  GetSystemMetrics( SM_CXSCREEN ), // Ancho = Ancho de pantalla
		  GetSystemMetrics( SM_CYSCREEN ), // Alto = Alto de pantalla
		  NULL,                            // Handle ventana padre
		  NULL,                            // Handle del menu
		  hInstance,                       // Handle de la aplicacion
		  NULL );				              // lPParam

   if( !hwnd ) return FALSE;
   ShowWindow( hwnd, nCmdShow );
   UpdateWindow( hwnd );
   SetCursor(NULL);

   // Hasta aqui todo lo habitual de un programa de Windows. Lo unico un
   // poco distinto es que hemos puesto como ancho y alto de la ventana
   // toda la pantalla, porque vamos a hacer un juego, y cuando se esta jugando
   // nadie quiere ver otras ventanas :)

   // Inicializar DirectDraw
   bActive = FALSE;
   if( ! DirectDrawInit( hwnd ) )
   {
      // Si ha habido algun error, nos vamos.
      DestroyWindow( hwnd );
      return FALSE;
   }

   // DirectDraw esta inicializado. Ahora cargamos los sprites, y
   // preparamos el fondo
   if( bActive )
   {
      randomize();
      if( LoadPacmanBitmap() ) { DestroyWindow( hwnd ); return FALSE; }
      SetupFondo();
   }

   // Bucle de mensajes
   while( 1 )
   {
      if( PeekMessage( &msg, NULL, 0, 0, PM_NOREMOVE ) )
      {
         if( !GetMessage( &msg, NULL, 0, 0 ) )
            return msg.wParam;
         TranslateMessage(&msg);
         DispatchMessage(&msg);
      }
      else if( bActive )
      {
         UpdateFrame();
      }
      else
      {
         // make sure we go to sleep if we have nothing else to do
         WaitMessage();
      }
   }
}

