
* equates

ROM      EQU   $F800    start of program
RAM      EQU   $2000    start of variables
STACK    EQU   $3FFF    bottom of stack

SCI_IRQV EQU   $FFD6    serial communications interrupt vector
TIC1_IRQ EQU   $FFEE    timer input capture 1 interrupt vector
RESETV   EQU   $FFFE    reset interrupt vector

* these are ports on the 68HC11.  they have been remapped to low RAM;
* otherwise the high byte of their address would be $10xx.

PORTA   EQU   $0000
PIOC    EQU   $0002
PORTC   EQU   $0003
PORTB   EQU   $0004
PORTCL  EQU   $0005
DDRC    EQU   $0007
PORTD   EQU   $0008
DDRD    EQU   $0009
PORTE   EQU   $000A
CFORC   EQU   $000B
OC1M    EQU   $000C
OC1D    EQU   $000D
TCNT    EQU   $000E
TIC1    EQU   $0010
TIC2    EQU   $0012
TIC3    EQU   $0014
TOC1    EQU   $0016
TOC2    EQU   $0018
TOC3    EQU   $001A
TOC4    EQU   $001C
TOC5    EQU   $001E
TCTL1   EQU   $0020
TCTL2   EQU   $0021
TMSK1   EQU   $0022
TFLG1   EQU   $0023
TMSK2   EQU   $0024
TFLG2   EQU   $0025
PACTL   EQU   $0026
PACNT   EQU   $0027
SPCR    EQU   $0028
SPSR    EQU   $0029
SPDR    EQU   $002A
BAUD    EQU   $002B
SCCR1   EQU   $002C
SCCR2   EQU   $002D
SCSR    EQU   $002E
SCDR    EQU   $002F
ADCTL   EQU   $0030
ADR1    EQU   $0031
ADR2    EQU   $0032
ADR3    EQU   $0033
ADR4    EQU   $0034
OPTION  EQU   $0039
COPRST  EQU   $003A
PPROG   EQU   $003B
HPRIO   EQU   $003C
INIT    EQU   $003D
TEST1   EQU   $003E
CONFIG  EQU   $003F

* constants

BLINKCOUNT EQU $1FFF countdown for led toggle
DELAY1     EQU 100   binary search calculation delay (in clocks)
DELAY2     EQU 145   binary search calculation delay (in clocks)
REV_LIMIT  EQU 1250  rpm limit (low value is high rpm, $1250 = 12000 rpm)

* variables

       ORG   RAM

blinkcnt       RMB 2    led blinker counter

rev_limit      RMB 2    rev limit count value
rev_limit_flag RMB 1    set if rev limiter activated

sci_complete   RMB 1    serial message completely received flag
sci_continuous RMB 1    send serial data continuously flag
sci_type       RMB 1    serial message type
sci_chksum     RMB 1    serial message checksum
sci_index      RMB 1    index into serial receive message buffer
sci_buffer     RMB 256  serial message buffer

sci_c_start    RMB 2    serial data send start address
sci_c_index    RMB 1    index into serial send message buffer
sci_c_size     RMB 1    serial send data size in bytes

sync_count     RMB 1    sync pulse count
pulse          RMB 1    inductive pickup pulse counter
new_tic1       RMB 2    time captured by timer input capture port 1
old_tic1       RMB 2    last time captured by timer input capture port 1
new_time       RMB 2    time between pulses
old_time       RMB 2    time between last two pulses
rpmcount       RMB 2    RPM count for PC to read

sparkadv       RMB 2    spark advance calculated from binary search
bin_offset     RMB 2    binary search offset
bin_count      RMB 1    binary search iteration count
dummy          RMB 1    not used - realign variables to even address

tempword       RMB 2    temporary 2-byte variable
tempbyte       RMB 1    two temporary 1-byte variables
tempbyte2      RMB 1

rpm            RMB 512  TIC1 count -> RPM lookup table
advance        RMB 512  RPM -> spark advance lookup table

nc_dest        RMB 2    new code destination
nc_size        RMB 2    new code size
nc_buffer      RMB 2048 new code buffer

prog_code      RMB 50   storage space for EEPROM programming subprogram

* code

       ORG    ROM

teststring FCC \Ignition Controller is running! \

rpmstatic  FDB $EA60,$C5F3,$AB53,$9703,$8702,$7A12,$6F65,$666F
           FDB $5ECF,$583E,$5286,$4D81,$490F,$4518,$418A,$3E55
           FDB $3B6C,$38C6,$365A,$3421,$3214,$302F,$2E6D,$2CCB
           FDB $2B46,$29DA,$2885,$2745,$2618,$24FD,$23F2,$22F5
           FDB $2206,$2124,$204D,$1F80,$1EBD,$1E04,$1D53,$1CAA
           FDB $1C09,$1B6E,$1ADA,$1A4C,$19C4,$1941,$18C3,$184A
           FDB $17D6,$1766,$16FA,$1692,$162D,$15CC,$156E,$1513
           FDB $14BB,$1466,$1414,$13C4,$1377,$132C,$12E3,$129C
           FDB $1258,$1215,$11D4,$1195,$1158,$111C,$10E2,$10AA
           FDB $1073,$103D,$1009,$FD6, $FA4, $F73, $F44, $F15
           FDB $EE8, $EBC, $E91, $E67, $E3E, $E15, $DEE, $DC7
           FDB $DA1, $D7D, $D58, $D35, $D12, $CF0, $CCF, $CAE
           FDB $C8E, $C6F, $C50, $C32, $C14, $BF7, $BDB, $BBF
           FDB $BA3, $B88, $B6E, $B54, $B3A, $B21, $B08, $AF0
           FDB $AD8, $AC1, $AAA, $A93, $A7D, $A67, $A51, $A3C
           FDB $A27, $A13, $9FE, $9EA, $9D7, $9C4, $9B0, $99E
           FDB $98B, $979, $967, $956, $944, $933, $922, $911
           FDB $901, $8F1, $8E1, $8D1, $8C2, $8B2, $8A3, $894
           FDB $886, $877, $869, $85B, $84D, $83F, $831, $824
           FDB $817, $80A, $7FD, $7F0, $7E3, $7D7, $7CB, $7BF
           FDB $7B3, $7A7, $79B, $78F, $784, $779, $76E, $763
           FDB $758, $74D, $742, $738, $72D, $723, $719, $70F
           FDB $705, $6FB, $6F1, $6E8, $6DE, $6D5, $6CB, $6C2
           FDB $6B9, $6B0, $6A7, $69E, $695, $68D, $684, $67C
           FDB $673, $66B, $663, $65A, $652, $64A, $642, $63B
           FDB $633, $62B, $623, $61C, $614, $60D, $606, $5FE
           FDB $5F7, $5F0, $5E9, $5E2, $5DB, $5D4, $5CE, $5C7
           FDB $5C0, $5B9, $5B3, $5AC, $5A6, $5A0, $599, $593
           FDB $58D, $587, $580, $57A, $574, $56E, $569, $563
           FDB $55D, $557, $551, $54C, $546, $541, $53B, $536
           FDB $530, $52B, $525, $520, $51B, $516, $510, $50B
           FDB $506, $501, $4FC, $4F7, $4F2, $4ED, $4E8, $4E4

advstatic  FDB $DAC0,$B71C,$9D0F,$8930,$7987,$6CDF,$626D,$59A9
           FDB $5233,$4BC6,$462E,$4145,$3CEB,$390A,$3590,$326D
           FDB $2F94,$2CFD,$2A9E,$2871,$2670,$2495,$22DD,$2144
           FDB $1FC7,$1E63,$1D15,$1BDC,$1AB6,$19A1,$189C,$17A4
           FDB $16BB,$15DD,$150B,$1442,$1384,$12CF,$1221,$117C
           FDB $10DE,$1047,$FB6, $F2B, $EA6, $E26, $DAB, $D35
           FDB $CC3, $C55, $BEC, $B86, $B23, $AC4, $A68, $A0F
           FDB $9B9, $966, $916, $8C8, $8A6, $878, $84B, $820
           FDB $7F6, $7CD, $7A5, $77F, $759, $735, $711, $6EF
           FDB $6CD, $6AC, $68C, $66D, $64E, $630, $613, $5F6
           FDB $5DB, $5C0, $5A5, $58C, $572, $559, $541, $52A
           FDB $512, $4FC, $4E6, $4D0, $4BB, $4A6, $491, $47D
           FDB $46A, $457, $444, $431, $41F, $40D, $3FC, $3EB
           FDB $3E1, $3D8, $3CF, $3C6, $3BE, $3B5, $3AD, $3A5
           FDB $39D, $395, $38E, $386, $37F, $377, $370, $369
           FDB $362, $35B, $354, $34E, $347, $341, $33A, $334
           FDB $32E, $328, $322, $31C, $316, $311, $30B, $305
           FDB $300, $2FB, $2F5, $2F0, $2EB, $2E6, $2E1, $2DC
           FDB $2D7, $2D2, $2CD, $2C9, $2BE, $2B4, $2AA, $2A0
           FDB $296, $28D, $283, $27A, $270, $267, $25E, $256
           FDB $24D, $244, $23B, $233, $22A, $222, $21A, $212
           FDB $20A, $202, $1FA, $1F2, $1EA, $1E3, $1DB, $1D4
           FDB $1CD, $1C5, $1BE, $1B7, $1B0, $1A9, $1A2, $19B
           FDB $195, $18E, $188, $181, $17A, $174, $172, $170
           FDB $16E, $16D, $16B, $169, $167, $165, $164, $162
           FDB $160, $15E, $15D, $15B, $159, $158, $156, $154
           FDB $153, $151, $150, $14E, $14D, $14B, $14A, $148
           FDB $147, $145, $144, $142, $141, $140, $13E, $13D
           FDB $13B, $13A, $138, $137, $136, $134, $133, $132
           FDB $131, $12F, $12E, $12D, $12C, $12A, $129, $128
           FDB $127, $126, $124, $123, $122, $121, $120, $11E
           FDB $11D, $11C, $11B, $11A, $119, $118, $117, $116

main:

   lds    #STACK       load stack pointer
   ldaa   #$10         remap ram=10xx and regs=00xx
   staa   $103D          by writing to CONFIG register
   jsr    initial      do other initialization

loop:

   ldaa  sci_complete
   tsta                entire serial message received?
   bne   decod_mes       yes, decode the message
   jmp   chk_continuous  no, jump to continuous mode check

decod_mes:

   ldaa  sci_type      get message type
   cmpa  #0            type = 0? (send bytes)
   bne   type1           no, check for type 1

* send size, tag, N data bytes, and checksum back to PC via RS 232.

   ldx   sci_buffer+1 get address of bytes to send
   ldab  sci_buffer+3 get number of bytes to send

type0size:
*                     loop until serial xmitter ready to send
   brclr SCSR #%1000000 type0size
   stab SCDR          send the size
   stab tempbyte      checksum = size

   ldaa sci_buffer    get the data tag

type0tag:
*                     loop until serial xmitter ready to send
   brclr SCSR #%10000000 type0tag
   staa SCDR          send tag
   adda tempbyte      checksum = checksum + tag
   staa tempbyte      store checksum

type0loop:

   tstb               done sending all the bytes?
   beq   end_type0      yes, skip sending code
   ldaa  ,X           get byte to send

type0wait:
*                     loop until serial xmitter ready to send
   brclr SCSR #%10000000 type0wait
   staa  SCDR         send the byte
   adda  tempbyte     checksum = checksum + data byte
   staa  tempbyte     store checksum
   inx                point to next byte to send
   decb               decrement number of bytes to send
   bra   type0loop    loop back

end_type0:
*                     loop until serial xmitter ready to send
   brclr SCSR #%10000000 end_type0
   ldaa  tempbyte     get the checksum
   staa  SCDR         send the checksum

   clra
   staa  sci_complete clear flag, ready for another message

   ldaa  PORTD        read Port D
   anda  #%11110111   clear CTS (ready for another serial message)
   staa  PORTD        write the byte

   bra   chk_continuous jump down

type1:

   cmpa  #1           type = 1? (write bytes)
   bne   type2          no, check for type 2

   ldx   sci_buffer   get address of bytes to write over
   ldy   #sci_buffer+3 get address of source of data
   ldab  sci_buffer+2 get number of bytes to write

type1loop:

   tstb               all bytes written?
   beq   end_type1      yes, skip write
   ldaa  ,Y           get data byte to write
   staa  ,X           write data byte to destination
   inx                increment destination pointer
   iny                increment source pointer
   decb               decrement byte counter
   bra   type1loop    loop back

end_type1:

   clra
   staa  sci_complete clear flag, ready for another message

   ldaa  PORTD        read Port D
   anda  #%11110111   clear CTS (ready for another serial message)
   staa  PORTD        write the byte

   bra   chk_continuous jump down

type2:

   cmpa  #2           type = 2? (read bytes continuously)
   bne   type3          no, check for type 3

   ldx   sci_buffer   get address of bytes to read
   stx   sci_c_start  save address in continuous mode start address
   ldaa  sci_buffer+2 get number of bytes
   staa  sci_c_size   save in continuous mode size variable
   ldaa  #1
   staa  sci_continuous set continuous mode flag
   deca
   staa  sci_c_index  set continuous mode index to start of buffer
   staa  sci_complete clear flag, ready for another message

   ldaa  PORTD        read Port D
   anda  #%11110111   clear CTS (ready for another serial message)
   staa  PORTD        write the byte

   bra   chk_continuous jump down

type3:

   cmpa  #3           type = 3? (jump to new code)
   bne   type4          no, check for type 4

   ldx   sci_buffer   get address to jump to
   clr   sci_complete clear flag, ready for another message

   ldaa  PORTD        read Port D
   anda  #%11110111   clear CTS (ready for another serial message)
   staa  PORTD        write the byte

   jmp   ,X           jump to new code

type4:

   clra               error, no more types
   staa sci_complete  clear flag, ready for another message

   ldaa  PORTD        read Port D
   anda  #%11110111   clear CTS (ready for another serial message)
   staa  PORTD        write the byte

chk_continuous:

   ldaa  sci_continuous
   tsta                                serial continuous mode set?
   beq   cont_done                       no, skip next block of code

   brclr SCSR #%10000000 cont_done     if serial xmitter busy, skip send
   ldx   sci_c_start                   get beginning of send buffer
   ldab  sci_c_index                   get send buffer index
   abx                                 add buffer start address to index
   ldaa  ,X                            get byte from send buffer + index
   staa  SCDR                          transmit byte to PC
   incb                                increment send buffer index
   cmpb  sci_c_size                    past end of buffer?
   bne   no_reset                        no, dont reset index
   clrb                                  yes, reset index to zero

no_reset:

   stab  sci_c_index

cont_done:

   ldd   blinkcnt                      get blink count
   xgdx
   dex                                 decrement blink count
   xgdx
   std   blinkcnt                      store blink count
   beq   blinkled                      if zero, skip down
   jmp   loop                            else, jump to main loop

blinkled:

   ldaa  PORTA                         read port A
   eora  #%01000000                    toggle LED bit
   staa  PORTA                         write to port A
   ldd   #BLINKCOUNT                   load new blink count
   std   blinkcnt
   jmp   loop                          loop back

*---------------------------------------------------------------------------

initial:

   ldd   #BLINKCOUNT
   std   blinkcnt                      led blink count down value

   ldaa  #1
   staa  tempbyte2                     going to perform 2 copies
   clr   tempbyte                      each copy is 256 (=0) words
   ldx   #rpmstatic                    source
   ldy   #rpm                          destination

copy:

   ldd   ,X                            get word from source
   std   ,Y                            store in destination
   inx
   inx
   iny
   iny                                 increment indexes to next word
   inc   tempbyte                      increment word count
   bne   copy                          has count rolled over?
*                                        no, keep copying
   dec   tempbyte2                       yes, decrement number of copies
   beq   copy                          if only 1 copy done, copy 256 more

   ldaa  #%00100000
   staa  TCTL2                         capture on falling edges of TIC 1

   ldaa  #%00000100
   staa  TMSK1                         enable interrupt on falling edge of TIC 1
   staa  TFLG1                         clear TIC 1 flag by writing 1 to it

   ldaa  #2
   staa  sync_count                    initialize sync count to 2
   deca                                initialize rev limit flag to 1 so no
   staa  rev_limit_flag                  firing occurs until safe rpm

   ldaa  #%00001110
   staa  DDRD                          configure Port D data direction register
*                                        bit 5  input handshake bit 1  (PC=RQS)
*                                        bit 4  input handshake bit 0  (PC=DTR)
*                                        bit 3  output handshake bit 1 (PC=CTS)
*                                        bit 2  output handshake bit 0 (PC=DSR)
*                                        bit 1  serial data out        (to PC RxD)
*                                        bit 0  serial data in         (from PC TxD)

   ldaa  #%00110000
   staa  BAUD                          serial interface runs at 9600 baud
   ldaa  #%00101100
   staa  SCCR2                         enable serial output & input
   clra
   staa  sci_complete                  clear serial message complete flag
   staa  sci_continuous                clear continuous send mode flag
   staa  sci_chksum                    clear serial checksum
   staa  sci_index                     reset receive buffer pointer to zero

   staa  rev_limit_flag                clear rev limiter flag
   staa  pulse                         clear inductive pulse count

   ldd   #REV_LIMIT
   std   rev_limit                     set rev limit

   ldaa  PORTD                         read Port D
   anda  #%11110011                    clear CTS and DSR
   staa  PORTD                         write the byte

   cli                                 enable interrupts

   rts   return

*---------------------------------------------------------------------------

*                                      entry point for serial IRQ
sci_irq:

   ldaa   SCSR                         get serial status
   anda   #%00001110                   clear all but error bits
   tsta                                any error bits set?
   beq    sci_receive                    no, go to rest of IRQ

   ldaa   SCDR                           yes, read SCDR to clear
   rti                                 return

sci_receive:

   ldaa  PORTD                         read Port D
   oraa  #%00001000                    set CTS (serial I/O busy)
   staa  PORTD                         write the byte

   ldab   SCDR                         get byte from serial port
   ldaa   PORTD                        get handshake bytes
   anda   #%00110000                   clear all but input handshake bytes
   cmpa   #%00000000                   handshake = 00? (type byte)
   bne    sci_chk_data                   no, check if data byte
   stab   sci_type                       yes, save type byte
   stab   sci_chksum                   store checksum
   clra
   staa   sci_continuous               clear continuous flag
   rti                                 return

sci_chk_data:

   cmpa   #%00100000                   handshake = 10? (data byte)
   bne    sci_chk_chksum                 no, check if checksum byte
   ldx    #sci_buffer                    yes, get offset to input buffer
   tba                                 move serial byte to register A
   ldab   sci_index                    get index into input buffer
   abx                                 add index to buffer
   staa   ,X                           put data byte in input buffer
   incb                                increment index into buffer
   stab   sci_index                      and store back in memory
   adda   sci_chksum                   add data byte to checksum
   staa   sci_chksum                     and store checksum
   rti                                 return
                                       
sci_chk_chksum:
                                       
   cmpb   sci_chksum                   does input byte match sci_chksum?
   bne    sci_bad_message                no, dont set message complete flag
   ldaa   #1                             yes,
   staa   sci_complete                     set incoming message complete flag

sci_bad_message:

   clra
   staa   sci_chksum                   clear checksum byte
   staa   sci_index                    reset index into input buffer to zero
   staa   sci_continuous               clear continuous mode flag

   ldaa  PORTD                         read Port D
   anda  #%11110111                    clear CTS (serial I/O ready)
   staa  PORTD                         write the byte

   rti                                 return

*---------------------------------------------------------------------------

tic1_irq:

   ldaa   #%00000100                   clear TIC 1 interrupt flag so next
   staa   TFLG1                          interrupt will be caught

   ldx    TIC1                         get time from free running clock
   stx    new_tic1                     store it

   ldaa   sync_count                   get the sync count
   cmpa   #2                           1st pulse counted?

   bne    chk_if_2nd                     no, not 1st pulse, jump
   dec    sync_count                     yes, decrement sync count
   jmp    endTICirq                           and exit

chk_if_2nd:

   cmpa   #1                           2nd pulse?
   bne    chk_if_3rd                     no, jump

   xgdx                                put TIC 1 count in D register
   subd   old_tic1                     get time difference
   std    new_time                     store it
   dec    sync_count                   decrement sync counter
   jmp    endTICirq                    jump to exit

chk_if_3rd:

   xgdx                                put TIC 1 count in D register
   subd   old_tic1                     get time difference
   std    new_time                     store it

   lsrd                                D = time / 2
   pshb
   psha                                push D onto stack
   pulx                                X = D = time / 2
   lsrd                                D = time / 4
   stx    tempword
   addd   tempword                     done! D = 3/4 * new_time

   cpd    old_time                     3/4*new_time > old_time ?
   bhi    synch_pulse                    yes, synch pulse just occured...jump

   tst    pulse                          no, not sync pulse. already synched?
   bne    synched                          yes, jump
   jmp    endTICirq                        no, exit

synch_pulse:
*                                      *** 1st pulse ***
   ldaa   #1
   staa   pulse                        set pulse count = 1

   tst    rev_limit_flag               rev limiter on?
   bne    nofire                         yes, skip firing sequence

   ldd    sparkadv                       no, get spark advance count
   subd   #DELAY1                      subtract delay (contant)
   addd   TCNT                         add to current time
   std    TOC5                         put into Output Capture 5
   bclr   TCTL1 #%00000001
   bset   TCTL1 #%00000010             enable TOC 5 to go low

nofire:

   jmp    endTICirq                    exit

synched:

   ldaa   #1                           A = 1
   cmpa   pulse                        is this pulse 2?
   bne    chk_pulse3                     no, check if 3rd pulse

*                                      *** 2nd pulse ***

   inc    pulse                        increment pulse counter

   jmp    endTICirq                    exit

chk_pulse3:

   inca                                A = 2
   cmpa   pulse                        is this pulse 3?
   bne    chk_pulse4                     no, check if 4th pulse

*                                      *** 3rd pulse ***

   inc    pulse                        increment pulse counter

   bclr   TCTL1 #%00000011             disable OC 5
   bset   PORTA #%00001000             set OC 5 high to start dwelling

   ldd    new_time                     get time
   std    rpmcount                     put in memory for PC to read
   cpd    rev_limit                    over rev limit?
   bhi    below_rev_limit                no, jump down

   ldab   #1
   stab   rev_limit_flag                 yes, set flag
   jmp    endTICirq                           exit

below_rev_limit:

   clr    rev_limit_flag               clear rev limiter flag

   xgdx                                move rpm count to X
   ldab   #8
   stab   bin_count                    binary search has 8 iterations
   ldd    #256
   std    bin_offset                   index = 128
   xgdx                                move rpm count back to D

   ldx    #rpm                         get offset into rpm lookup table

bin_loop_top:

   pshx                                save X
   xgdx
   addd   bin_offset                   get offset
   xgdx
   cpd    ,X                           new rpm < rpm in rpm+offset
   bhi    belowRPM                       yes, jump down

   puly                                  no, throw away X value from stack
   lsr    bin_offset+0
   ror    bin_offset+1                       offset = offset / 2
   bra    bin_loop_bottom                    jump down

belowRPM:

   lsr    bin_offset+0
   ror    bin_offset+1                 offset = offset / 2
   pulx                                restore X
   nop
   nop                                 add delay to equalize branches

bin_loop_bottom:

   dec    bin_count                    counter = counter - 1
   bne    bin_loop_top                 loop until 8 iterations are done

   xgdx                                put offset into rpm lt in D
   subd   #rpm                         get just the offset
   addd   #advance                     offset into advance lookup table
   xgdx                                put offset in X
   ldd    ,X                           get spark advance count
   std    sparkadv                     save spark advance count

   jmp    endTICirq                    exit

chk_pulse4:

   inca                                A = 3
   cmpa   pulse                        is this pulse 4?
   bne    chk_pulse5                     no, check if 5th pulse

*                                      *** 4th pulse ***

   inc    pulse                        increment pulse counter

   jmp    endTICirq                    exit

chk_pulse5:

   inca                                A = 4
   cmpa   pulse                        is this pulse 5?
   bne    chk_pulse6                     no, check if 6th pulse

*                                      *** 5th pulse ***

   inc    pulse                        increment pulse counter

   ldd    sparkadv                       no, get spark advance count
   subd   #DELAY2                      subtract delay (contant)
   addd   TCNT                         add to current time
   std    TOC4                         put into Output Capture 4
   bclr   TCTL1 #%00000100
   bset   TCTL1 #%00001000             enable TOC 4

   jmp    endTICirq                    exit

chk_pulse6:

   inca                                A = 5
   cmpa   pulse                        is this pulse 6?
   bne    chk_pulse7                     no, check if 7th pulse

*                                      *** 6th pulse ***

   inc    pulse                        increment pulse counter

   jmp    endTICirq                    exit

chk_pulse7:
* since there are no more pulses...
*                                      *** 7th pulse ***
   bclr   TCTL1 #%00001100             disable OC 4
   bset   PORTA #%00010000             set OC 4 high

   ldd    new_time                     get time
   std    rpmcount                     put in memory for PC to read
   cpd    rev_limit                    over rev limit?
   bhi    below_rev_limit2               no, jump down

   ldab   #1
   stab   rev_limit_flag                 yes, set flag
   jmp    endTICirq                           exit

below_rev_limit2:

   clr    rev_limit_flag               clear rev limiter flag

   xgdx                                move rpm count to X
   ldab   #8
   stab   bin_count                    binary search has 8 iterations
   ldd    #256
   std    bin_offset                   index = 128
   xgdx                                move rpm count back to X

   ldx    #rpm                         get offset into rpm lookup table

bin_loop_top2:

   pshx                                save X
   xgdx
   addd   bin_offset                   get offset
   xgdx
   cpd    ,X                           new rpm < rpm in rpm+offset
   bhi    belowRPM2                      yes, jump down

   puly                                  no, throw away X value from stack
   lsr    bin_offset+0
   ror    bin_offset+1                       offset = offset / 2
   bra    bin_loop_bottom2                   jump down


belowRPM2:

   lsr    bin_offset+0
   ror    bin_offset+1                 offset = offset / 2
   pulx                                restore X
   nop
   nop                                 add delay to equalize branches

bin_loop_bottom2:

   dec    bin_count                    counter = counter - 1
   bne    bin_loop_top2                loop until 8 iterations are done

   xgdx                                  yes, put offset into rpm lt in D
   subd   #rpm                                get just the offset
   addd   #advance                     offset into advance lookup table
   xgdx                                put offset in X
   ldd    ,X                           get spark advance count
   std    sparkadv                     save spark advance count

* dont need to reset pulse counter since the IRQ checks for synching
* each time it is called.  when the synch pulse occurs the pulse counter
* will be reset to zero.

endTICirq:

   ldd    new_tic1                     get TIC 1 count from this interrupt
   std    old_tic1                     store as old TIC 1 count for next IRQ
   ldd    new_time                     get time calculated from this IRQ
   std    old_time                     store as old time for next IRQ
   rti                                 return

*---------------------------------------------------------------------------

   ORG   SCI_IRQV                      vector to serial interrupt
   FDB   sci_irq                       replace with new vector

   ORG   TIC1_IRQ                      vector to timer input capture interrupt
   FDB   tic1_irq                      replace with new vector

   ORG   RESETV                        vector to reset interrupt
   FDB   main                          replace with new vector
