// Djgpp graphics engine V3.0
//
// A complete multi-color, multi-resolution, multi-functional
// graphics engine for DJGPP and Windows. DirectDraw is required for
// Windows.
//
// CopyRight (c) 2001, Johan Bos
// www.sludgesoft.com/johan
// bosjoh@fcmail.com
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
// Check if the compiler is either DJGPP or a Win32 compiler
//////////////////////////////////////////////////////////////////////

#ifndef DJGPP
   #ifndef WIN32
      #error "The graphics engine is only for DJGPP or Win32 compilers"
   #endif
#endif

//////////////////////////////////////////////////////////////////////
// Define the graphics engine
//////////////////////////////////////////////////////////////////////

#ifndef GFXINCLUDED
   #define GFXINCLUDED
   #define G_V3_0
#endif

//////////////////////////////////////////////////////////////////////
// Includes
//////////////////////////////////////////////////////////////////////

#ifndef INPUTINCLUDED
   #ifdef DJGPP
      #include <malloc.h>
      #include <dos.h>
      #include <go32.h>
      #include <dpmi.h>
   #endif
   #ifdef WIN32
      #include <windows.h>
   #endif
#endif
#ifdef DJGPP
   #include <sys/nearptr.h>
   #include <crt0.h>
#endif
#ifdef WIN32
   #include <ddraw.h>
#endif
#include <fstream.h>

//////////////////////////////////////////////////////////////////////
// Global variable for DJGPP
//////////////////////////////////////////////////////////////////////

#ifdef DJGPP
   int _crt0_startup_flags=_CRT0_FLAG_NEARPTR|_CRT0_FLAG_NONMOVE_SBRK;
#endif

//////////////////////////////////////////////////////////////////////
// 2 VESA Structures for DJGPP
//////////////////////////////////////////////////////////////////////

#ifdef DJGPP
typedef struct{
   long             vesasignature __attribute__((packed));
   unsigned short   version __attribute__((packed));
   char             *oemstringptr __attribute__((packed));
   long             capabilities __attribute__((packed));
   long             videomodeptr __attribute__((packed));
   unsigned short   totalmemory __attribute__((packed));
   char             reserved [235] __attribute__((packed));
}VBEINFOREC;//Information about the videocard

typedef struct{
   unsigned short   modeattributes __attribute__((packed));
   unsigned char    winaattributes __attribute__((packed));
   unsigned char    winbattributes __attribute__((packed));
   unsigned short   wingranularity __attribute__((packed));
   unsigned short   winsize __attribute__((packed));
   unsigned short   winasegment __attribute__((packed));
   unsigned short   winbsegment __attribute__((packed));
   void             *winfuncptr __attribute__((packed));
   unsigned short   bytesperscanline __attribute__((packed));
   unsigned short   xresolution __attribute__((packed));
   unsigned short   yresolution __attribute__((packed));
   unsigned char    xcharsize __attribute__((packed));
   unsigned char    ycharsize __attribute__((packed));
   unsigned char    numberofplanes __attribute__((packed));
   unsigned char    bitsperpixel __attribute__((packed));
   char             dummy [14] __attribute__((packed));
   unsigned int     physbaseptr __attribute__((packed));
   char             reserved [212] __attribute__((packed));
}MODEINFOREC;//Information about a specific videomode
#endif

//////////////////////////////////////////////////////////////////////
// Engine Structures/Classes
//////////////////////////////////////////////////////////////////////

typedef struct{
   unsigned short *CharStart;
   unsigned short *CharWidth;
   unsigned long *Data32;
   unsigned short *Data16;
   unsigned char *Data8;
   unsigned short Height;
   unsigned short Width;
   unsigned char ColorDepth;
}FONTINFO;

typedef struct{
   unsigned long *Data32;
   unsigned short *Data16;
   unsigned char *Data8;
   unsigned long NumOfImages;
   unsigned short Height;
   unsigned short Width;
   unsigned char ColorDepth;
}IMAGEINFO;

typedef struct{
   unsigned long *Data32;
   unsigned short *Data16;
   unsigned char *Data8;
   unsigned char *PaletteLookup;
   unsigned long BankSize;
   unsigned short Height;
   unsigned short Width;
   unsigned short CurrentBank;
   unsigned short VESAFound;
   unsigned short LineWidth;
   unsigned char ColorDepth;
   bool LFBFound;
   bool RecreateLookupTable;
   bool Locked;
}SCREENINFO;

typedef struct{
   unsigned long *Data32;
   unsigned short *Data16;
   unsigned char *Data8;
   unsigned short Height;
   unsigned short Width;
   unsigned char ColorDepth;
}BUFFERINFO;

class IMAGECLASS;
class FONTCLASS;
class BUFFERCLASS;
class SCREENCLASS;
class GFXCLASS;

class IMAGECLASS{
   private:
      void FreeImage(void);
      SCREENCLASS *screenptr;
      unsigned long *ImageData32;
      unsigned short *ImageData16;
      unsigned char *ImageData8;
      unsigned long ImagesLoaded;
      unsigned short Width;
      unsigned short Height;
      unsigned char ColorDepth;
   public:
      IMAGECLASS(SCREENCLASS *);
      void Load(char *,unsigned long);
      void LoadBmp(char *);
      void LoadMultiBmp(char *,unsigned short,unsigned short);
      void Make(unsigned short,unsigned short,unsigned char *);
      void Make(unsigned short,unsigned short,unsigned short *);
      void Make15(unsigned short,unsigned short,unsigned short *);
      void Make(unsigned short,unsigned short,unsigned long *);
      void Duplicate(IMAGECLASS *);
      bool GetInfo(IMAGEINFO *);
      void Grab(short,short,short,short,SCREENCLASS *);
      void Grab(short,short,short,short,BUFFERCLASS *);
      void Convert(unsigned char);
      void Release(void);
};

class FONTCLASS{
   private:
      void FreeFont(void);
      void CalculateWidth(void);
      SCREENCLASS *screenptr;
      unsigned long *FontData32;
      unsigned short *FontData16;
      unsigned short *fstart;
      unsigned short *fwidth;
      unsigned char *FontData8;
      unsigned short Width;
      unsigned short Height;
      unsigned char ColorDepth;
      bool Loaded;
   public:
      FONTCLASS(SCREENCLASS *);
      void Load(char *);
      void LoadBmp(char *);
      void Duplicate(FONTCLASS *);
      bool GetInfo(FONTINFO *);
      void Convert(unsigned char);
      void Release(void);
};

class BUFFERCLASS{
   private:
      SCREENCLASS *screenptr;
      unsigned long OffsTable[1200];
      unsigned long *BufMem32;
      unsigned short *BufMem16;
      unsigned char *BufMem8;
      unsigned short XRes;
      unsigned short YRes;
      unsigned char ColorDepth;
   public:
      BUFFERCLASS(SCREENCLASS *,unsigned short,unsigned short,unsigned char);
      bool GetInfo(BUFFERINFO *);
      void Clear(void);
      void PutPixel(unsigned short,unsigned short,unsigned long);
      unsigned long GetPixel(unsigned short,unsigned short);
      void HLine(short,short,short,unsigned long);
      void VLine(short,short,short,unsigned long);
      void Line(short,short,short,short,unsigned long);
      void Rectangle(short,short,short,short,unsigned long);
      void Bar(short,short,short,short,unsigned long);
      void PutFontChar(short,short,unsigned char,FONTCLASS *);
      void WriteFont(short,short,char *,FONTCLASS *);
      void WriteFont(short,short,unsigned long,FONTCLASS *);
      void WriteFontAlign(short,short,char *,FONTCLASS *);
      void WriteFontAlign(short,short,unsigned long,FONTCLASS *);
      void Blit(short,short,unsigned long,IMAGECLASS *);
      void BlitColorKey(short,short,unsigned long,unsigned long,IMAGECLASS *);
      void AlphaBlend(short,short,unsigned long,unsigned long,IMAGECLASS *);
      void AlphaBlendAdd(short,short,unsigned long,unsigned long,IMAGECLASS *);
      void AlphaBlendSemiAdd(short,short,unsigned long,unsigned long,IMAGECLASS *);
      void Flip(SCREENCLASS *);
      void Flip(BUFFERCLASS *);
      void Convert(unsigned short,unsigned short,unsigned char);
      void Release(void);
};

class SCREENCLASS{
   private:
      #ifdef DJGPP
	  unsigned short FindMode(unsigned short,unsigned short,unsigned char);
      unsigned short QueryVESA(void);
      bool QueryLFB(unsigned short);
      unsigned long GetCardMemOfs(unsigned short);
      void SetMode(unsigned short);
      void SetVars(unsigned short);
      #endif
      void ConvertToTempScreen(unsigned short,unsigned short,unsigned char);
      void ShowTempScreen(void);
      #ifdef WIN32
      HWND windowhandle;
      LPDIRECTDRAW7 ddraw;
      LPDIRECTDRAWSURFACE7 dsurface;
      DDSURFACEDESC2 ddesc;
      DDPIXELFORMAT dpixinfo;
      PALETTEENTRY palentry[256];
      LPDIRECTDRAWPALETTE palapi;
      #endif
      unsigned long OffsTable[1200];
      unsigned char defaultpal[768];
      unsigned long *Surface32;
      unsigned short *Surface16;
      unsigned char *Surface8;
      unsigned char *TempScreen;
      unsigned char *PaletteLookup;
      unsigned long memaddress;
      unsigned long memsize;
      unsigned long BankSize;
	  unsigned short Width;
      unsigned short Height;
      unsigned short CurrentBank;
      unsigned short CurrentVideoMode;
      unsigned short VESAFound;
      unsigned char ColorDepth;
      bool LFBEnabled;
      bool UpdateLookupTable;
	  bool Locked;
   public:
      #ifdef DJGPP
	  SCREENCLASS();
      #endif
      #ifdef WIN32
      SCREENCLASS(HWND);
      #endif
	  bool CreateFont(FONTCLASS *&);
      bool CreateImage(IMAGECLASS *&);
      void SwitchBank(unsigned short);
      bool GetInfo(SCREENINFO *);
      bool SetGfxMode(unsigned short,unsigned short,unsigned char);
      void CloseGfxMode(void);
      void VSync(void);
      void Clear(void);
      void PutPixel(unsigned short,unsigned short,unsigned long);
      unsigned long GetPixel(unsigned short,unsigned short);
      void HLine(short,short,short,unsigned long);
      void VLine(short,short,short,unsigned long);
      void Line(short,short,short,short,unsigned long);
      void Rectangle(short,short,short,short,unsigned long);
      void Bar(short,short,short,short,unsigned long);
      void SetPalette(unsigned char *);
      void GetCurrentPalette(unsigned char *);
      void LoadPalette(char *,unsigned char *);
      void LoadBmpPalette(char *,unsigned char *);
      void SavePalette(char *,unsigned char *);
      void PutFontChar(short,short,unsigned char,FONTCLASS *);
      void WriteFont(short,short,char *,FONTCLASS *);
      void WriteFont(short,short,unsigned long,FONTCLASS *);
      void WriteFontAlign(short,short,char *,FONTCLASS *);
      void WriteFontAlign(short,short,unsigned long,FONTCLASS *);
      void Blit(short,short,unsigned long,IMAGECLASS *);
      void BlitColorKey(short,short,unsigned long,unsigned long,IMAGECLASS *);
      void AlphaBlend(short,short,unsigned long,unsigned long,IMAGECLASS *);
      void AlphaBlendAdd(short,short,unsigned long,unsigned long,IMAGECLASS *);
      void AlphaBlendSemiAdd(short,short,unsigned long,unsigned long,IMAGECLASS *);
      void CreatePaletteLookupTable(void);
      void RecreateLookupTable(bool);
      bool AttachBuffer(BUFFERCLASS *&);
      void Flip(BUFFERCLASS *);
	  void Lock(void);
	  void Unlock(void);
      void Release(void);
};

class GFXCLASS{
   #ifdef WIN32
   private:
	  bool windowcreated;
	  HWND windowhandle;
	  HINSTANCE instancehandle;
   #endif
   public:
      GFXCLASS();
	  bool CreateScreen(SCREENCLASS *&);
      #ifdef WIN32
      bool MakeWindow(HINSTANCE);
	  bool RegisterWindow(HWND);
      #endif
	  void Release(void);
};

//////////////////////////////////////////////////////////////////////
// New types
//////////////////////////////////////////////////////////////////////

typedef GFXCLASS * GFXENGINE;
typedef SCREENCLASS * SCREEN;
typedef BUFFERCLASS * BUFFER;
typedef FONTCLASS * FONT;
typedef IMAGECLASS * IMAGE;
typedef unsigned char PALETTE[768];

//////////////////////////////////////////////////////////////////////
// Global functions
//////////////////////////////////////////////////////////////////////

void memset8(void *dest,unsigned char value,unsigned long msize){
   #ifdef DJGPP
      __asm__("
         movl %1, %%eax\n
         movl %2, %%ecx\n
         cld\n\t
         rep\n\t
         stosb"
         :
         : "D" (dest), "g" (value), "g" (msize)
         : "eax", "ecx", "memory"
      );
   #endif
   #ifdef WIN32
	  _asm{
	     mov edi, dest
		 mov ecx, msize
		 mov al, value
		 rep stosb
	  };
   #endif
};

void memset16(void *dest,unsigned short value,unsigned long msize){
   #ifdef DJGPP
      __asm__("
         movl %1, %%eax\n
         movl %2, %%ecx\n
         cld\n\t
         rep\n\t
         stosw"
         :
         : "D" (dest), "g" (value), "g" (msize)
         : "eax", "ecx", "memory"
      );
   #endif
   #ifdef WIN32
	  _asm{
	     mov edi, dest
		 mov ecx, msize
		 mov ax, value
		 rep stosw
	  };
   #endif
};

void memset32(void *dest,unsigned long value,unsigned long msize){
   #ifdef DJGPP
      __asm__("
         movl %1, %%eax\n
         movl %2, %%ecx\n
         cld\n\t
         rep\n\t
         stosl"
         :
         : "D" (dest), "g" (value), "g" (msize)
         : "eax", "ecx", "memory"
      );
   #endif
   #ifdef WIN32
	  _asm{
	     mov edi, dest
		 mov ecx, msize
		 mov eax, value
		 rep stosd
	  };
   #endif
};

bool CreateEngine(GFXENGINE &gfxengine){
   gfxengine=new GFXCLASS();
   if (gfxengine==NULL) return(false);
   return(true);
};

#ifdef WIN32
long WINAPI GfxMsgHandler(HWND hwnd,UINT wmessage,WPARAM wparam,LPARAM lparam){
   return(DefWindowProc(hwnd,wmessage,wparam,lparam));
};
#endif

//////////////////////////////////////////////////////////////////////
// GFXCLASS member functions (public)
//////////////////////////////////////////////////////////////////////

GFXCLASS::GFXCLASS(){
   #ifdef WIN32
      windowcreated=false;
   #endif
};

bool GFXCLASS::CreateScreen(SCREENCLASS *&screen){
   #ifdef DJGPP
      screen=new SCREENCLASS();
   #endif
   #ifdef WIN32
      screen=new SCREENCLASS(windowhandle);
   #endif
   if (screen==NULL) return(false);
   return(true);
};

#ifdef WIN32
bool GFXCLASS::MakeWindow(HINSTANCE hinstance){
   WNDCLASS windclass;
   instancehandle=hinstance;
   windclass.style=0;
   windclass.lpfnWndProc=GfxMsgHandler;
   windclass.cbClsExtra=0;
   windclass.cbWndExtra=0;
   windclass.hInstance=instancehandle;
   windclass.hIcon=LoadIcon(instancehandle,IDI_APPLICATION);
   windclass.hCursor=LoadCursor(instancehandle,IDC_ARROW);
   windclass.hbrBackground=(HBRUSH__*)GetStockObject(BLACK_BRUSH);
   windclass.lpszMenuName=NULL;
   windclass.lpszClassName="graphics engine";
   RegisterClass(&windclass);
   windowhandle=CreateWindow("graphics engine","initializing...",WS_OVERLAPPED,CW_USEDEFAULT,0,0,0,NULL,NULL,instancehandle,NULL);
   SetWindowLong(windowhandle,GWL_STYLE,WS_POPUP || WS_VISIBLE);
   ShowWindow(windowhandle,SW_SHOW);
   SetWindowPos(windowhandle,HWND_TOPMOST,0,0,0,0,SWP_NOMOVE);
   windowcreated=true;
   return(true);
};

bool GFXCLASS::RegisterWindow(HWND hwnd){
   windowhandle=hwnd;
   instancehandle=(HINSTANCE)GetWindowLong(windowhandle,GWL_HINSTANCE);
   windowcreated=true;
   return(windowcreated);
};
#endif

void GFXCLASS::Release(void){
   if (this!=NULL) delete(this);
};

//////////////////////////////////////////////////////////////////////
// SCREENCLASS member functions (private)
//////////////////////////////////////////////////////////////////////

#ifdef DJGPP
unsigned short SCREENCLASS::FindMode(unsigned short xres,unsigned short yres,unsigned char bpp){
   MODEINFOREC modeinfo;
   __dpmi_regs DRegs;
   unsigned short Mode;
   VESAFound=QueryVESA();
   if (xres==320 && yres==200 && bpp==8) return(0x13);
   if (!VESAFound) return(0);
   for (Mode=256;Mode<65535;Mode++){
      DRegs.x.ax=0x4F01;
      DRegs.x.cx=Mode;
      DRegs.x.di=0;
      DRegs.x.es=__tb >> 4;
      __dpmi_int(0x10,&DRegs);
      dosmemget(__tb,sizeof(modeinfo),&modeinfo);
      if (modeinfo.xresolution==xres && modeinfo.yresolution==yres && modeinfo.bitsperpixel==bpp) return(Mode);
   };
   return(0);
};

unsigned short SCREENCLASS::QueryVESA(void){
   VBEINFOREC vbeinfo;
   __dpmi_regs DRegs;
   DRegs.x.ax=0x4F00;
   DRegs.x.di=0;
   DRegs.x.es=__tb >> 4;
   __dpmi_int(0x10,&DRegs);
   if(DRegs.x.ax!=0x4F) return(0);
   dosmemget(__tb,sizeof(vbeinfo),&vbeinfo);
   return(vbeinfo.version);
};

bool SCREENCLASS::QueryLFB(unsigned short Mode){
   MODEINFOREC modeinfo;
   __dpmi_regs DRegs;
   DRegs.x.ax=0x4F01;
   DRegs.x.cx=Mode;
   DRegs.x.di=0;
   DRegs.x.es=__tb >> 4;
   __dpmi_int(0x10,&DRegs);
   dosmemget(__tb,sizeof(modeinfo),&modeinfo);
   if (modeinfo.modeattributes&0x08) return(true);
   return(false);
};

unsigned long SCREENCLASS::GetCardMemOfs(unsigned short Mode){
   MODEINFOREC modeinfo;
   __dpmi_regs DRegs;
   DRegs.x.ax=0x4F01;
   DRegs.x.cx=Mode;
   DRegs.x.di=0;
   DRegs.x.es=__tb >> 4;
   __dpmi_int(0x10,&DRegs);
   dosmemget(__tb,sizeof(modeinfo),&modeinfo);
   return(modeinfo.physbaseptr);
};

void SCREENCLASS::SetMode(unsigned short Mode){
   __dpmi_regs DRegs;
   DRegs.x.ax=0x4F02;
   DRegs.x.bx=Mode;
   if (LFBEnabled && Mode>255) DRegs.x.bx+=0x4000;
   __dpmi_int(0x10,&DRegs);
};

void SCREENCLASS::SetVars(unsigned short Mode){
   MODEINFOREC modeinfo;
   __dpmi_regs DRegs;
   if (VESAFound>0x1FF) LFBEnabled=QueryLFB(Mode); else LFBEnabled=false;
   if (Mode>255){
      DRegs.x.ax=0x4F01;
      DRegs.x.cx=Mode;
      DRegs.x.di=0;
      DRegs.x.es=__tb >> 4;
      __dpmi_int(0x10,&DRegs);
      dosmemget(__tb,sizeof(modeinfo),&modeinfo);
      Width=modeinfo.xresolution;
      Height=modeinfo.yresolution;
      ColorDepth=modeinfo.bitsperpixel;
      BankSize=modeinfo.wingranularity*1024;
      return;
   };
   if (Mode==0x13){
      Width=320;
      Height=200;
      ColorDepth=8;
   };
};
#endif

void SCREENCLASS::ConvertToTempScreen(unsigned short toxres,unsigned short toyres,unsigned char tobpp){
   unsigned long Color;
   unsigned short xctr,yctr;
   unsigned char re,gr,bl;
   bool NoLock=Locked;
   if (tobpp==15) TempScreen=(unsigned char *)malloc(toxres*toyres*2);
   else TempScreen=(unsigned char *)malloc(toxres*toyres*tobpp/8);
   if (tobpp==8 && UpdateLookupTable) CreatePaletteLookupTable();
   if (!NoLock) Lock();
   for (yctr=0;yctr<toyres;yctr++) for (xctr=0;xctr<toxres;xctr++){
      Color=GetPixel(xctr,yctr);
      if (ColorDepth==8){
         re=defaultpal[Color*3]>>1;
         gr=defaultpal[Color*3+1]>>1;
         bl=defaultpal[Color*3+2]>>1;
      };
      if (ColorDepth==15){
         bl=(unsigned char)(Color&31);
         gr=(unsigned char)((Color&992)>>5);
         re=(unsigned char)((Color&31744)>>10);
      };
      if (ColorDepth==16){
         bl=(unsigned char)(Color&31);
         gr=(unsigned char)((Color&1984)>>6);
         re=(unsigned char)((Color&63488)>>11);
      };
      if (ColorDepth==32){
         bl=(unsigned char)((Color>>3)&31);
         gr=(unsigned char)((Color>>11)&31);
         re=(unsigned char)(Color>>19);
      };
      if (tobpp==8) Color=PaletteLookup[(re<<10)+(gr<<5)+bl];
      if (tobpp==15) Color=(re<<10)+(gr<<5)+bl;
      if (tobpp==16) Color=(re<<11)+(gr<<6)+bl;
      if (tobpp==32) Color=(re<<19)+(gr<<11)+(bl<<3);
      if (tobpp==15) memcpy(&TempScreen[yctr*toxres*2+xctr*2],&Color,2);
      else memcpy(&TempScreen[yctr*toxres*tobpp/8+xctr*tobpp/8],&Color,tobpp/8);
   };
   if (!NoLock) Unlock();
};

void SCREENCLASS::ShowTempScreen(void){
   unsigned long Color;
   unsigned short xctr,yctr;
   bool NoLock=Locked;
   if (!NoLock) Lock();
   for (yctr=0;yctr<Height;yctr++) for (xctr=0;xctr<Width;xctr++){
      if (ColorDepth==15) memcpy(&Color,&TempScreen[yctr*Width*2+xctr*2],2);
      else memcpy(&Color,&TempScreen[yctr*Width*ColorDepth/8+xctr*ColorDepth/8],ColorDepth/8);
      PutPixel(xctr,yctr,Color);
   };
   if (!NoLock) Unlock();
   free(TempScreen);
   TempScreen=NULL;
};

//////////////////////////////////////////////////////////////////////
// SCREENCLASS member functions (public)
//////////////////////////////////////////////////////////////////////

#ifdef DJGPP
SCREENCLASS::SCREENCLASS(){
#endif
#ifdef WIN32
SCREENCLASS::SCREENCLASS(HWND hwnd){
   windowhandle=hwnd;
   ddraw=NULL;
   dsurface=NULL;
   palapi=NULL;
#endif
   unsigned char ctr,ctr2;
   CurrentVideoMode=0;
   Width=0;
   Height=0;
   ColorDepth=0;
   BankSize=0;
   LFBEnabled=false;
   UpdateLookupTable=true;
   Locked=false;
   VESAFound=0;
   PaletteLookup=NULL;
   TempScreen=NULL;
   for (ctr2=0;ctr2<4;ctr2++) for (ctr=0;ctr<64;ctr++){
      defaultpal[ctr2*192+ctr*3]=ctr/16*16+ctr2*4;
      defaultpal[ctr2*192+ctr*3+1]=ctr/4%4*16+ctr2*4;
      defaultpal[ctr2*192+ctr*3+2]=ctr%4*16+ctr2*4;
   };
};

bool SCREENCLASS::CreateFont(FONTCLASS *&font){
   font=new FONTCLASS(this);
   if (font==NULL) return(false);
   return(true);
};

bool SCREENCLASS::CreateImage(IMAGECLASS *&image){
   image=new IMAGECLASS(this);
   if (image==NULL) return(false);
   return(true);
};

void SCREENCLASS::SwitchBank(unsigned short ToBank){
   #ifdef DJGPP
   if (ToBank==CurrentBank) return;
   __dpmi_regs DRegs;
   DRegs.x.ax=0x4F05;
   DRegs.x.bx=0;
   DRegs.x.dx=ToBank;
   __dpmi_int(0x10,&DRegs);
   CurrentBank=ToBank;
   #endif
};

bool SCREENCLASS::GetInfo(SCREENINFO *info){
   info->Data32=Surface32;
   info->Data16=Surface16;
   info->Data8=Surface8;
   info->PaletteLookup=PaletteLookup;
   info->Height=Height;
   info->Width=Width;
   info->CurrentBank=CurrentBank;
   info->BankSize=BankSize;
   info->ColorDepth=ColorDepth;
   info->LFBFound=LFBEnabled;
   info->VESAFound=VESAFound;
   info->RecreateLookupTable=UpdateLookupTable;
   info->Locked=Locked;
   info->LineWidth=(unsigned short)OffsTable[1];
   return(true);
};

bool SCREENCLASS::SetGfxMode(unsigned short xres,unsigned short yres,unsigned char bpp){
   if (bpp==24) return(false);
   if (Locked) Unlock();
   #ifdef DJGPP
   __dpmi_meminfo info;
   __dpmi_regs DRegs;
   unsigned short NewVideoMode;
   NewVideoMode=FindMode(xres,yres,bpp);
   if (NewVideoMode==0) return(false);
   #endif
   unsigned short Counter;
   if (xres==Width && yres==Height && bpp==ColorDepth) return(true);
   if (Width!=0){
      ConvertToTempScreen(xres,yres,bpp);
      CloseGfxMode();
   }
   #ifdef DJGPP
   SetVars(NewVideoMode);
   if (NewVideoMode>255){
      if (LFBEnabled){
         if (bpp==15) info.size=Width*Height*2;
         else info.size=Width*Height*bpp/8;
      } else info.size=65536;
      if (LFBEnabled) info.address=GetCardMemOfs(NewVideoMode); else info.address=0xA0000;
   } else {
      info.size=xres*yres*bpp/8;
      info.address=0xA0000;
   };
   memsize=info.size;
   memaddress=info.address;
   __dpmi_physical_address_mapping(&info);
   if (bpp==8) Surface8=(unsigned char *)(info.address+__djgpp_conventional_base);
   if (bpp==15 || bpp==16) Surface16=(unsigned short *)(info.address+__djgpp_conventional_base);
   if (bpp==32) Surface32=(unsigned long *)(info.address+__djgpp_conventional_base);
   SetMode(NewVideoMode);
   if (bpp==8) SetPalette(defaultpal);
   #endif
   #ifdef WIN32
   ShowCursor(0);
   LPDIRECTDRAW ddrawtemp;
   if (DirectDrawCreate(NULL,&ddrawtemp,NULL)!=DD_OK) return(false);
   if (ddrawtemp->QueryInterface(IID_IDirectDraw7,(void**)&ddraw)!=DD_OK) return(false);
   ddrawtemp->Release();
   if (ddraw->SetCooperativeLevel(windowhandle,DDSCL_EXCLUSIVE|DDSCL_FULLSCREEN|DDSCL_ALLOWREBOOT)!=DD_OK) return(false);
   if (bpp==8 && xres==320 && yres==200){
	  if (ddraw->SetDisplayMode(320,200,8,0,DDSDM_STANDARDVGAMODE)!=DD_OK) return(false);
   } else {
      if (bpp!=15) if (ddraw->SetDisplayMode(xres,yres,bpp,0,NULL)!=DD_OK) return(false);
      if (bpp==15) if (ddraw->SetDisplayMode(xres,yres,16,0,NULL)!=DD_OK) return(false);
   };
   memset(&ddesc,0,sizeof(ddesc));
   ddesc.dwSize=sizeof(ddesc);
   ddesc.dwFlags=DDSD_CAPS;
   ddesc.ddsCaps.dwCaps=DDSCAPS_PRIMARYSURFACE;
   ddesc.dwBackBufferCount=0;
   if (ddraw->CreateSurface(&ddesc,&dsurface,NULL)!=DD_OK) return(false);
   dpixinfo.dwSize=sizeof(DDPIXELFORMAT);
   if (bpp==15 || bpp==16){
      if (dsurface->GetPixelFormat(&dpixinfo)!=DD_OK) return(false);
      if (dpixinfo.dwGBitMask==992 && bpp==16) return(false);
	  if (dpixinfo.dwGBitMask!=992 && bpp==15) return(false);
   };
   if (bpp==8) SetPalette(defaultpal);
   dsurface->Lock(NULL,&ddesc,DDLOCK_WAIT,NULL);
   dsurface->Unlock(NULL);
   if (bpp==15) for (Counter=0;Counter<yres;Counter++) OffsTable[Counter]=ddesc.lPitch/2*Counter;
   if (bpp!=15) for (Counter=0;Counter<yres;Counter++) OffsTable[Counter]=ddesc.lPitch/(bpp/8)*Counter;
   ColorDepth=bpp;
   Width=xres;
   Height=yres;
   LFBEnabled=true;
   #endif
   CurrentBank=65535;
   if (!LFBEnabled && CurrentVideoMode>255) SwitchBank(0);
   if (TempScreen!=NULL) ShowTempScreen();
   #ifdef DJGPP
   CurrentVideoMode=NewVideoMode;
   for (Counter=0;Counter<yres;Counter++) OffsTable[Counter]=Counter*xres;
   #endif
   Locked=false;
   return(true);
};

void SCREENCLASS::CloseGfxMode(void){
   if (Locked) Unlock();
   #ifdef DJGPP
   __dpmi_meminfo info;
   __dpmi_regs DRegs;
   DRegs.x.ax=0x3;
   __dpmi_int(0x10,&DRegs);
   LFBEnabled=false;
   VESAFound=0;
   CurrentVideoMode=0;
   info.size=memsize;
   info.address=memaddress;
   __dpmi_free_physical_address_mapping(&info);
   #endif
   #ifdef WIN32
   if (ddraw!=NULL){
	  if (dsurface!=NULL){
		 if (ColorDepth==8 && palapi!=NULL){
			palapi->Release();
			palapi=NULL;
		 };
		 dsurface->Release();
		 dsurface=NULL;
	  };
	  ddraw->Release();
	  ddraw=NULL;
   };
   ShowCursor(1);
   #endif
   Width=0;
   Height=0;
   ColorDepth=0;
   Locked=false;
};

void SCREENCLASS::VSync(void){
   #ifdef DJGPP
   while (inp(0x3DA) & 0x8);
   while (!(inp(0x3DA) & 0x8));
   #endif
   #ifdef WIN32
   ddraw->WaitForVerticalBlank(DDWAITVB_BLOCKBEGIN,NULL);
   #endif
};

void SCREENCLASS::Clear(void){
   #ifdef DJGPP
   unsigned long ScreenSize,Counter;
   if (ColorDepth==15) ScreenSize=Height*Width*2;
   else ScreenSize=Height*Width*ColorDepth/8;
   if (LFBEnabled || CurrentVideoMode<256){
      if (ColorDepth==8) memset8(Surface8,0,ScreenSize);
      if (ColorDepth==15 || ColorDepth==16) memset8(Surface16,0,ScreenSize);
      if (ColorDepth==32) memset8(Surface32,0,ScreenSize);
   } else {
      for (Counter=0;Counter<ScreenSize/65536;Counter++){
         SwitchBank(Counter*(65536/BankSize));
         if (ColorDepth==8) memset8(Surface8,0,65536);
         if (ColorDepth==15 || ColorDepth==16) memset8(Surface16,0,65536);
         if (ColorDepth==32) memset8(Surface32,0,65536);
      };
      if (ScreenSize%65536){
         SwitchBank(ScreenSize/65536*(65536/BankSize));
         if (ColorDepth==8) memset8(Surface8,0,ScreenSize%65536);
         if (ColorDepth==15 || ColorDepth==16) memset8(Surface16,0,ScreenSize%65536);
         if (ColorDepth==32) memset8(Surface32,0,ScreenSize%65536);
      };
   };
   #endif
   #ifdef WIN32
   unsigned short Counter;
   bool NoLock=Locked;
   if (!NoLock) Lock();
   for (Counter=0;Counter<Height;Counter++){
      if (ColorDepth==8) memset8(&Surface8[OffsTable[Counter]],0,OffsTable[1]);
      if (ColorDepth==15 || ColorDepth==16) memset16(&Surface16[OffsTable[Counter]],0,OffsTable[1]);
      if (ColorDepth==32) memset32(&Surface32[OffsTable[Counter]],0,OffsTable[1]);
   };
   if (!NoLock) Unlock();
   #endif
};

void SCREENCLASS::PutPixel(unsigned short x,unsigned short y,unsigned long Color){
   bool NoLock=Locked;
   if (x>=Width) return;
   if (y>=Height) return;
   if (!NoLock) Lock();
   if (LFBEnabled){
      if (ColorDepth==8) Surface8[OffsTable[y]+x]=(unsigned char)(Color%256);
      if (ColorDepth==15 || ColorDepth==16) Surface16[OffsTable[y]+x]=(unsigned short)(Color%65536);
      if (ColorDepth==32) Surface32[OffsTable[y]+x]=Color;
   } else {
      #ifdef DJGPP
	  if (ColorDepth==15) if ((OffsTable[y]+x)*2<CurrentBank*BankSize || (OffsTable[y]+x)*2>CurrentBank*BankSize+65535) SwitchBank((OffsTable[y]+x)*2/BankSize);
      if (ColorDepth!=15) if ((OffsTable[y]+x)*ColorDepth/8<CurrentBank*BankSize || (OffsTable[y]+x)*ColorDepth/8>CurrentBank*BankSize+65535) SwitchBank((OffsTable[y]+x)*ColorDepth/8/BankSize);
      if (ColorDepth==8) Surface8[(OffsTable[y]+x)-CurrentBank*BankSize]=Color%256;
      if (ColorDepth==15 || ColorDepth==16) Surface16[(OffsTable[y]+x)-CurrentBank*BankSize/2]=Color%65536;
      if (ColorDepth==32) Surface32[(OffsTable[y]+x)-CurrentBank*BankSize/4]=Color;
      #endif
   };
   if (!NoLock) Unlock();
};

unsigned long SCREENCLASS::GetPixel(unsigned short x,unsigned short y){
   bool NoLock=Locked;
   if (x>=Width) return(0);
   if (y>=Height) return(0);
   if (!NoLock) Lock();
   if (LFBEnabled){
	  if (ColorDepth==8){
         if (!NoLock) Unlock();
	     return(Surface8[OffsTable[y]+x]);
	  };
	  if (ColorDepth==15 || ColorDepth==16){
         if (!NoLock) Unlock();
	     return(Surface16[OffsTable[y]+x]);
	  };
      if (ColorDepth==32){
         if (!NoLock) Unlock();
		 return(Surface32[OffsTable[y]+x]);
	  };
   } else {
      #ifdef DJGPP
	  if (ColorDepth==15) if ((OffsTable[y]+x)*2<CurrentBank*BankSize || (OffsTable[y]+x)*2>CurrentBank*BankSize+65535) SwitchBank((OffsTable[y]+x)*2/BankSize);
      if (ColorDepth!=15) if ((OffsTable[y]+x)*ColorDepth/8<CurrentBank*BankSize || (OffsTable[y]+x)*ColorDepth/8>CurrentBank*BankSize+65535) SwitchBank((OffsTable[y]+x)*ColorDepth/8/BankSize);
      if (ColorDepth==8) return(Surface8[(OffsTable[y]+x)-CurrentBank*BankSize]);
      if (ColorDepth==15 || ColorDepth==16) return(Surface16[(OffsTable[y]+x)-CurrentBank*BankSize/2]);
      if (ColorDepth==32) return(Surface32[(OffsTable[y]+x)-CurrentBank*BankSize/4]);
      #endif
   };
   if (!NoLock) Unlock();
   return(0);
};

void SCREENCLASS::HLine(short x,short x2,short y,unsigned long Color){
   short Counter,tempx;
   bool NoLock=Locked;
   if (x>x2){
      tempx=x2;
      x2=x;
      x=tempx;
   };
   if (!NoLock) Lock();
   for (Counter=x;Counter<=x2;Counter++) PutPixel(Counter,y,Color);
   if (!NoLock) Unlock();
};

void SCREENCLASS::VLine(short y,short y2,short x,unsigned long Color){
   short Counter,tempy;
   bool NoLock=Locked;
   if (y>y2){
      tempy=y2;
      y2=y;
      y=tempy;
   };
   if (!NoLock) Lock();
   for (Counter=y;Counter<=y2;Counter++) PutPixel(x,Counter,Color);
   if (!NoLock) Unlock();
};

void SCREENCLASS::Line(short x,short y,short x2,short y2,unsigned long Color){
   short XLen,YLen,XPlus=0,YPlus=0,Step=0;
   unsigned short Counter;
   char YUp,XUp;
   bool NoLock=Locked;
   if (y==y2){
      HLine(x,x2,y,Color);
      return;
   };
   if (x==x2){
      VLine(y,y2,x,Color);
      return;
   };
   if (!NoLock) Lock();
   XLen=x2-x;
   YLen=y2-y;
   if (XLen>=0) XUp=1; else {
      XUp=-1;
      XLen=-XLen;
   };
   if (YLen>=0) YUp=1; else {
      YUp=-1;
      YLen=-YLen;
   };
   if (XLen>YLen){
      for (Counter=0;Counter<=XLen;Counter++){
         PutPixel(x+XPlus,y+YPlus,Color);
         Step+=YLen;
         if (Step>XLen){
            Step-=XLen;
            YPlus+=YUp;
         };
         XPlus+=XUp;
      };
   } else for (Counter=0;Counter<=YLen;Counter++){
      PutPixel(x+XPlus,y+YPlus,Color);
      Step+=XLen;
      if (Step>0){
         Step-=YLen;
         XPlus+=XUp;
      };
      YPlus+=YUp;
   };
   if (!NoLock) Unlock();
};

void SCREENCLASS::Rectangle(short x,short y,short x2,short y2,unsigned long Color){
   bool NoLock=Locked;
   if (!NoLock) Lock();
   HLine(x,x2,y,Color);
   HLine(x,x2,y2,Color);
   VLine(y,y2,x,Color);
   VLine(y,y2,x2,Color);
   if (!NoLock) Unlock();
};

void SCREENCLASS::Bar(short x,short y,short x2,short y2,unsigned long Color){
   short YCtr;
   bool NoLock=Locked;
   if (!NoLock) Lock();
   for (YCtr=y;YCtr<=y2;YCtr++) HLine(x,x2,YCtr,Color);
   if (!NoLock) Unlock();
};

void SCREENCLASS::SetPalette(unsigned char *PaletteData){
   unsigned short Counter;
   memcpy(defaultpal,PaletteData,sizeof(PALETTE));
   #ifdef DJGPP
   outp(0x3C8,0);
   for (Counter=0;Counter<768;Counter+=3){
      outp(0x3C9,PaletteData[Counter]);
      outp(0x3C9,PaletteData[Counter+1]);
      outp(0x3C9,PaletteData[Counter+2]);
   };
   #endif
   #ifdef WIN32
   for (Counter=0;Counter<256;Counter++){
      palentry[Counter].peRed=PaletteData[Counter*3]<<2;
      palentry[Counter].peGreen=PaletteData[Counter*3+1]<<2;
      palentry[Counter].peBlue=PaletteData[Counter*3+2]<<2;
	  palentry[Counter].peFlags=PC_NOCOLLAPSE;
   };
   if (palapi!=NULL) palapi->Release();
   ddraw->CreatePalette(DDPCAPS_8BIT|DDPCAPS_ALLOW256,palentry,&palapi,NULL);
   dsurface->SetPalette(palapi);
   #endif
};

void SCREENCLASS::GetCurrentPalette(unsigned char *PaletteData){
   memcpy(PaletteData,defaultpal,768);
};

void SCREENCLASS::LoadPalette(char *filename,unsigned char *palette){
   fstream palettefile;
   palettefile.open(filename,0x81);
   palettefile.read(palette,768);
   palettefile.close();
};

void SCREENCLASS::LoadBmpPalette(char *filename,unsigned char *palette){
   fstream bmpfile;
   unsigned long rd,headersize;
   unsigned short ctr,bmpbpp;
   bmpfile.open(filename,0x81);
   bmpfile.read((char *)&rd,2);
   if (rd!=19778){
      bmpfile.close();
      return;
   };
   bmpfile.seekg(10);
   bmpfile.read((char *)&headersize,4);
   bmpfile.read((char *)&rd,4);
   if (rd!=40){
      bmpfile.close();
      return;
   };
   bmpfile.seekg(28);
   bmpfile.read((char *)&bmpbpp,2);
   if (bmpbpp!=8){
      bmpfile.close();
      return;
   };
   bmpfile.read((char *)&rd,4);
   if (rd!=0){
      bmpfile.close();
      return;
   };
   bmpfile.seekg(headersize-1024);
   for (ctr=0;ctr<256;ctr++){
      bmpfile.read((char *)&rd,4);
      palette[ctr*3]=(unsigned char)((rd>>18)&63);
      palette[ctr*3+1]=(unsigned char)((rd>>10)&63);
      palette[ctr*3+2]=(unsigned char)((rd>>2)&63);
   };
   bmpfile.close();
};

void SCREENCLASS::SavePalette(char *filename,unsigned char *palette){
   fstream palettefile;
   palettefile.open(filename,0x92);
   palettefile.write(palette,768);
   palettefile.close();
};

void SCREENCLASS::PutFontChar(short x,short y,unsigned char chr,FONTCLASS *font){
   FONTINFO info;
   unsigned long Color;
   unsigned short XCtr,YCtr;
   bool NoLock=Locked;
   if (!font->GetInfo(&info)) return;
   if (info.ColorDepth!=ColorDepth){
      font->Convert(ColorDepth);
      if (!font->GetInfo(&info)) return;
   };
   if (!NoLock) Lock();
   for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
      if (info.ColorDepth==8) Color=info.Data8[chr*info.Height*info.Width+YCtr*info.Width+XCtr];
      if (info.ColorDepth==15 || info.ColorDepth==16) Color=info.Data16[chr*info.Height*info.Width+YCtr*info.Width+XCtr];
      if (info.ColorDepth==32) Color=info.Data32[chr*info.Height*info.Width+YCtr*info.Width+XCtr];
      if (Color) PutPixel(x+XCtr,y+YCtr,Color);
   };
   if (!NoLock) Unlock();
};

void SCREENCLASS::WriteFont(short x,short y,char *txt,FONTCLASS *font){
   FONTINFO info;
   unsigned char len,Counter;
   bool NoLock=Locked;
   if (!font->GetInfo(&info)) return;
   len=strlen(txt);
   if (!NoLock) Lock();
   for (Counter=0;Counter<len;Counter++){
      PutFontChar(x,y,txt[Counter],font);
      x+=info.Width;
   };
   if (!NoLock) Unlock();
};

void SCREENCLASS::WriteFont(short x,short y,unsigned long nr,FONTCLASS *font){
   char *valstr;
   unsigned long tnr=nr;
   unsigned char len=0,Counter;
   do{
      len++;
      tnr/=10;
   }while(tnr>0);
   valstr=(char *)malloc(len+1);
   for (Counter=0;Counter<len;Counter++){
      valstr[len-Counter-1]=(unsigned char)(nr%10+48);
      nr/=10;
   };
   valstr[len]=0;
   WriteFont(x,y,valstr,font);
   free(valstr);
};

void SCREENCLASS::WriteFontAlign(short x,short y,char *txt,FONTCLASS *font){
   FONTINFO info;
   unsigned long Color;
   unsigned short XCtr,YCtr;
   unsigned char len,Counter;
   bool NoLock=Locked;
   if (!font->GetInfo(&info)) return;
   if (info.ColorDepth!=ColorDepth){
      font->Convert(ColorDepth);
      if (!font->GetInfo(&info)) return;
   };
   len=strlen(txt);
   if (!NoLock) Lock();
   for (Counter=0;Counter<len;Counter++){
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.CharWidth[txt[Counter]];XCtr++){
         if (info.ColorDepth==8) Color=info.Data8[txt[Counter]*info.Height*info.Width+YCtr*info.Width+XCtr+info.CharStart[txt[Counter]]];
         if (info.ColorDepth==15 || info.ColorDepth==16) Color=info.Data16[txt[Counter]*info.Height*info.Width+YCtr*info.Width+XCtr+info.CharStart[txt[Counter]]];
         if (info.ColorDepth==32) Color=info.Data32[txt[Counter]*info.Height*info.Width+YCtr*info.Width+XCtr+info.CharStart[txt[Counter]]];
         if (Color) PutPixel(x+XCtr,y+YCtr,Color);
      };
      x+=info.CharWidth[txt[Counter]]+1;
      if (txt[Counter]==32) x+=info.Width/2;
   };
   if (!NoLock) Unlock();
};

void SCREENCLASS::WriteFontAlign(short x,short y,unsigned long nr,FONTCLASS *font){
   char *valstr;
   unsigned long tnr=nr;
   unsigned char len=0,Counter;
   do{
      len++;
      tnr/=10;
   }while(tnr>0);
   valstr=(char *)malloc(len+1);
   for (Counter=0;Counter<len;Counter++){
      valstr[len-Counter-1]=(unsigned char)(nr%10+48);
      nr/=10;
   };
   valstr[len]=0;
   WriteFontAlign(x,y,valstr,font);
   free(valstr);
};

void SCREENCLASS::Blit(short x,short y,unsigned long inr,IMAGECLASS *image){
   IMAGEINFO info;
   unsigned long Color;
   unsigned short XCtr,YCtr;
   bool NoLock=Locked;
   if (!image->GetInfo(&info)) return;
   if (info.ColorDepth!=ColorDepth){
      image->Convert(ColorDepth);
      if (!image->GetInfo(&info)) return;
   };
   if (!NoLock) Lock();
   for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
      if (info.ColorDepth==8) Color=info.Data8[inr*info.Height*info.Width+YCtr*info.Width+XCtr];
      if (info.ColorDepth==15 || info.ColorDepth==16) Color=info.Data16[inr*info.Height*info.Width+YCtr*info.Width+XCtr];
      if (info.ColorDepth==32) Color=info.Data32[inr*info.Height*info.Width+YCtr*info.Width+XCtr];
      PutPixel(x+XCtr,y+YCtr,Color);
   };
   if (!NoLock) Unlock();
};

void SCREENCLASS::BlitColorKey(short x,short y,unsigned long inr,unsigned long colorkey,IMAGECLASS *image){
   IMAGEINFO info;
   unsigned long Color;
   unsigned short XCtr,YCtr;
   bool NoLock=Locked;
   if (!image->GetInfo(&info)) return;
   if (info.ColorDepth!=ColorDepth){
      image->Convert(ColorDepth);
      if (!image->GetInfo(&info)) return;
   };
   if (!NoLock) Lock();
   for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
      if (info.ColorDepth==8) Color=info.Data8[inr*info.Height*info.Width+YCtr*info.Width+XCtr];
      if (info.ColorDepth==15 || info.ColorDepth==16) Color=info.Data16[inr*info.Height*info.Width+YCtr*info.Width+XCtr];
      if (info.ColorDepth==32) Color=info.Data32[inr*info.Height*info.Width+YCtr*info.Width+XCtr];
      if (Color!=colorkey) PutPixel(x+XCtr,y+YCtr,Color);
   };
   if (!NoLock) Unlock();
};

void SCREENCLASS::AlphaBlend(short x,short y,unsigned long inr,unsigned long colorkey,IMAGECLASS *image){
   IMAGEINFO info;
   unsigned long *iptr32,Color;
   unsigned short *iptr16,XCtr,YCtr;
   unsigned char *iptr8,re,gr,bl;
   bool NoLock=Locked;
   if (!image->GetInfo(&info)) return;
   if (info.ColorDepth!=ColorDepth){
      image->Convert(ColorDepth);
      if (!image->GetInfo(&info)) return;
   };
   if (!NoLock) Lock();
   if (info.ColorDepth==8){
      if (PaletteLookup==NULL) return;
      iptr8=&info.Data8[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr8++;
         if (Color!=colorkey){
            re=defaultpal[Color*3]>>2;
            gr=defaultpal[Color*3+1]>>2;
            bl=defaultpal[Color*3+2]>>2;
            Color=GetPixel(x+XCtr,y+YCtr);
            re+=defaultpal[Color*3]>>2;
            gr+=defaultpal[Color*3+1]>>2;
            bl+=defaultpal[Color*3+2]>>2;
            PutPixel(x+XCtr,y+YCtr,PaletteLookup[(re<<10)+(gr<<5)+bl]);
         };
      };
   };
   if (info.ColorDepth==15){
      iptr16=&info.Data16[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr16++;
         if (Color!=colorkey){
            Color=(Color>>1)&15855;
            Color+=(GetPixel(x+XCtr,y+YCtr)>>1)&15855;
            PutPixel(x+XCtr,y+YCtr,Color);
         };
      };
   };
   if (info.ColorDepth==16){
      iptr16=&info.Data16[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr16++;
         if (Color!=colorkey){
            Color=(Color>>1)&31727;
            Color+=(GetPixel(x+XCtr,y+YCtr)>>1)&31727;
            PutPixel(x+XCtr,y+YCtr,Color);
         };
      };
   };
   if (info.ColorDepth==32){
      iptr32=&info.Data32[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr32++;
         if (Color!=colorkey){
            Color=(Color>>1)&8355711;
            Color+=(GetPixel(x+XCtr,y+YCtr)>>1)&8355711;
            PutPixel(x+XCtr,y+YCtr,Color);
         };
      };
   };
   if (!NoLock) Unlock();
};

void SCREENCLASS::AlphaBlendAdd(short x,short y,unsigned long inr,unsigned long colorkey,IMAGECLASS *image){
   IMAGEINFO info;
   unsigned long *iptr32,Color;
   unsigned short *iptr16,XCtr,YCtr;
   unsigned char *iptr8,re,gr,bl;
   bool NoLock=Locked;
   if (!image->GetInfo(&info)) return;
   if (info.ColorDepth!=ColorDepth){
      image->Convert(ColorDepth);
      if (!image->GetInfo(&info)) return;
   };
   if (!NoLock) Lock();
   if (info.ColorDepth==8){
      if (PaletteLookup==NULL) return;
      iptr8=&info.Data8[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr8++;
         if (Color!=colorkey){
            re=defaultpal[Color*3]>>1;
            gr=defaultpal[Color*3+1]>>1;
            bl=defaultpal[Color*3+2]>>1;
            Color=GetPixel(x+XCtr,y+YCtr);
            re+=defaultpal[Color*3]>>1;
            gr+=defaultpal[Color*3+1]>>1;
            bl+=defaultpal[Color*3+2]>>1;
            if (re>31) re=31;
            if (gr>31) gr=31;
            if (bl>31) bl=31;
            PutPixel(x+XCtr,y+YCtr,PaletteLookup[(re<<10)+(gr<<5)+bl]);
         };
      };
   };
   if (info.ColorDepth==15){
      iptr16=&info.Data16[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr16++;
         if (Color!=colorkey){
            re=(unsigned char)(Color>>10);
            gr=(unsigned char)((Color>>5)&31);
            bl=(unsigned char)(Color&31);
            Color=GetPixel(x+XCtr,y+YCtr);
            re+=(unsigned char)(Color>>10);
            gr+=(unsigned char)((Color>>5)&31);
            bl+=(unsigned char)(Color&31);
            if (re>31) re=31;
            if (gr>31) gr=31;
            if (bl>31) bl=31;
            PutPixel(x+XCtr,y+YCtr,(re<<10)+(gr<<5)+bl);
         };
      };
   };
   if (info.ColorDepth==16){
      iptr16=&info.Data16[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr16++;
         if (Color!=colorkey){
            re=(unsigned char)(Color>>11);
            gr=(unsigned char)((Color>>5)&63);
            bl=(unsigned char)(Color&31);
            Color=GetPixel(x+XCtr,y+YCtr);
            re+=(unsigned char)(Color>>11);
            gr+=(unsigned char)((Color>>5)&63);
            bl+=(unsigned char)(Color&31);
            if (re>31) re=31;
            if (gr>63) gr=63;
            if (bl>31) bl=31;
            PutPixel(x+XCtr,y+YCtr,(re<<11)+(gr<<5)+bl);
         };
      };
   };
   if (info.ColorDepth==32){
      iptr32=&info.Data32[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr32++;
         if (Color!=colorkey){
            re=(unsigned char)(Color>>16);
            gr=(unsigned char)((Color>>8)&255);
            bl=(unsigned char)(Color&255);
            Color=GetPixel(x+XCtr,y+YCtr);
            if (re+(Color>>16)<256) re+=(unsigned char)(Color>>16); else re=255;
            if (gr+((Color>>8)&255)<256) gr+=(unsigned char)((Color>>8)&255); else gr=255;
            if (bl+(Color&255)<256) bl+=(unsigned char)(Color&255); else bl=255;
            PutPixel(x+XCtr,y+YCtr,(re<<16)+(gr<<8)+bl);
         };
      };
   };
   if (!NoLock) Unlock();
};

void SCREENCLASS::AlphaBlendSemiAdd(short x,short y,unsigned long inr,unsigned long colorkey,IMAGECLASS *image){
   IMAGEINFO info;
   unsigned long *iptr32,Color;
   unsigned short *iptr16,XCtr,YCtr;
   unsigned char *iptr8,re,gr,bl;
   bool NoLock=Locked;
   if (!image->GetInfo(&info)) return;
   if (info.ColorDepth!=ColorDepth){
      image->Convert(ColorDepth);
      if (!image->GetInfo(&info)) return;
   };
   if (!NoLock) Lock();
   if (info.ColorDepth==8){
      if (PaletteLookup==NULL) return;
      iptr8=&info.Data8[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr8++;
         if (Color!=colorkey){
            re=defaultpal[Color*3]>>1;
            gr=defaultpal[Color*3+1]>>1;
            bl=defaultpal[Color*3+2]>>1;
            Color=GetPixel(x+XCtr,y+YCtr);
            re+=defaultpal[Color*3]>>2;
            gr+=defaultpal[Color*3+1]>>2;
            bl+=defaultpal[Color*3+2]>>2;
            if (re>31) re=31;
            if (gr>31) gr=31;
            if (bl>31) bl=31;
            PutPixel(x+XCtr,y+YCtr,PaletteLookup[(re<<10)+(gr<<5)+bl]);
         };
      };
   };
   if (info.ColorDepth==15){
      iptr16=&info.Data16[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr16++;
         if (Color!=colorkey){
            re=(unsigned char)(Color>>10);
            gr=(unsigned char)((Color>>5)&31);
            bl=(unsigned char)(Color&31);
            Color=GetPixel(x+XCtr,y+YCtr);
            re+=(unsigned char)(Color>>11);
            gr+=(unsigned char)((Color>>6)&15);
            bl+=(unsigned char)((Color>>1)&15);
            if (re>31) re=31;
            if (gr>31) gr=31;
            if (bl>31) bl=31;
            PutPixel(x+XCtr,y+YCtr,(re<<10)+(gr<<5)+bl);
         };
      };
   };
   if (info.ColorDepth==16){
      iptr16=&info.Data16[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr16++;
         if (Color!=colorkey){
            re=(unsigned char)(Color>>11);
            gr=(unsigned char)((Color>>5)&63);
            bl=(unsigned char)(Color&31);
            Color=GetPixel(x+XCtr,y+YCtr);
            re+=(unsigned char)(Color>>12);
            gr+=(unsigned char)((Color>>6)&31);
            bl+=(unsigned char)((Color>>1)&15);
            if (re>31) re=31;
            if (gr>63) gr=63;
            if (bl>31) bl=31;
            PutPixel(x+XCtr,y+YCtr,(re<<11)+(gr<<5)+bl);
         };
      };
   };
   if (info.ColorDepth==32){
      iptr32=&info.Data32[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr32++;
         if (Color!=colorkey){
            re=(unsigned char)(Color>>16);
            gr=(unsigned char)((Color>>8)&255);
            bl=(unsigned char)(Color&255);
            Color=GetPixel(x+XCtr,y+YCtr);
            if (re+(Color>>17)<256) re+=(unsigned char)(Color>>17); else re=255;
            if (gr+((Color>>9)&127)<256) gr+=(unsigned char)((Color>>9)&127); else gr=255;
            if (bl+((Color>>1)&127)<256) bl+=(unsigned char)((Color>>1)&127); else bl=255;
            PutPixel(x+XCtr,y+YCtr,(re<<16)+(gr<<8)+bl);
         };
      };
   };
   if (!NoLock) Unlock();
};

void SCREENCLASS::CreatePaletteLookupTable(void){
   PALETTE pal;
   unsigned short Color;
   unsigned char *palptr,PalIndex,re,gr,bl,ColDiff,NewColDiff;
   if (PaletteLookup==NULL) PaletteLookup=(unsigned char *)malloc(33536);
   if (!memcmp(defaultpal,&PaletteLookup[32768],768)) return;
   memcpy(&PaletteLookup[32768],defaultpal,768);
   memcpy(pal,defaultpal,768);
   for (Color=0;Color<768;Color++) pal[Color]>>=1;
   for (Color=0;Color<32768;Color++){
      PaletteLookup[Color]=0;
      re=Color&31;
      gr=(Color>>5)&31;
      bl=Color>>10;
      palptr=pal;
      ColDiff=abs(bl-*palptr++);
      ColDiff+=abs(gr-*palptr++);
      ColDiff+=abs(re-*palptr++);
      for (PalIndex=1;PalIndex>0;PalIndex++){
         NewColDiff=abs(bl-*palptr++);
         NewColDiff+=abs(gr-*palptr++);
         NewColDiff+=abs(re-*palptr++);
         if (NewColDiff<ColDiff){
            PaletteLookup[Color]=PalIndex;
            ColDiff=NewColDiff;
         };
      };
   };
};

void SCREENCLASS::RecreateLookupTable(bool param){
   UpdateLookupTable=param;
};

bool SCREENCLASS::AttachBuffer(BUFFERCLASS *&buffer){
   buffer=new BUFFERCLASS(this,Width,Height,ColorDepth);
   if (buffer==NULL) return(false);
   return(true);
};

void SCREENCLASS::Flip(BUFFERCLASS *buffer){
   BUFFERINFO info;
   unsigned long Counter;
   if (!buffer->GetInfo(&info)) return;
   if (ColorDepth!=info.ColorDepth || Width!=info.Width || Height!=info.Height){
      buffer->Convert(Width,Height,ColorDepth);
      if (!buffer->GetInfo(&info)) return;
   };
   #ifdef DJGPP
   unsigned long ScreenSize;
   if (ColorDepth==15) ScreenSize=Width*Height*2;
   else ScreenSize=Width*Height*ColorDepth/8;
   if (LFBEnabled || CurrentVideoMode<256){
      if (ColorDepth==8) memcpy(Surface8,info.Data8,ScreenSize);
      if (ColorDepth==15 || ColorDepth==16) memcpy(Surface16,info.Data16,ScreenSize);
      if (ColorDepth==32) memcpy(Surface32,info.Data32,ScreenSize);
   } else {
      for (Counter=0;Counter<ScreenSize/65536;Counter++){
         SwitchBank(Counter*(65536/BankSize));
         if (ColorDepth==8) memcpy(Surface8,&info.Data8[Counter*65536],65536);
         if (ColorDepth==15 || ColorDepth==16) memcpy(Surface16,&info.Data16[Counter*32768],65536);
         if (ColorDepth==32) memcpy(Surface32,&info.Data32[Counter*16384],65536);
      };
      if (ScreenSize%65536){
         SwitchBank(ScreenSize/65536*(65536/BankSize));
         if (ColorDepth==8) memcpy(Surface8,&info.Data8[ScreenSize/65536*65536],ScreenSize%65536);
         if (ColorDepth==15 || ColorDepth==16) memcpy(Surface16,&info.Data16[ScreenSize/65536*32768],ScreenSize%65536);
         if (ColorDepth==32) memcpy(Surface32,&info.Data32[ScreenSize/65536*16384],ScreenSize%65536);
      };
   };
   #endif
   #ifdef WIN32
   unsigned short RealWidth=Width;
   bool NoLock=Locked;
   if (ColorDepth==15 || ColorDepth==16) RealWidth*=2;
   if (ColorDepth==32) RealWidth*=4;
   if (!NoLock) Lock();
   for (Counter=0;Counter<Height;Counter++){
	  if (ColorDepth==8) memcpy(&Surface8[OffsTable[Counter]],&info.Data8[Counter*Width],RealWidth);
      if (ColorDepth==15 || ColorDepth==16) memcpy(&Surface16[OffsTable[Counter]],&info.Data16[Counter*Width],RealWidth);
      if (ColorDepth==32) memcpy(&Surface32[OffsTable[Counter]],&info.Data32[Counter*Width],RealWidth);
   };
   if (!NoLock) Unlock();
   #endif
};

void SCREENCLASS::Lock(void){
   #ifdef WIN32
   if (Locked) return;
   dsurface->Lock(NULL,&ddesc,DDLOCK_WAIT,NULL);
   if (ColorDepth==8) Surface8=(unsigned char *)ddesc.lpSurface;
   if (ColorDepth==15 || ColorDepth==16) Surface16=(unsigned short *)ddesc.lpSurface;
   if (ColorDepth==32) Surface32=(unsigned long *)ddesc.lpSurface;
   Locked=true;
   #endif
};

void SCREENCLASS::Unlock(void){
   #ifdef WIN32
   if (!Locked) return;
   dsurface->Unlock(NULL);
   Locked=false;
   #endif
};

void SCREENCLASS::Release(void){
   if (Width!=0) CloseGfxMode();
   if (PaletteLookup!=NULL) free(PaletteLookup);
   if (this!=NULL) delete(this);
};

//////////////////////////////////////////////////////////////////////
// BUFFERCLASS member functions (public)
//////////////////////////////////////////////////////////////////////

BUFFERCLASS::BUFFERCLASS(SCREENCLASS *screen,unsigned short xres,unsigned short yres,unsigned char bpp){
   unsigned short Counter;
   if (bpp==8) BufMem8=(unsigned char *)malloc(xres*yres);
   if (bpp==15 || bpp==16) BufMem16=(unsigned short *)malloc(xres*yres*2);
   if (bpp==32) BufMem32=(unsigned long *)malloc(xres*yres*4);
   screenptr=screen;
   ColorDepth=bpp;
   XRes=xres;
   YRes=yres;
   for (Counter=0;Counter<yres;Counter++) OffsTable[Counter]=Counter*XRes;
   Clear();
};

bool BUFFERCLASS::GetInfo(BUFFERINFO *info){
   info->Data32=BufMem32;
   info->Data16=BufMem16;
   info->Data8=BufMem8;
   info->Height=YRes;
   info->Width=XRes;
   info->ColorDepth=ColorDepth;
   return(true);
};

void BUFFERCLASS::Clear(void){
   if (ColorDepth==8) memset8(BufMem8,0,YRes*XRes);
   if (ColorDepth==15 || ColorDepth==16) memset8(BufMem16,0,YRes*XRes*2);
   if (ColorDepth==32) memset8(BufMem32,0,YRes*XRes*4);
};

void BUFFERCLASS::PutPixel(unsigned short x,unsigned short y,unsigned long Color){
   if (x>=XRes) return;
   if (y>=YRes) return;
   if (ColorDepth==8) BufMem8[OffsTable[y]+x]=(unsigned char)(Color%256);
   if (ColorDepth==15 || ColorDepth==16) BufMem16[OffsTable[y]+x]=(unsigned short)(Color%65536);
   if (ColorDepth==32) BufMem32[OffsTable[y]+x]=Color;
};

unsigned long BUFFERCLASS::GetPixel(unsigned short x,unsigned short y){
   if (x>=XRes) return(0);
   if (y>=YRes) return(0);
   if (ColorDepth==8) return(BufMem8[OffsTable[y]+x]);
   if (ColorDepth==15 || ColorDepth==16) return(BufMem16[OffsTable[y]+x]);
   if (ColorDepth==32) return(BufMem32[OffsTable[y]+x]);
   return(0);
};

void BUFFERCLASS::HLine(short x,short x2,short y,unsigned long Color){
   short tempx;
   if (y<0) return;
   if (y>=YRes) return;
   if (x>x2){
      tempx=x2;
      x2=x;
      x=tempx;
   };
   if (x2<0) return;
   if (x>=XRes) return;
   if (x<0) x=0;
   if (x2>=XRes) x2=XRes-1;
   if (ColorDepth==8) memset8(&BufMem8[OffsTable[y]+x],(unsigned char)Color,x2-x+1);
   if (ColorDepth==15 || ColorDepth==16) memset16(&BufMem16[OffsTable[y]+x],(unsigned short)Color,x2-x+1);
   if (ColorDepth==32) memset32(&BufMem32[OffsTable[y]+x],Color,x2-x+1);
};

void BUFFERCLASS::VLine(short y,short y2,short x,unsigned long Color){
   short Counter,tempy;
   if (x<0) return;
   if (x>=XRes) return;
   if (y>y2){
      tempy=y2;
      y2=y;
      y=tempy;
   };
   if (y2<0) return;
   if (y>=YRes) return;
   if (y<0) y=0;
   if (y2>=YRes) y2=YRes-1;
   for (Counter=y;Counter<=y2;Counter++){
      if (ColorDepth==8) BufMem8[OffsTable[Counter]+x]=(unsigned char)Color;
      if (ColorDepth==15 || ColorDepth==16) BufMem16[OffsTable[Counter]+x]=(unsigned short)Color;
      if (ColorDepth==32) BufMem32[OffsTable[Counter]+x]=Color;
   };
};

void BUFFERCLASS::Line(short x,short y,short x2,short y2,unsigned long Color){
   short XLen,YLen,XPlus=0,YPlus=0,Step=0;
   unsigned short Counter;
   char YUp,XUp;
   if (y==y2){
      HLine(x,x2,y,Color);
      return;
   };
   if (x==x2){
      VLine(y,y2,x,Color);
      return;
   };
   XLen=x2-x;
   YLen=y2-y;
   if (XLen>=0) XUp=1; else {
      XUp=-1;
      XLen=-XLen;
   };
   if (YLen>=0) YUp=1; else {
      YUp=-1;
      YLen=-YLen;
   };
   if (XLen>YLen){
      for (Counter=0;Counter<=XLen;Counter++){
         PutPixel(x+XPlus,y+YPlus,Color);
         Step+=YLen;
         if (Step>XLen){
            Step-=XLen;
            YPlus+=YUp;
         };
         XPlus+=XUp;
      };
   } else for (Counter=0;Counter<=YLen;Counter++){
      PutPixel(x+XPlus,y+YPlus,Color);
      Step+=XLen;
      if (Step>0){
         Step-=YLen;
         XPlus+=XUp;
      };
      YPlus+=YUp;
   };
};

void BUFFERCLASS::Rectangle(short x,short y,short x2,short y2,unsigned long Color){
   HLine(x,x2,y,Color);
   HLine(x,x2,y2,Color);
   VLine(y,y2,x,Color);
   VLine(y,y2,x2,Color);
};

void BUFFERCLASS::Bar(short x,short y,short x2,short y2,unsigned long Color){
   short YCtr;
   for (YCtr=y;YCtr<=y2;YCtr++) HLine(x,x2,YCtr,Color);
};

void BUFFERCLASS::PutFontChar(short x,short y,unsigned char chr,FONTCLASS *font){
   FONTINFO info;
   unsigned long Color;
   unsigned short XCtr,YCtr;
   if (!font->GetInfo(&info)) return;
   if (info.ColorDepth!=ColorDepth){
      font->Convert(ColorDepth);
      if (!font->GetInfo(&info)) return;
   };
   for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
      if (info.ColorDepth==8) Color=info.Data8[chr*info.Height*info.Width+YCtr*info.Width+XCtr];
      if (info.ColorDepth==15 || info.ColorDepth==16) Color=info.Data16[chr*info.Height*info.Width+YCtr*info.Width+XCtr];
      if (info.ColorDepth==32) Color=info.Data32[chr*info.Height*info.Width+YCtr*info.Width+XCtr];
      if (Color) PutPixel(x+XCtr,y+YCtr,Color);
   };
};

void BUFFERCLASS::WriteFont(short x,short y,char *txt,FONTCLASS *font){
   FONTINFO info;
   unsigned char len,Counter;
   if (!font->GetInfo(&info)) return;
   len=strlen(txt);
   for (Counter=0;Counter<len;Counter++){
      PutFontChar(x,y,txt[Counter],font);
      x+=info.Width;
   };
};

void BUFFERCLASS::WriteFont(short x,short y,unsigned long nr,FONTCLASS *font){
   char *valstr;
   unsigned long tnr=nr;
   unsigned char len=0,Counter;
   do{
      len++;
      tnr/=10;
   }while(tnr>0);
   valstr=(char *)malloc(len+1);
   for (Counter=0;Counter<len;Counter++){
      valstr[len-Counter-1]=(unsigned char)(nr%10+48);
      nr/=10;
   };
   valstr[len]=0;
   WriteFont(x,y,valstr,font);
   free(valstr);
};

void BUFFERCLASS::WriteFontAlign(short x,short y,char *txt,FONTCLASS *font){
   FONTINFO info;
   unsigned long Color;
   unsigned short XCtr,YCtr;
   unsigned char len,Counter;
   if (!font->GetInfo(&info)) return;
   if (info.ColorDepth!=ColorDepth){
      font->Convert(ColorDepth);
      if (!font->GetInfo(&info)) return;
   };
   len=strlen(txt);
   for (Counter=0;Counter<len;Counter++){
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.CharWidth[txt[Counter]];XCtr++){
         if (info.ColorDepth==8) Color=info.Data8[txt[Counter]*info.Height*info.Width+YCtr*info.Width+XCtr+info.CharStart[txt[Counter]]];
         if (info.ColorDepth==15 || info.ColorDepth==16) Color=info.Data16[txt[Counter]*info.Height*info.Width+YCtr*info.Width+XCtr+info.CharStart[txt[Counter]]];
         if (info.ColorDepth==32) Color=info.Data32[txt[Counter]*info.Height*info.Width+YCtr*info.Width+XCtr+info.CharStart[txt[Counter]]];
         if (Color) PutPixel(x+XCtr,y+YCtr,Color);
      };
      x+=info.CharWidth[txt[Counter]]+1;
      if (txt[Counter]==32) x+=info.Width/2;
   };
};

void BUFFERCLASS::WriteFontAlign(short x,short y,unsigned long nr,FONTCLASS *font){
   char *valstr;
   unsigned long tnr=nr;
   unsigned char len=0,Counter;
   do{
      len++;
      tnr/=10;
   }while(tnr>0);
   valstr=(char *)malloc(len+1);
   for (Counter=0;Counter<len;Counter++){
      valstr[len-Counter-1]=(unsigned char)(nr%10+48);
      nr/=10;
   };
   valstr[len]=0;
   WriteFontAlign(x,y,valstr,font);
   free(valstr);
};

void BUFFERCLASS::Blit(short x,short y,unsigned long inr,IMAGECLASS *image){
   IMAGEINFO info;
   unsigned long Color;
   unsigned short XCtr,YCtr;
   if (!image->GetInfo(&info)) return;
   if (info.ColorDepth!=ColorDepth){
      image->Convert(ColorDepth);
      if (!image->GetInfo(&info)) return;
   };
   for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
      if (info.ColorDepth==8) Color=info.Data8[inr*info.Height*info.Width+YCtr*info.Width+XCtr];
      if (info.ColorDepth==15 || info.ColorDepth==16) Color=info.Data16[inr*info.Height*info.Width+YCtr*info.Width+XCtr];
      if (info.ColorDepth==32) Color=info.Data32[inr*info.Height*info.Width+YCtr*info.Width+XCtr];
      PutPixel(x+XCtr,y+YCtr,Color);
   };
};

void BUFFERCLASS::BlitColorKey(short x,short y,unsigned long inr,unsigned long colorkey,IMAGECLASS *image){
   IMAGEINFO info;
   unsigned long Color;
   unsigned short XCtr,YCtr;
   if (!image->GetInfo(&info)) return;
   if (info.ColorDepth!=ColorDepth){
      image->Convert(ColorDepth);
      if (!image->GetInfo(&info)) return;
   };
   for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
      if (info.ColorDepth==8) Color=info.Data8[inr*info.Height*info.Width+YCtr*info.Width+XCtr];
      if (info.ColorDepth==15 || info.ColorDepth==16) Color=info.Data16[inr*info.Height*info.Width+YCtr*info.Width+XCtr];
      if (info.ColorDepth==32) Color=info.Data32[inr*info.Height*info.Width+YCtr*info.Width+XCtr];
      if (Color!=colorkey) PutPixel(x+XCtr,y+YCtr,Color);
   };
};

void BUFFERCLASS::AlphaBlend(short x,short y,unsigned long inr,unsigned long colorkey,IMAGECLASS *image){
   IMAGEINFO info;
   SCREENINFO scrinfo;
   PALETTE defaultpal;
   unsigned long *iptr32,Color;
   unsigned short *iptr16,XCtr,YCtr;
   unsigned char *iptr8,re,gr,bl;
   if (!image->GetInfo(&info)) return;
   if (info.ColorDepth!=ColorDepth){
      image->Convert(ColorDepth);
      if (!image->GetInfo(&info)) return;
   };
   if (!screenptr->GetInfo(&scrinfo)) return;
   if (ColorDepth==8) screenptr->GetCurrentPalette(defaultpal);
   if (info.ColorDepth==8){
      if (scrinfo.PaletteLookup==NULL) return;
      iptr8=&info.Data8[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr8++;
         if (Color!=colorkey){
            re=defaultpal[Color*3]>>2;
            gr=defaultpal[Color*3+1]>>2;
            bl=defaultpal[Color*3+2]>>2;
            Color=GetPixel(x+XCtr,y+YCtr);
            re+=defaultpal[Color*3]>>2;
            gr+=defaultpal[Color*3+1]>>2;
            bl+=defaultpal[Color*3+2]>>2;
            PutPixel(x+XCtr,y+YCtr,scrinfo.PaletteLookup[(re<<10)+(gr<<5)+bl]);
         };
      };
   };
   if (info.ColorDepth==15){
      iptr16=&info.Data16[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr16++;
         if (Color!=colorkey){
            Color=(Color>>1)&15855;
            Color+=(GetPixel(x+XCtr,y+YCtr)>>1)&15855;
            PutPixel(x+XCtr,y+YCtr,Color);
         };
      };
   };
   if (info.ColorDepth==16){
      iptr16=&info.Data16[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr16++;
         if (Color!=colorkey){
            Color=(Color>>1)&31727;
            Color+=(GetPixel(x+XCtr,y+YCtr)>>1)&31727;
            PutPixel(x+XCtr,y+YCtr,Color);
         };
      };
   };
   if (info.ColorDepth==32){
      iptr32=&info.Data32[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr32++;
         if (Color!=colorkey){
            Color=(Color>>1)&8355711;
            Color+=(GetPixel(x+XCtr,y+YCtr)>>1)&8355711;
            PutPixel(x+XCtr,y+YCtr,Color);
         };
      };
   };
};

void BUFFERCLASS::AlphaBlendAdd(short x,short y,unsigned long inr,unsigned long colorkey,IMAGECLASS *image){
   IMAGEINFO info;
   SCREENINFO scrinfo;
   PALETTE defaultpal;
   unsigned long *iptr32,Color;
   unsigned short *iptr16,XCtr,YCtr;
   unsigned char *iptr8,re,gr,bl;
   if (!image->GetInfo(&info)) return;
   if (info.ColorDepth!=ColorDepth){
      image->Convert(ColorDepth);
      if (!image->GetInfo(&info)) return;
   };
   if (!screenptr->GetInfo(&scrinfo)) return;
   if (ColorDepth==8) screenptr->GetCurrentPalette(defaultpal);
   if (info.ColorDepth==8){
      if (scrinfo.PaletteLookup==NULL) return;
      iptr8=&info.Data8[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr8++;
         if (Color!=colorkey){
            re=defaultpal[Color*3]>>1;
            gr=defaultpal[Color*3+1]>>1;
            bl=defaultpal[Color*3+2]>>1;
            Color=GetPixel(x+XCtr,y+YCtr);
            re+=defaultpal[Color*3]>>1;
            gr+=defaultpal[Color*3+1]>>1;
            bl+=defaultpal[Color*3+2]>>1;
            if (re>31) re=31;
            if (gr>31) gr=31;
            if (bl>31) bl=31;
            PutPixel(x+XCtr,y+YCtr,scrinfo.PaletteLookup[(re<<10)+(gr<<5)+bl]);
         };
      };
   };
   if (info.ColorDepth==15){
      iptr16=&info.Data16[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr16++;
         if (Color!=colorkey){
            re=(unsigned char)(Color>>10);
            gr=(unsigned char)((Color>>5)&31);
            bl=(unsigned char)(Color&31);
            Color=GetPixel(x+XCtr,y+YCtr);
            re+=(unsigned char)(Color>>10);
            gr+=(unsigned char)((Color>>5)&31);
            bl+=(unsigned char)(Color&31);
            if (re>31) re=31;
            if (gr>31) gr=31;
            if (bl>31) bl=31;
            PutPixel(x+XCtr,y+YCtr,(re<<10)+(gr<<5)+bl);
         };
      };
   };
   if (info.ColorDepth==16){
      iptr16=&info.Data16[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr16++;
         if (Color!=colorkey){
            re=(unsigned char)(Color>>11);
            gr=(unsigned char)((Color>>5)&63);
            bl=(unsigned char)(Color&31);
            Color=GetPixel(x+XCtr,y+YCtr);
            re+=(unsigned char)(Color>>11);
            gr+=(unsigned char)((Color>>5)&63);
            bl+=(unsigned char)(Color&31);
            if (re>31) re=31;
            if (gr>63) gr=63;
            if (bl>31) bl=31;
            PutPixel(x+XCtr,y+YCtr,(re<<11)+(gr<<5)+bl);
         };
      };
   };
   if (info.ColorDepth==32){
      iptr32=&info.Data32[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr32++;
         if (Color!=colorkey){
            re=(unsigned char)(Color>>16);
            gr=(unsigned char)((Color>>8)&255);
            bl=(unsigned char)(Color&255);
            Color=GetPixel(x+XCtr,y+YCtr);
            if (re+(Color>>16)<256) re+=(unsigned char)(Color>>16); else re=255;
            if (gr+((Color>>8)&255)<256) gr+=(unsigned char)((Color>>8)&255); else gr=255;
            if (bl+(Color&255)<256) bl+=(unsigned char)(Color&255); else bl=255;
            PutPixel(x+XCtr,y+YCtr,(re<<16)+(gr<<8)+bl);
         };
      };
   };
};

void BUFFERCLASS::AlphaBlendSemiAdd(short x,short y,unsigned long inr,unsigned long colorkey,IMAGECLASS *image){
   IMAGEINFO info;
   SCREENINFO scrinfo;
   PALETTE defaultpal;
   unsigned long *iptr32,Color;
   unsigned short *iptr16,XCtr,YCtr;
   unsigned char *iptr8,re,gr,bl;
   if (!image->GetInfo(&info)) return;
   if (info.ColorDepth!=ColorDepth){
      image->Convert(ColorDepth);
      if (!image->GetInfo(&info)) return;
   };
   if (!screenptr->GetInfo(&scrinfo)) return;
   if (ColorDepth==8) screenptr->GetCurrentPalette(defaultpal);
   if (info.ColorDepth==8){
      if (scrinfo.PaletteLookup==NULL) return;
      iptr8=&info.Data8[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr8++;
         if (Color!=colorkey){
            re=defaultpal[Color*3]>>1;
            gr=defaultpal[Color*3+1]>>1;
            bl=defaultpal[Color*3+2]>>1;
            Color=GetPixel(x+XCtr,y+YCtr);
            re+=defaultpal[Color*3]>>2;
            gr+=defaultpal[Color*3+1]>>2;
            bl+=defaultpal[Color*3+2]>>2;
            if (re>31) re=31;
            if (gr>31) gr=31;
            if (bl>31) bl=31;
            PutPixel(x+XCtr,y+YCtr,scrinfo.PaletteLookup[(re<<10)+(gr<<5)+bl]);
         };
      };
   };
   if (info.ColorDepth==15){
      iptr16=&info.Data16[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr16++;
         if (Color!=colorkey){
            re=(unsigned char)(Color>>10);
            gr=(unsigned char)((Color>>5)&31);
            bl=(unsigned char)(Color&31);
            Color=GetPixel(x+XCtr,y+YCtr);
            re+=(unsigned char)(Color>>11);
            gr+=(unsigned char)((Color>>6)&15);
            bl+=(unsigned char)((Color>>1)&15);
            if (re>31) re=31;
            if (gr>31) gr=31;
            if (bl>31) bl=31;
            PutPixel(x+XCtr,y+YCtr,(re<<10)+(gr<<5)+bl);
         };
      };
   };
   if (info.ColorDepth==16){
      iptr16=&info.Data16[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr16++;
         if (Color!=colorkey){
            re=(unsigned char)(Color>>11);
            gr=(unsigned char)((Color>>5)&63);
            bl=(unsigned char)(Color&31);
            Color=GetPixel(x+XCtr,y+YCtr);
            re+=(unsigned char)(Color>>12);
            gr+=(unsigned char)((Color>>6)&31);
            bl+=(unsigned char)((Color>>1)&15);
            if (re>31) re=31;
            if (gr>63) gr=63;
            if (bl>31) bl=31;
            PutPixel(x+XCtr,y+YCtr,(re<<11)+(gr<<5)+bl);
         };
      };
   };
   if (info.ColorDepth==32){
      iptr32=&info.Data32[inr*info.Width*info.Height];
      for (YCtr=0;YCtr<info.Height;YCtr++) for (XCtr=0;XCtr<info.Width;XCtr++){
         Color=*iptr32++;
         if (Color!=colorkey){
            re=(unsigned char)(Color>>16);
            gr=(unsigned char)((Color>>8)&255);
            bl=(unsigned char)(Color&255);
            Color=GetPixel(x+XCtr,y+YCtr);
            if (re+(Color>>17)<256) re+=(unsigned char)(Color>>17); else re=255;
            if (gr+((Color>>9)&127)<256) gr+=(unsigned char)((Color>>9)&127); else gr=255;
            if (bl+((Color>>1)&127)<256) bl+=(unsigned char)((Color>>1)&127); else bl=255;
            PutPixel(x+XCtr,y+YCtr,(re<<16)+(gr<<8)+bl);
         };
      };
   };
};

void BUFFERCLASS::Flip(SCREENCLASS *screen){
   SCREENINFO info;
   bool NoLock;
   if (!screen->GetInfo(&info)) return;
   NoLock=info.Locked;
   if (!NoLock) screen->Lock();
   if (!screen->GetInfo(&info)) return;
   if (XRes!=info.Width || YRes!=info.Height || ColorDepth!=info.ColorDepth) Convert(info.Width,info.Height,ColorDepth);
   #ifdef DJGPP
   unsigned short x,y;
   if (info.LFBFound){
      if (ColorDepth==8) memcpy(BufMem8,info.Data8,XRes*YRes);
      if (ColorDepth==15 || ColorDepth==16) memcpy(BufMem16,info.Data16,XRes*YRes);
      if (ColorDepth==32) memcpy(BufMem32,info.Data32,XRes*YRes);
   } else for (y=0;y<YRes;y++) for (x=0;x<XRes;x++) PutPixel(x,y,screen->GetPixel(x,y));
   #endif
   #ifdef WIN32
   unsigned short y,realwidth;
   if (ColorDepth==15) realwidth=info.Width*2;
   else realwidth=info.Width*ColorDepth/8;
   for (y=0;y<YRes;y++){
      if (ColorDepth==8) memcpy(&BufMem8[OffsTable[y]],&info.Data8[info.LineWidth*y],realwidth);
      if (ColorDepth==15 || ColorDepth==16) memcpy(&BufMem16[OffsTable[y]],&info.Data16[info.LineWidth*y],realwidth);
      if (ColorDepth==32) memcpy(&BufMem32[OffsTable[y]],&info.Data32[info.LineWidth*y],realwidth);
   };
   #endif
   if (!NoLock) screen->Unlock();
};

void BUFFERCLASS::Flip(BUFFERCLASS *buffer){
   BUFFERINFO info;
   if (!buffer->GetInfo(&info)) return;
   if (XRes!=info.Width || YRes!=info.Height || ColorDepth!=info.ColorDepth){
      buffer->Convert(XRes,YRes,ColorDepth);
      if (!buffer->GetInfo(&info)) return;
   };
   if (ColorDepth==8) memcpy(BufMem8,info.Data8,XRes*YRes);
   if (ColorDepth==15 || ColorDepth==16) memcpy(BufMem16,info.Data16,XRes*YRes*2);
   if (ColorDepth==32) memcpy(BufMem32,info.Data32,XRes*YRes*4);
};

void BUFFERCLASS::Convert(unsigned short toxres,unsigned short toyres,unsigned char tobpp){
   PALETTE defaultpal;
   SCREENINFO info;
   unsigned long *temp32,Color;
   unsigned short *temp16,xctr,yctr;
   unsigned char *temp8,re,gr,bl;
   if (!screenptr->GetInfo(&info)) return;
   if (tobpp==8) temp8=(unsigned char *)malloc(toxres*toyres);
   if (tobpp==15 || tobpp==16) temp16=(unsigned short *)malloc(toxres*toyres*2);
   if (tobpp==32) temp32=(unsigned long *)malloc(toxres*toyres*4);
   if (tobpp==8 && info.RecreateLookupTable) screenptr->CreatePaletteLookupTable();
   if (tobpp==8 || ColorDepth==8) screenptr->GetCurrentPalette(defaultpal);
   for (yctr=0;yctr<toyres;yctr++) for (xctr=0;xctr<toxres;xctr++){
      Color=GetPixel(xctr,yctr);
      if (ColorDepth==8){
         re=defaultpal[Color*3]>>1;
         gr=defaultpal[Color*3+1]>>1;
         bl=defaultpal[Color*3+2]>>1;
      };
      if (ColorDepth==15){
         bl=(unsigned char)(Color&31);
         gr=(unsigned char)((Color&992)>>5);
         re=(unsigned char)((Color&31744)>>10);
      };
      if (ColorDepth==16){
         bl=(unsigned char)(Color&31);
         gr=(unsigned char)((Color&1984)>>6);
         re=(unsigned char)((Color&63488)>>11);
      };
      if (ColorDepth==32){
         bl=(unsigned char)((Color>>3)&31);
         gr=(unsigned char)((Color>>11)&31);
         re=(unsigned char)(Color>>19);
      };
      if (tobpp==8) Color=info.PaletteLookup[(re<<10)+(gr<<5)+bl];
      if (tobpp==15) Color=(re<<10)+(gr<<5)+bl;
      if (tobpp==16) Color=(re<<11)+(gr<<6)+bl;
      if (tobpp==32) Color=(re<<19)+(gr<<11)+(bl<<3);
      if (tobpp==8) temp8[yctr*toxres+xctr]=(unsigned char)Color;
      if (tobpp==15 || tobpp==16) temp16[yctr*toxres+xctr]=(unsigned short)Color;
      if (tobpp==32) temp32[yctr*toxres+xctr]=Color;
   };
   if (ColorDepth==8) free(BufMem8);
   if (ColorDepth==15 || ColorDepth==16) free(BufMem16);
   if (ColorDepth==32) free(BufMem32);
   ColorDepth=tobpp;
   if (ColorDepth==8) BufMem8=temp8;
   if (ColorDepth==15 || ColorDepth==16) BufMem16=temp16;
   if (ColorDepth==32) BufMem32=temp32;
   XRes=toxres;
   YRes=toyres;
   for (yctr=0;yctr<YRes;yctr++) OffsTable[yctr]=yctr*XRes;
};

void BUFFERCLASS::Release(void){
   if (ColorDepth==8 && BufMem8!=NULL) free(BufMem8);
   if ((ColorDepth==15 || ColorDepth==16) && BufMem16!=NULL) free(BufMem16);
   if (ColorDepth==32 && BufMem32!=NULL) free(BufMem32);
   if (this!=NULL) delete(this);
};

//////////////////////////////////////////////////////////////////////
// FONTCLASS member function (private)
//////////////////////////////////////////////////////////////////////

void FONTCLASS::FreeFont(void){
   if (FontData8!=NULL && ColorDepth==8) free(FontData8);
   if (FontData16!=NULL && (ColorDepth==15 || ColorDepth==16)) free(FontData16);
   if (FontData32!=NULL && ColorDepth==32) free(FontData32);
   free(fstart);
   free(fwidth);
};

void FONTCLASS::CalculateWidth(void){
   unsigned short cctr,hctr;
   short pctr;
   fstart=(unsigned short *)malloc(512);
   fwidth=(unsigned short *)malloc(512);
   for (cctr=0;cctr<256;cctr++){
      fstart[cctr]=Width-1;
      for (hctr=0;hctr<Height;hctr++) for (pctr=0;pctr<fstart[cctr];pctr++){
         if (ColorDepth==8) if (FontData8[cctr*Height*Width+hctr*Width+pctr]) fstart[cctr]=pctr;
         if (ColorDepth==15 || ColorDepth==16) if (FontData16[cctr*Height*Width+hctr*Width+pctr]) fstart[cctr]=pctr;
         if (ColorDepth==32) if (FontData32[cctr*Height*Width+hctr*Width+pctr]) fstart[cctr]=pctr;
      };
      fwidth[cctr]=0;
      for (hctr=0;hctr<Height;hctr++) for (pctr=Width-1;pctr>=fstart[cctr]+fwidth[cctr];pctr--){
         if (ColorDepth==8) if (FontData8[cctr*Height*Width+hctr*Width+pctr]) fwidth[cctr]=pctr-fstart[cctr]+1;
         if (ColorDepth==15 || ColorDepth==16) if (FontData16[cctr*Height*Width+hctr*Width+pctr]) fwidth[cctr]=pctr-fstart[cctr]+1;
         if (ColorDepth==32) if (FontData32[cctr*Height*Width+hctr*Width+pctr]) fwidth[cctr]=pctr-fstart[cctr]+1;
      };
   };
};

//////////////////////////////////////////////////////////////////////
// FONTCLASS member functions (public)
//////////////////////////////////////////////////////////////////////

FONTCLASS::FONTCLASS(SCREENCLASS *screen){
   FontData8=NULL;
   FontData16=NULL;
   FontData32=NULL;
   Loaded=false;
   screenptr=screen;
};

void FONTCLASS::Load(char *filename){
   fstream fontfile;
   if (Loaded) FreeFont();
   fontfile.open(filename,0x81);
   fontfile.read(&ColorDepth,1);
   fontfile.read((char *)&Width,2);
   fontfile.read((char *)&Height,2);
   if (ColorDepth==8) FontData8=(unsigned char *)malloc(Width*Height*256);
   if (ColorDepth==15 || ColorDepth==16) FontData16=(unsigned short *)malloc(Width*Height*512);
   if (ColorDepth==32) FontData32=(unsigned long *)malloc(Width*Height*1024);
   if (ColorDepth==8) fontfile.read(FontData8,Width*Height*256);
   if (ColorDepth==15 || ColorDepth==16) fontfile.read((char *)FontData16,Width*Height*512);
   if (ColorDepth==32) fontfile.read((char *)FontData32,Width*Height*1024);
   fontfile.close();
   Loaded=true;
   CalculateWidth();
};

void FONTCLASS::LoadBmp(char *filename){
   fstream bmpfile;
   unsigned long rd=0,headersize;
   unsigned short bmpbpp,yctr,xctr,ictr,height,width;
   if (Loaded) FreeFont();
   bmpfile.open(filename,0x81);
   bmpfile.read((char *)&rd,2);
   if (rd!=19778){
      bmpfile.close();
      return;
   };
   bmpfile.seekg(10);
   bmpfile.read((char *)&headersize,4);
   bmpfile.read((char *)&rd,4);
   if (rd!=40){
      bmpfile.close();
      return;
   };
   bmpfile.read((char *)&Width,4);
   bmpfile.read((char *)&Height,4);
   width=Width/16;
   height=Height/16;
   if (Height%height!=0 || Width%width!=0){
      bmpfile.close();
      return;
   };
   bmpfile.read((char *)&rd,2);
   bmpfile.read((char *)&bmpbpp,2);
   bmpfile.read((char *)&rd,4);
   if (rd!=0){
      bmpfile.close();
      return;
   };
   bmpfile.seekg(headersize);
   if (bmpbpp!=8 && bmpbpp!=16 && bmpbpp!=24 && bmpbpp!=32){
      bmpfile.close();
      return;
   };
   ColorDepth=(unsigned char)bmpbpp;
   if (ColorDepth==24) ColorDepth=32;
   if (ColorDepth==8) FontData8=(unsigned char *)malloc(Width*Height);
   if (ColorDepth==15 || ColorDepth==16) FontData16=(unsigned short *)malloc(Width*Height*2);
   if (ColorDepth==32) FontData32=(unsigned long *)malloc(Width*Height*4);
   rd=0;
   for (ictr=0;ictr<Width/width;ictr++) for (yctr=0;yctr<Height;yctr++){
      bmpfile.seekg(headersize+Width*Height*bmpbpp/8-(yctr+1)*Width*bmpbpp/8+ictr*width*bmpbpp/8);
      if (ColorDepth==8) bmpfile.read(&FontData8[yctr%height*width+(yctr/height*Width/width+ictr)*width*height],width);
      if (ColorDepth==15 || ColorDepth==16) bmpfile.read((char *)&FontData16[yctr%height*width+(yctr/height*Width/width+ictr)*width*height],width*2);
      if (bmpbpp==32) bmpfile.read((char *)&FontData32[yctr%height*width+(yctr/height*Width/width+ictr)*width*height],width*4);
      if (bmpbpp==24) for (xctr=0;xctr<width;xctr++){
         bmpfile.read((char *)&rd,3);
         FontData32[yctr%height*width+(yctr/height*Width/width+ictr)*width*height+xctr]=rd;
      };
   };
   bmpfile.close();
   Loaded=true;
   Height=height;
   Width=width;
   CalculateWidth();
};

void FONTCLASS::Duplicate(FONTCLASS *font){
   FONTINFO info;
   if (Loaded) FreeFont();
   if (!font->GetInfo(&info)) return;
   ColorDepth=info.ColorDepth;
   Width=info.Width;
   Height=info.Height;
   Loaded=true;
   if (ColorDepth==8) FontData8=(unsigned char *)malloc(Height*Width*256);
   if (ColorDepth==15 || ColorDepth==16) FontData16=(unsigned short *)malloc(Height*Width*512);
   if (ColorDepth==32) FontData32=(unsigned long *)malloc(Height*Width*1024);
   if (ColorDepth==8) memcpy(FontData8,info.Data8,Height*Width*256);
   if (ColorDepth==15 || ColorDepth==16) memcpy(FontData16,info.Data16,Height*Width*512);
   if (ColorDepth==32) memcpy(FontData32,info.Data32,Height*Width*1024);
};

bool FONTCLASS::GetInfo(FONTINFO *info){
   if (!Loaded) return(false);
   info->Data32=FontData32;
   info->Data16=FontData16;
   info->Data8=FontData8;
   info->CharStart=fstart;
   info->CharWidth=fwidth;
   info->Height=Height;
   info->Width=Width;
   info->ColorDepth=ColorDepth;
   return(true);
};

void FONTCLASS::Convert(unsigned char tobpp){
   PALETTE defaultpal;
   SCREENINFO info;
   unsigned long *temp32,Color,ctr,maxctr;
   unsigned short *temp16;
   unsigned char *temp8,re,gr,bl;
   if (!screenptr->GetInfo(&info)) return;
   if (tobpp==8) temp8=(unsigned char *)malloc(Height*Width*256);
   if (tobpp==15 || tobpp==16) temp16=(unsigned short *)malloc(Height*Width*512);
   if (tobpp==32) temp32=(unsigned long *)malloc(Height*Width*1024);
   if (tobpp==8 && info.RecreateLookupTable) screenptr->CreatePaletteLookupTable();
   if (tobpp==8 || ColorDepth==8) screenptr->GetCurrentPalette(defaultpal);
   maxctr=Height*Width*256;
   for (ctr=0;ctr<maxctr;ctr++){
      if (ColorDepth==8) Color=FontData8[ctr];
      if (ColorDepth==15 || ColorDepth==16) Color=FontData16[ctr];
      if (ColorDepth==32) Color=FontData32[ctr];
      if (ColorDepth==8){
         re=defaultpal[Color*3]>>1;
         gr=defaultpal[Color*3+1]>>1;
         bl=defaultpal[Color*3+2]>>1;
      };
      if (ColorDepth==15){
         bl=(unsigned char)(Color&31);
         gr=(unsigned char)((Color&992)>>5);
         re=(unsigned char)((Color&31744)>>10);
      };
      if (ColorDepth==16){
         bl=(unsigned char)(Color&31);
         gr=(unsigned char)((Color&1984)>>6);
         re=(unsigned char)((Color&63488)>>11);
      };
      if (ColorDepth==32){
         bl=(unsigned char)((Color>>3)&31);
         gr=(unsigned char)((Color>>11)&31);
         re=(unsigned char)(Color>>19);
      };
      if (tobpp==8) Color=info.PaletteLookup[(re<<10)+(gr<<5)+bl];
      if (tobpp==15) Color=(re<<10)+(gr<<5)+bl;
      if (tobpp==16) Color=(re<<11)+(gr<<6)+bl;
      if (tobpp==32) Color=(re<<19)+(gr<<11)+(bl<<3);
      if (tobpp==8) temp8[ctr]=(unsigned char)Color;
      if (tobpp==15 || tobpp==16) temp16[ctr]=(unsigned short)Color;
      if (tobpp==32) temp32[ctr]=Color;
   };
   if (ColorDepth==8) free(FontData8);
   if (ColorDepth==15 || ColorDepth==16) free(FontData16);
   if (ColorDepth==32) free(FontData32);
   ColorDepth=tobpp;
   if (ColorDepth==8) FontData8=temp8;
   if (ColorDepth==15 || ColorDepth==16) FontData16=temp16;
   if (ColorDepth==32) FontData32=temp32;
};

void FONTCLASS::Release(void){
   if (Loaded) FreeFont();
   if (this!=NULL) delete(this);
};

//////////////////////////////////////////////////////////////////////
// IMAGECLASS member function (private)
//////////////////////////////////////////////////////////////////////

void IMAGECLASS::FreeImage(void){
   if (ImageData8!=NULL && ColorDepth==8) free(ImageData8);
   if (ImageData16!=NULL && (ColorDepth==15 || ColorDepth==16)) free(ImageData16);
   if (ImageData32!=NULL && ColorDepth==32) free(ImageData32);
};

//////////////////////////////////////////////////////////////////////
// IMAGECLASS member functions (public)
//////////////////////////////////////////////////////////////////////

IMAGECLASS::IMAGECLASS(SCREENCLASS *screen){
   ImageData8=NULL;
   ImageData16=NULL;
   ImageData32=NULL;
   ImagesLoaded=0;
   screenptr=screen;
};

void IMAGECLASS::Load(char *filename,unsigned long NumImages){
   fstream imagefile;
   if (ImagesLoaded) FreeImage();
   imagefile.open(filename,0x81);
   imagefile.read(&ColorDepth,1);
   imagefile.read((char *)&Width,2);
   imagefile.read((char *)&Height,2);
   if (ColorDepth==8) ImageData8=(unsigned char *)malloc(Width*Height*NumImages);
   if (ColorDepth==15 || ColorDepth==16) ImageData16=(unsigned short *)malloc(Width*Height*NumImages*2);
   if (ColorDepth==32) ImageData32=(unsigned long *)malloc(Width*Height*NumImages*4);
   if (ColorDepth==8) imagefile.read(ImageData8,Width*Height*NumImages);
   if (ColorDepth==15 || ColorDepth==16) imagefile.read((char *)ImageData16,Width*Height*NumImages*2);
   if (ColorDepth==32) imagefile.read((char *)ImageData32,Width*Height*NumImages*4);
   imagefile.close();
   ImagesLoaded=NumImages;
};

void IMAGECLASS::LoadBmp(char *filename){
   fstream bmpfile;
   unsigned long rd=0,headersize;
   unsigned short bmpbpp,yctr,xctr;
   if (ImagesLoaded) FreeImage();
   bmpfile.open(filename,0x81);
   bmpfile.read((char *)&rd,2);
   if (rd!=19778){
      bmpfile.close();
      return;
   };
   bmpfile.seekg(10);
   bmpfile.read((char *)&headersize,4);
   bmpfile.read((char *)&rd,4);
   if (rd!=40){
      bmpfile.close();
      return;
   };
   bmpfile.read((char *)&Width,4);
   bmpfile.read((char *)&Height,4);
   bmpfile.read((char *)&rd,2);
   bmpfile.read((char *)&bmpbpp,2);
   bmpfile.read((char *)&rd,4);
   if (rd!=0){
      bmpfile.close();
      return;
   };
   bmpfile.seekg(headersize);
   if (bmpbpp!=8 && bmpbpp!=16 && bmpbpp!=24 && bmpbpp!=32){
      bmpfile.close();
      return;
   };
   ColorDepth=(unsigned char)bmpbpp;
   if (ColorDepth==24) ColorDepth=32;
   if (ColorDepth==8) ImageData8=(unsigned char *)malloc(Width*Height);
   if (ColorDepth==15 || ColorDepth==16) ImageData16=(unsigned short *)malloc(Width*Height*2);
   if (ColorDepth==32) ImageData32=(unsigned long *)malloc(Width*Height*4);
   rd=0;
   for (yctr=0;yctr<Height;yctr++){
      bmpfile.seekg(headersize+Width*Height*bmpbpp/8-(yctr+1)*Width*bmpbpp/8);
      if (ColorDepth==8) bmpfile.read(&ImageData8[yctr*Width],Width);
      if (ColorDepth==15 || ColorDepth==16) bmpfile.read((char *)&ImageData16[yctr*Width],Width*2);
      if (bmpbpp==32) bmpfile.read((char *)&ImageData32[yctr*Width],Width*4);
      if (bmpbpp==24) for (xctr=0;xctr<Width;xctr++){
         bmpfile.read((char *)&rd,3);
         ImageData32[yctr*Width+xctr]=rd;
      };
   };
   bmpfile.close();
   ImagesLoaded=1;
};

void IMAGECLASS::LoadMultiBmp(char *filename,unsigned short width,unsigned short height){
   fstream bmpfile;
   unsigned long rd=0,headersize;
   unsigned short bmpbpp,yctr,xctr,ictr;
   if (ImagesLoaded) FreeImage();
   bmpfile.open(filename,0x81);
   bmpfile.read((char *)&rd,2);
   if (rd!=19778){
      bmpfile.close();
      return;
   };
   bmpfile.seekg(10);
   bmpfile.read((char *)&headersize,4);
   bmpfile.read((char *)&rd,4);
   if (rd!=40){
      bmpfile.close();
      return;
   };
   bmpfile.read((char *)&Width,4);
   bmpfile.read((char *)&Height,4);
   if (Height%height!=0 || Width%width!=0){
      bmpfile.close();
      return;
   };
   bmpfile.read((char *)&rd,2);
   bmpfile.read((char *)&bmpbpp,2);
   bmpfile.read((char *)&rd,4);
   if (rd!=0){
      bmpfile.close();
      return;
   };
   bmpfile.seekg(headersize);
   if (bmpbpp!=8 && bmpbpp!=16 && bmpbpp!=24 && bmpbpp!=32){
      bmpfile.close();
      return;
   };
   ColorDepth=(unsigned char)bmpbpp;
   if (ColorDepth==24) ColorDepth=32;
   if (ColorDepth==8) ImageData8=(unsigned char *)malloc(Width*Height);
   if (ColorDepth==15 || ColorDepth==16) ImageData16=(unsigned short *)malloc(Width*Height*2);
   if (ColorDepth==32) ImageData32=(unsigned long *)malloc(Width*Height*4);
   rd=0;
   for (ictr=0;ictr<Width/width;ictr++) for (yctr=0;yctr<Height;yctr++){
      bmpfile.seekg(headersize+Width*Height*bmpbpp/8-(yctr+1)*Width*bmpbpp/8+ictr*width*bmpbpp/8);
      if (ColorDepth==8) bmpfile.read(&ImageData8[yctr%height*width+(yctr/height*Width/width+ictr)*width*height],width);
      if (ColorDepth==15 || ColorDepth==16) bmpfile.read((char *)&ImageData16[yctr%height*width+(yctr/height*Width/width+ictr)*width*height],width*2);
      if (bmpbpp==32) bmpfile.read((char *)&ImageData32[yctr%height*width+(yctr/height*Width/width+ictr)*width*height],width*4);
      if (bmpbpp==24) for (xctr=0;xctr<width;xctr++){
         bmpfile.read((char *)&rd,3);
         ImageData32[yctr%height*width+(yctr/height*Width/width+ictr)*width*height+xctr]=rd;
      };
   };
   bmpfile.close();
   ImagesLoaded=Height/height*Width/width;
   Height=height;
   Width=width;
};

void IMAGECLASS::Make(unsigned short width,unsigned short height,unsigned char *ptr){
   if (ImagesLoaded) FreeImage();
   ImageData8=(unsigned char *)malloc(width*height);
   memcpy(ImageData8,ptr,width*height);
   ColorDepth=8;
   Width=width;
   Height=height;
   ImagesLoaded=1;
};

void IMAGECLASS::Make(unsigned short width,unsigned short height,unsigned short *ptr){
   if (ImagesLoaded) FreeImage();
   ImageData16=(unsigned short *)malloc(width*height*2);
   memcpy(ImageData16,ptr,width*height*2);
   ColorDepth=16;
   Width=width;
   Height=height;
   ImagesLoaded=1;
};

void IMAGECLASS::Make15(unsigned short width,unsigned short height,unsigned short *ptr){
   if (ImagesLoaded) FreeImage();
   ImageData16=(unsigned short *)malloc(width*height*2);
   memcpy(ImageData16,ptr,width*height*2);
   ColorDepth=15;
   Width=width;
   Height=height;
   ImagesLoaded=1;
};

void IMAGECLASS::Make(unsigned short width,unsigned short height,unsigned long *ptr){
   if (ImagesLoaded) FreeImage();
   ImageData32=(unsigned long *)malloc(width*height*4);
   memcpy(ImageData32,ptr,width*height*4);
   ColorDepth=32;
   Width=width;
   Height=height;
   ImagesLoaded=1;
};

void IMAGECLASS::Duplicate(IMAGECLASS *image){
   IMAGEINFO info;
   if (ImagesLoaded) FreeImage();
   if (!image->GetInfo(&info)) return;
   ColorDepth=info.ColorDepth;
   Width=info.Width;
   Height=info.Height;
   ImagesLoaded=info.NumOfImages;
   if (ColorDepth==8) ImageData8=(unsigned char *)malloc(Height*Width*ImagesLoaded);
   if (ColorDepth==15 || ColorDepth==16) ImageData16=(unsigned short *)malloc(Height*Width*ImagesLoaded*2);
   if (ColorDepth==32) ImageData32=(unsigned long *)malloc(Height*Width*ImagesLoaded*4);
   if (ColorDepth==8) memcpy(ImageData8,info.Data8,Height*Width*ImagesLoaded);
   if (ColorDepth==15 || ColorDepth==16) memcpy(ImageData16,info.Data16,Height*Width*ImagesLoaded*2);
   if (ColorDepth==32) memcpy(ImageData32,info.Data32,Height*Width*ImagesLoaded*4);
};

bool IMAGECLASS::GetInfo(IMAGEINFO *info){
   if (!ImagesLoaded) return(false);
   info->Data32=ImageData32;
   info->Data16=ImageData16;
   info->Data8=ImageData8;
   info->Height=Height;
   info->Width=Width;
   info->ColorDepth=ColorDepth;
   info->NumOfImages=ImagesLoaded;
   return(true);
};

void IMAGECLASS::Grab(short x,short y,short x2,short y2,SCREENCLASS *screen){
   SCREENINFO info;
   short temp,XCtr,YCtr;
   bool NoLock;
   if (ImagesLoaded) FreeImage();
   screen->GetInfo(&info);
   NoLock=info.Locked;
   if (x>x2){
      temp=x2;
      x2=x;
      x=temp;
   };
   if (y>y2){
      temp=y2;
      y2=y;
      y=temp;
   };
   ColorDepth=info.ColorDepth;
   Height=y2-y+1;
   Width=x2-x+1;
   if (ColorDepth==8) ImageData8=(unsigned char *)malloc(Height*Width);
   if (ColorDepth==15 || ColorDepth==16) ImageData16=(unsigned short *)malloc(Height*Width*2);
   if (ColorDepth==32) ImageData32=(unsigned long *)malloc(Height*Width*4);
   if (!NoLock) screen->Lock();
   if (x2<0 || y2<0 || x>=info.Width || y>=info.Width){
      if (ColorDepth==8) memset8(ImageData8,0,Height*Width);
      if (ColorDepth==15 || ColorDepth==16) memset16(ImageData16,0,Height*Width);
      if (ColorDepth==32) memset32(ImageData32,0,Height*Width);
   } else for (YCtr=0;YCtr<Height;YCtr++) for (XCtr=0;XCtr<Width;XCtr++){
      if (ColorDepth==8) ImageData8[YCtr*Width+XCtr]=(unsigned char)screen->GetPixel(XCtr+x,YCtr+y);
      if (ColorDepth==15 || ColorDepth==16) ImageData16[YCtr*Width+XCtr]=(unsigned short)screen->GetPixel(XCtr+x,YCtr+y);
      if (ColorDepth==32) ImageData32[YCtr*Width+XCtr]=screen->GetPixel(XCtr+x,YCtr+y);
   };
   if (!NoLock) screen->Unlock();
   ImagesLoaded=1;
};

void IMAGECLASS::Grab(short x,short y,short x2,short y2,BUFFERCLASS *buffer){
   BUFFERINFO info;
   unsigned short XCopy,XStart;
   short temp,YCtr;
   if (ImagesLoaded) FreeImage();
   buffer->GetInfo(&info);
   if (x>x2){
      temp=x2;
      x2=x;
      x=temp;
   };
   if (y>y2){
      temp=y2;
      y2=y;
      y=temp;
   };
   ColorDepth=info.ColorDepth;
   Height=y2-y+1;
   Width=x2-x+1;
   if (ColorDepth==8) ImageData8=(unsigned char *)malloc(Height*Width);
   if (ColorDepth==15 || ColorDepth==16) ImageData16=(unsigned short *)malloc(Height*Width*2);
   if (ColorDepth==32) ImageData32=(unsigned long *)malloc(Height*Width*4);
   if (x2<0 || y2<0 || x>=info.Width || y>=info.Width){
      if (ColorDepth==8) memset8(ImageData8,0,Height*Width);
      if (ColorDepth==15 || ColorDepth==16) memset16(ImageData16,0,Height*Width);
      if (ColorDepth==32) memset32(ImageData32,0,Height*Width);
   } else {
      XStart=x;
      XCopy=Width;
      if (x<0){
         XStart=0;
         XCopy+=x;
      };
      if (x2>=info.Width) XCopy-=x2+1-info.Width;
      for (YCtr=0;YCtr<Height;YCtr++){
         if (YCtr+y<0 || YCtr+y>=info.Height){
            if (ColorDepth==8) memset8(&ImageData8[YCtr*Width],0,Width);
            if (ColorDepth==15 || ColorDepth==16) memset16(&ImageData16[YCtr*Width],0,Width);
            if (ColorDepth==32) memset32(&ImageData32[YCtr*Width],0,Width);
         } else {
            if (x<0){
               if (ColorDepth==8) memset8(&ImageData8[YCtr*Width],0,abs(x));
               if (ColorDepth==15 || ColorDepth==16) memset16(&ImageData16[YCtr*Width],0,abs(x));
               if (ColorDepth==32) memset32(&ImageData32[YCtr*Width],0,abs(x));
            };
            if (ColorDepth==8) memcpy(&ImageData8[YCtr*Width+XStart-x],&info.Data8[YCtr*info.Width+y*info.Width+XStart],XCopy);
            if (ColorDepth==15 || ColorDepth==16) memcpy(&ImageData16[YCtr*Width+XStart-x],&info.Data16[YCtr*info.Width+y*info.Width+XStart],XCopy*2);
            if (ColorDepth==32) memcpy(&ImageData32[YCtr*Width+XStart-x],&info.Data32[YCtr*info.Width+y*info.Width+XStart],XCopy*4);
            if (x<0) XCopy-=abs(x);
            if (x2>=info.Width){
               if (ColorDepth==8) memset8(&ImageData8[YCtr*Width+XCopy],0,x2+1-info.Width);
               if (ColorDepth==15 || ColorDepth==16) memset16(&ImageData16[YCtr*Width+XCopy],0,x2+1-info.Width);
               if (ColorDepth==32) memset32(&ImageData32[YCtr*Width+XCopy],0,x2+1-info.Width);
            };
         };
      };
   };
   ImagesLoaded=1;
};

void IMAGECLASS::Convert(unsigned char tobpp){
   PALETTE defaultpal;
   SCREENINFO info;
   unsigned long *temp32,Color,ctr;
   unsigned short *temp16;
   unsigned char *temp8,re,gr,bl;
   if (!screenptr->GetInfo(&info)) return;
   if (tobpp==8) temp8=(unsigned char *)malloc(Height*Width*ImagesLoaded);
   if (tobpp==15 || tobpp==16) temp16=(unsigned short *)malloc(Height*Width*ImagesLoaded*2);
   if (tobpp==32) temp32=(unsigned long *)malloc(Height*Width*ImagesLoaded*4);
   if (tobpp==8 && info.RecreateLookupTable) screenptr->CreatePaletteLookupTable();
   if (tobpp==8 || ColorDepth==8) screenptr->GetCurrentPalette(defaultpal);
   for (ctr=0;ctr<Height*Width*ImagesLoaded;ctr++){
      if (ColorDepth==8) Color=ImageData8[ctr];
      if (ColorDepth==15 || ColorDepth==16) Color=ImageData16[ctr];
      if (ColorDepth==32) Color=ImageData32[ctr];
      if (ColorDepth==8){
         re=defaultpal[Color*3]>>1;
         gr=defaultpal[Color*3+1]>>1;
         bl=defaultpal[Color*3+2]>>1;
      };
      if (ColorDepth==15){
         bl=(unsigned char)(Color&31);
         gr=(unsigned char)((Color&992)>>5);
         re=(unsigned char)((Color&31744)>>10);
      };
      if (ColorDepth==16){
         bl=(unsigned char)(Color&31);
         gr=(unsigned char)((Color&1984)>>6);
         re=(unsigned char)((Color&63488)>>11);
      };
      if (ColorDepth==32){
         bl=(unsigned char)((Color>>3)&31);
         gr=(unsigned char)((Color>>11)&31);
         re=(unsigned char)(Color>>19);
      };
      if (tobpp==8) Color=info.PaletteLookup[(re<<10)+(gr<<5)+bl];
      if (tobpp==15) Color=(re<<10)+(gr<<5)+bl;
      if (tobpp==16) Color=(re<<11)+(gr<<6)+bl;
      if (tobpp==32) Color=(re<<19)+(gr<<11)+(bl<<3);
      if (tobpp==8) temp8[ctr]=(unsigned char)Color;
      if (tobpp==15 || tobpp==16) temp16[ctr]=(unsigned short)Color;
      if (tobpp==32) temp32[ctr]=Color;
   };
   if (ColorDepth==8) free(ImageData8);
   if (ColorDepth==15 || ColorDepth==16) free(ImageData16);
   if (ColorDepth==32) free(ImageData32);
   ColorDepth=tobpp;
   if (ColorDepth==8) ImageData8=temp8;
   if (ColorDepth==15 || ColorDepth==16) ImageData16=temp16;
   if (ColorDepth==32) ImageData32=temp32;
};

void IMAGECLASS::Release(void){
   if (ImagesLoaded) FreeImage();
   if (this!=NULL) delete(this);
};

