// Djgpp input control V1.0
//
// A library to handle input in DJGPP
//
// CopyRight (c) 2000, Johan Bos
// www.sludgesoft.com/johan
// bosjoh@fcmail.com
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// Define the input control
//////////////////////////////////////////////////////////////////////////////

#ifndef INPUTINCLUDED
#define INPUTINCLUDED
#define I_V1_0
#endif

//////////////////////////////////////////////////////////////////////////////
// Includes
//////////////////////////////////////////////////////////////////////////////

#ifndef GFXINCLUDED
#include <dpmi.h>
#include <go32.h>
#include <malloc.h>
#include <dos.h>
#endif

//////////////////////////////////////////////////////////////////////////////
// Structures/Classes
//////////////////////////////////////////////////////////////////////////////

class KEYBOARDCLASS{
   private:
      bool keystate[128];
   public:
      KEYBOARDCLASS();
      void UpdateState(unsigned char);
      bool Hold(unsigned char);
      unsigned char Pressed(void);
      unsigned char Read(void);
      void Release(void);
};

class MOUSECLASS{
   private:
      bool ButtonHold[6];
      unsigned short StdSensitivity;
   public:
      MOUSECLASS();
      void SetField(unsigned short,unsigned short,unsigned short,unsigned short);
      void SetXY(unsigned short,unsigned short);
      void GetXY(unsigned short &,unsigned short &);
      bool LeftButtonHold(void);
      bool RightButtonHold(void);
      bool MidButtonHold(void);
      bool LeftButtonPressed(void);
      bool RightButtonPressed(void);
      bool MidButtonPressed(void);
      bool LeftButtonReleased(void);
      bool RightButtonReleased(void);
      bool MidButtonReleased(void);
      void SetSensitivity(unsigned short);
      unsigned short GetSensitivity(void);
      void Release(void);
};

class INPUTCLASS{
   public:
      bool CreateMouse(MOUSECLASS *&);
      bool CreateKeyBoard(KEYBOARDCLASS *&);
      void Release(void);
};

//////////////////////////////////////////////////////////////////////////////
// New types
//////////////////////////////////////////////////////////////////////////////

typedef INPUTCLASS * INPUTCONTROL;
typedef MOUSECLASS * MOUSE;
typedef KEYBOARDCLASS * KEYBOARD;

//////////////////////////////////////////////////////////////////////////////
// Globals needed for the interrupt handler
//////////////////////////////////////////////////////////////////////////////

_go32_dpmi_seginfo OldKey,NewKey;
KEYBOARD keyboardptr;

void HookedKeyInt(void){
   unsigned char result;
   result=inp(0x60);
   keyboardptr->UpdateState(result);
};

void HookKeyInt(KEYBOARD kb){
   keyboardptr=kb;
   _go32_dpmi_get_protected_mode_interrupt_vector(0x9,&OldKey);
   NewKey.pm_offset=(long)HookedKeyInt;
   NewKey.pm_selector=_go32_my_cs();
   _go32_dpmi_chain_protected_mode_interrupt_vector(0x9,&NewKey);
};

void RestoreKeyInt(void){
   _go32_dpmi_set_protected_mode_interrupt_vector(0x9,&OldKey);
};

//////////////////////////////////////////////////////////////////////////////
// Global funtion to create input control
//////////////////////////////////////////////////////////////////////////////

bool CreateInputControl(INPUTCONTROL &input){
   input=new INPUTCLASS();
   if (input==NULL) return(false);
   return(true);
};

//////////////////////////////////////////////////////////////////////////////
// INPUTCLASS member functions (public)
//////////////////////////////////////////////////////////////////////////////

bool INPUTCLASS::CreateMouse(MOUSECLASS *&mouse){
   REGS Regs;
   Regs.x.ax=0;
   int86(0x33,&Regs,&Regs);
   if (!Regs.x.ax) return(false);
   mouse=new MOUSECLASS();
   if (mouse==NULL) return(false);
   return(true);
};

bool INPUTCLASS::CreateKeyBoard(KEYBOARDCLASS *&keyboard){
   keyboard=new KEYBOARDCLASS();
   if (keyboard==NULL) return(false);
   return(true);
};

void INPUTCLASS::Release(void){
   if (this!=NULL) delete(this);
};

//////////////////////////////////////////////////////////////////////////////
// MOUSECLASS member functions (public)
//////////////////////////////////////////////////////////////////////////////

MOUSECLASS::MOUSECLASS(){
   ButtonHold[0]=false;
   ButtonHold[1]=false;
   ButtonHold[2]=false;
   StdSensitivity=GetSensitivity();
};

void MOUSECLASS::SetField(unsigned short x,unsigned short y,unsigned short x2,unsigned short y2){
   REGS Regs;
   unsigned short temp;
   if (x2<x){
      temp=x2;
      x2=x;
      x=temp;
   };
   if (y2<y){
      temp=y2;
      y2=y;
      y=temp;
   };
   Regs.x.ax=7;
   Regs.x.cx=x;
   Regs.x.dx=x2;
   int86(0x33,&Regs,&Regs);
   Regs.x.ax=8;
   Regs.x.cx=y;
   Regs.x.dx=y2;
   int86(0x33,&Regs,&Regs);
};

void MOUSECLASS::SetXY(unsigned short x,unsigned short y){
   REGS Regs;
   Regs.x.ax=4;
   Regs.x.cx=x;
   Regs.x.dx=y;
   int86(0x33,&Regs,&Regs);
};

void MOUSECLASS::GetXY(unsigned short &x,unsigned short &y){
   REGS Regs;
   Regs.x.ax=3;
   int86(0x33,&Regs,&Regs);
   x=Regs.x.cx;
   y=Regs.x.dx;
};

bool MOUSECLASS::LeftButtonHold(void){
   REGS Regs;
   Regs.x.ax=3;
   int86(0x33,&Regs,&Regs);
   if (Regs.x.bx%2) return(true);
   return(false);
};

bool MOUSECLASS::RightButtonHold(void){
   REGS Regs;
   Regs.x.ax=3;
   int86(0x33,&Regs,&Regs);
   Regs.x.bx/=2;
   if (Regs.x.bx%2) return(true);
   return(false);
};

bool MOUSECLASS::MidButtonHold(void){
   REGS Regs;
   Regs.x.ax=3;
   int86(0x33,&Regs,&Regs);
   Regs.x.bx/=4;
   if (Regs.x.bx%2) return(true);
   return(false);
};

bool MOUSECLASS::LeftButtonPressed(void){
   if (LeftButtonHold()){
      if (!ButtonHold[0]){
         ButtonHold[0]=true;
         return(true);
      };
      ButtonHold[0]=true;
   } else ButtonHold[0]=false;
   return(false);
};

bool MOUSECLASS::RightButtonPressed(void){
   if (RightButtonHold()){
      if (!ButtonHold[1]){
         ButtonHold[1]=true;
         return(true);
      };
      ButtonHold[1]=true;
   } else ButtonHold[1]=false;
   return(false);
};

bool MOUSECLASS::MidButtonPressed(void){
   if (MidButtonHold()){
      if (!ButtonHold[2]){
         ButtonHold[2]=true;
         return(true);
      };
      ButtonHold[2]=true;
   } else ButtonHold[2]=false;
   return(false);
};

bool MOUSECLASS::LeftButtonReleased(void){
   if (!LeftButtonHold()){
      if (ButtonHold[3]){
         ButtonHold[3]=false;
         return(true);
      };
      ButtonHold[3]=false;
   } else ButtonHold[3]=true;
   return(false);
};

bool MOUSECLASS::RightButtonReleased(void){
   if (!RightButtonHold()){
      if (ButtonHold[4]){
         ButtonHold[4]=false;
         return(true);
      };
      ButtonHold[4]=false;
   } else ButtonHold[4]=true;
   return(false);
};

bool MOUSECLASS::MidButtonReleased(void){
   if (!MidButtonHold()){
      if (ButtonHold[5]){
         ButtonHold[5]=false;
         return(true);
      };
      ButtonHold[5]=false;
   } else ButtonHold[5]=true;
   return(false);
};

void MOUSECLASS::SetSensitivity(unsigned short speed){
   REGS Regs;
   Regs.x.ax=0x1B;
   int86(0x33,&Regs,&Regs);
   Regs.x.ax=0x1A;
   Regs.x.bx=speed*3;
   Regs.x.cx=speed*4;
   int86(0x33,&Regs,&Regs);
};

unsigned short MOUSECLASS::GetSensitivity(void){
   REGS Regs;
   Regs.x.ax=0x1B;
   int86(0x33,&Regs,&Regs);
   return(Regs.x.bx/3);
};

void MOUSECLASS::Release(void){
   SetSensitivity(StdSensitivity);
   if (this!=NULL) delete(this);
};

//////////////////////////////////////////////////////////////////////////////
// KEYBOARDCLASS member functions (public)
//////////////////////////////////////////////////////////////////////////////

KEYBOARDCLASS::KEYBOARDCLASS(){
   unsigned char ctr;
   for (ctr=0;ctr<128;ctr++) keystate[ctr]=0;
   HookKeyInt(this);
};

void KEYBOARDCLASS::UpdateState(unsigned char value){
   if (value>127) keystate[value-128]=false;
   else keystate[value]=true;
};

bool KEYBOARDCLASS::Hold(unsigned char keyid){
   Pressed();
   if (keyid>127) return(false);
   if (!keystate[keyid]) return(false);
   return(true);
};

unsigned char KEYBOARDCLASS::Pressed(void){
   REGS Regs;
   Regs.h.ah=11;
   int86(0x21,&Regs,&Regs);
   if (Regs.h.al) return(Read());
   return(0);
};

unsigned char KEYBOARDCLASS::Read(void){
   REGS Regs;
   Regs.x.ax=0;
   int86(0x16,&Regs,&Regs);
   if (Regs.x.ax%256==0) Regs.x.ax=128+(Regs.x.ax>>8);
   return(Regs.x.ax%256);
};

void KEYBOARDCLASS::Release(void){
   RestoreKeyInt();
   if (this!=NULL) delete(this);
};
