unit winamp;

interface
uses
 windows,Messages,classes,sysutils,ivsys,ivkernel;

type
 TAMPMode=(ampNotReady, ampStopped, ampPlaying,ampPaused);

 TAMPTrackInfo=record
  Length:integer;//secunds
  Samplerate:integer;// (i.e. 44100)
  Bitrate:integer;//  (i.e. 128)
  Channels:integer;// (i.e. 2)
 end;

 TAMPEQData=record //Equalizer
//(requires Winamp 2.05+)
  Bands:array[0..9]of integer;//The 10 bands of EQ data. 0-63 (+20db - -20db)
  Preamp:integer;//The preamp value. 0-63 (+20db - -20db)
  Enabled:boolean;
  AutoLoad:boolean;
 end;

 TWinamp=class(TComponent)
 private
  FAppHandle:HWND;
  FExeFile:string;
  FAutoOpen:boolean;
  FAutoClose:boolean;
 protected
  function GetWND:HWND;
  function GetWNDPE:HWND;
  function GetWNDEQ:HWND;
  function GetWNDMB:HWND;
  function GetVisible:boolean;
  procedure SetVisible(AValue:boolean);
  function GetPosition:integer;
  procedure SetPosition(ms:integer);
  function GetListPosition:integer;
  procedure SetListPosition(Index:integer);
  function GetMode:TAMPMode;
  function GetVersion:integer;
  function GetVersionStr:string;
  function GetTrackLength:integer;//sec
  procedure SetVolume(AValue:integer);//0..255
  procedure SetPanning(AValue:integer);//-128..127 (instead 0-255 in documentation)
  function GetListCount:integer;
  function GetTrackInfo:TAMPTrackInfo;
  function GetEQData:TAMPEQData;
  procedure SetEQData(const Data:TAMPEQData);
 public
  constructor Create(AOwner:TComponent);override;
  destructor Destroy;override;
  procedure Loaded;override;
  function FindExeFile:string;
  procedure SendCommand(Command:integer);
  procedure AboutDlg;
  procedure Open;
  procedure Close;
  procedure Show;
  procedure Hide;
  procedure HideAll;//hide all it's windows
  procedure ChangeDir(Dir:string);
  procedure AddFile(FileName:string);
  procedure AddFiles(FileNames:TStrings);
  procedure PlayFile(FileName:string);
  procedure StartPlay;//from begin of list
  procedure Play;
  procedure Pause;
  procedure Stop;
  procedure StopFadeOut;
  procedure Prev;
  procedure Next;
  procedure Back;
  procedure Step;
  procedure VolumeUp;
  procedure VolumeDown;
  procedure PlayFilesDlg;
  procedure OpenLocationDlg;
  procedure PreferencesDlg;
  procedure ToggleStayOnTop;
  procedure ToggleList;
  procedure ToggleEQ;
  procedure ListClear;
  procedure ListWrite;
  function GetPositionStr(TimeElapsed:boolean):string;
  property Version:integer read GetVersion;
  property VersionStr:string read GetVersionStr;
  property WND:HWND read GetWND;
  property WNDPE:HWND read GetWNDPE;
  property WNDEQ:HWND read GetWNDEQ;
  property WNDMB:HWND read GetWNDMB;
  property Visible:boolean read GetVisible write SetVisible;
  property Mode:TAMPMode read GetMode;
  property TrackLength:integer read GetTrackLength;//in secunds
  property Volume:integer write SetVolume;
  property Panning:integer write SetPanning;
  property Position:integer read GetPosition write SetPosition;//millisecunds
  property ListPosition:integer read GetListPosition write SetListPosition;
  property ListCount:integer read GetListCount;
  property TrackInfo:TAMPTrackInfo read GetTrackInfo;
  property EQData:TAMPEQDAta read GetEQDAta write SetEQData;
 published
  property ExeFile:string read FExeFile write FExeFile;
  property AutoOpen:boolean read FAutoOpen write FAutoOpen default True;
  property AutoClose:boolean read FAutoClose write FAutoClose default True;
 end;

const
{Winamp Messages}
 WM_WA_IPC=WM_USER;
 IPC_GETVERSION=0;
  {version:= SendMessage(hwnd_winamp,WM_WA_IPC,0,IPC_GETVERSION);
  Version will be 0x20yx for winamp 2.yx. versions previous to Winamp 2.0
  typically (but not always) use 0x1zyx for 1.zx versions}
 IPC_DELETE=101;
  {SendMessage(hwnd_winamp,WM_WA_IPC,0,IPC_DELETE);
  clear Winamp's internal playlist}
 IPC_STARTPLAY=102;
  {SendMessage(hwnd_winamp,WM_WA_IPC,0,IPC_STARTPLAY);
  Using IPC_STARTPLAY is like hitting 'Play' in Winamp, mostly}
 IPC_ISPLAYING=104;
  {res:= SendMessage(hwnd_winamp,WM_WA_IPC,0,IPC_ISPLAYING);
  IPC_ISPLAYING returns the status of playback.
  If it returns 1, it is playing. if it returns 3, it is paused,
  if it returns 0, it is not playing}
 IPC_GETOUTPUTTIME=105;
  {res = SendMessage(hwnd_winamp,WM_WA_IPC,mode,IPC_GETOUTPUTTIME);
  IPC_GETOUTPUTTIME returns the position in milliseconds of the
  current song (mode = 0), or the song length, in seconds (mode = 1).
  Returns -1 if not playing or error}
 IPC_JUMPTOTIME=106;
  {(requires Winamp 1.60+)
  SendMessage(hwnd_winamp,WM_WA_IPC,ms,IPC_JUMPTOTIME);
  IPC_JUMPTOTIME sets the position in milliseconds of the
  current song (approximately).
  Returns -1 if not playing, 1 on eof, or 0 if successful}
 IPC_WRITEPLAYLIST=120;
  {(requires Winamp 1.666+)
  SendMessage(hwnd_winamp,WM_WA_IPC,0,IPC_WRITEPLAYLIST);
  IPC_WRITEPLAYLIST writes the current playlist to <winampdir>\\Winamp.m3u,
  and returns the current playlist position.
  Kinda obsoleted by some of the 2.x new stuff, but still good for when
  using a front-end (instead of a plug-in)}
 IPC_SETPLAYLISTPOS=121;
  {(requires Winamp 2.0+)
   SendMessage(hwnd_winamp,WM_WA_IPC,position,IPC_SETPLAYLISTPOS)
  IPC_SETPLAYLISTPOS sets the playlsit position to 'position'}
 IPC_SETVOLUME=122;
  {(requires Winamp 2.0+)
  SendMessage(hwnd_winamp,WM_WA_IPC,volume,IPC_SETVOLUME);
  IPC_SETVOLUME sets the volume of Winamp (from 0-255)}
 IPC_SETPANNING=123;
  {(requires Winamp 2.0+)
  SendMessage(hwnd_winamp,WM_WA_IPC,panning,IPC_SETPANNING);
  IPC_SETPANNING sets the panning of Winamp (from 0 (left) to 255 (right))}
 IPC_GETLISTLENGTH=124;
  {(requires Winamp 2.0+)
  length:= SendMessage(hwnd_winamp,WM_WA_IPC,0,IPC_GETLISTLENGTH);
  IPC_GETLISTLENGTH returns the length of the current playlist, in tracks}
 IPC_SETSKIN=200;
  {(requires Winamp 2.04+, only usable from plug-ins (not external apps))
   SendMessage(hwnd_winamp,WM_WA_IPC,(WPARAM)"skinname",IPC_SETSKIN);
   IPC_SETSKIN sets the current skin to "skinname". Note that skinname
   can be the name of a skin, a skin .zip file, with or without path.
   If path isn't specified, the default search path is the winamp skins directory}
 IPC_GETSKIN=201;
  {(requires Winamp 2.04+, only usable from plug-ins (not external apps))
  SendMessage(hwnd_winamp,WM_WA_IPC,(WPARAM)skinname_buffer,IPC_GETSKIN);
  IPC_GETSKIN puts the directory where skin bitmaps can be found
  into  skinname_buffer.
  skinname_buffer must be MAX_PATH characters in length.
  When using a .zip'd skin file, it'll return a temporary directory
  where the ZIP was decompressed}
 IPC_EXECPLUG=202;
  {(requires Winamp 2.04+, only usable from plug-ins (not external apps))
  SendMessage(hwnd_winamp,WM_WA_IPC,(WPARAM)"vis_file.dll",IPC_EXECPLUG);
  IPC_EXECPLUG executes a visualization plug-in pointed to by WPARAM.
  the format of this string can be:
  "vis_whatever.dll"
  "vis_whatever.dll,0" // (first mod, file in winamp plug-in dir)
  "C:\\dir\\vis_whatever.dll,1"}
 IPC_GETPLAYLISTFILE=211;
  {(requires Winamp 2.04+, only usable from plug-ins (not external apps))
  name:=SendMessage(hwnd_winamp,WM_WA_IPC,index,IPC_GETPLAYLISTFILE);
  IPC_GETPLAYLISTFILE gets the filename of the playlist entry [index].
  returns a pointer to it. returns NULL on error}
 IPC_GETPLAYLISTTITLE=212;
  {(requires Winamp 2.04+, only usable from plug-ins (not external apps))
  name=SendMessage(hwnd_winamp,WM_WA_IPC,index,IPC_GETPLAYLISTTITLE);
  IPC_GETPLAYLISTTITLE gets the title of the playlist entry [index].
  returns a pointer to it. returns NULL on error}
 IPC_GETLISTPOS=125;
  {(requires Winamp 2.05+)
  pos:=SendMessage(hwnd_winamp,WM_WA_IPC,0,IPC_GETLISTPOS);
  IPC_GETLISTPOS returns the playlist position. A lot like IPC_WRITEPLAYLIST
  only faster since it doesn't have to write out the list}
 IPC_GETINFO=126;
  {(requires Winamp 2.05+)
  inf:=SendMessage(hwnd_winamp,WM_WA_IPC,mode,IPC_GETINFO);
  IPC_GETINFO returns info about the current playing song. The value
  it returns depends on the value of 'mode'.
  Mode      Meaning
  0         Samplerate (i.e. 44100)
  1         Bitrate  (i.e. 128)
  2         Channels (i.e. 2)}
 IPC_GETEQDATA=127;
   {(requires Winamp 2.05+)
   data:=SendMessage(hwnd_winamp,WM_WA_IPC,pos,IPC_GETEQDATA);
   IPC_GETEQDATA queries the status of the EQ.
   The value returned depends on what 'pos' is set to:
   Value      Meaning
   0-9        The 10 bands of EQ data. 0-63 (+20db - -20db)
   10         The preamp value. 0-63 (+20db - -20db)
   11         Enabled. zero if disabled, nonzero if enabled.
   12         Autoload. zero if disabled, nonzero if enabled}
 IPC_SETEQDATA=128;
   {(requires Winamp 2.05+)
   SendMessage(hwnd_winamp,WM_WA_IPC,pos,IPC_GETEQDATA);
   SendMessage(hwnd_winamp,WM_WA_IPC,value,IPC_SETEQDATA);
   IPC_SETEQDATA sets the value of the last position retrieved by IPC_GETEQDATA}
 IPC_PLAYFILE=100;
   {cds:TCopyDataStruct;
   Buffer: array[0..MAX_PATH] of Char;
   ....
   cds.dwData:=IPC_PLAYFILE;
   cds.lpData:=@Buffer;
   cds.cbData:=LengthBuffer;
   res:=SendMessage(hwnd_winamp,WM_COPYDATA,Application.Handle,integer(@cds));
   This will play file}
 IPC_CHDIR=103;
  {cds:TCopyDataStruct;
  cds.dwData:= IPC_CHDIR;
  cds.lpData <<'c:\download';
  cds.cbData = length('c:\download')+1; // include space for null
  SendMessage(hwnd_winamp,WM_COPYDATA,(WPARAM)NULL,(LPARAM)&cds);
  This will make Winamp change to the directory C:\\download}
 //WM_COMMAND messages that you can use to send
 // Winamp misc commands.
 // To send these, use:
 //
 // SendMessage(hwnd_winamp, WM_COMMAND,command_name,0);
 WINAMP_OPTIONS_EQ=40036; // toggles the EQ window
 WINAMP_OPTIONS_PLEDIT=40040; // toggles the playlist window
 WINAMP_VOLUMEUP=40058; // turns the volume up a little
 WINAMP_VOLUMEDOWN=40059; // turns the volume down a little
 WINAMP_FFWD5S=40060; // fast forwards 5 seconds
 WINAMP_REW5S=40061; // rewinds 5 seconds
// the following are the five main control buttons, with optionally shift
// or control pressed
 WINAMP_BUTTON1      =40044; //Previous Track
 WINAMP_BUTTON2      =40045; //Play
 WINAMP_BUTTON3      =40046; //Pause
 WINAMP_BUTTON4      =40047; //Stop
 WINAMP_BUTTON5      =40048; //Next Track
 WINAMP_BUTTON1_SHIFT=40144; //rewinds 5 seconds
 WINAMP_BUTTON2_SHIFT=40145; //Open file and play
 WINAMP_BUTTON3_SHIFT=40146; //Null
 WINAMP_BUTTON4_SHIFT=40147; //Fades out and stops
 WINAMP_BUTTON5_SHIFT=40148; //fast forwards 5 seconds
 WINAMP_BUTTON1_CTRL =40154; //Starts previous file in the playlist
 WINAMP_BUTTON2_CTRL =40155; //Open location
 WINAMP_BUTTON3_CTRL =40156; //Null
 WINAMP_BUTTON4_CTRL =40157; //Null
 WINAMP_BUTTON5_CTRL =40158; //Starts next file in the playlist
 WINAMP_FILE_PLAY    =40029; // pops up the load file(s) box
 WINAMP_OPTIONS_PREFS=40012; // pops up the preferences
 WINAMP_OPTIONS_AOT  =40019; // toggles always on top
 WINAMP_HELP_ABOUT   =40041; // pops up the about box

 Winamp_exe='Winamp.exe';
 Winamp_ClassName='Winamp v1.x';
 WinampEQ_ClassName='Winamp EQ';//equalizer
 WinampMB_ClassName='Winamp MB';//minibrowser
 WinampPE_ClassName='Winamp PE';//play list editor

function SecundTimeToStr(Secunds:integer):string;

implementation

uses
 shellapi;

function SecundTimeToStr(Secunds:integer):string;
var
 s,m:integer;
 strm,strs:string;
begin
 if Secunds<0 then Result:='--:--' else begin
  m:=Secunds div 60;
  s:=Secunds-m*60;
  strm:=intToStr(m);
  if length(strm)=1 then strm:='0'+strm;
  strs:=intTostr(s);
  if length(strs)=1 then strs:='0'+strs;
  Result:=strm+':'+strs;
 end;
end;

{TWinamp}
constructor TWinamp.Create(AOwner:Tcomponent);
begin
 inherited Create(AOwner);
 FAutoOpen:=true;
 FAutoClose:=true;
end;

destructor TWinamp.Destroy;
begin
 if (not (csDesigning in ComponentState)) and FAutoClose then Close;
 inherited Destroy;
end;

procedure TWinamp.Loaded;
begin
 inherited Loaded;
 if (not (csDesigning in ComponentState)) and FAutoOpen then Open;
end;

function TWinamp.GetVisible:boolean;
begin
 Result:=IsWindowVisible(WND);
end;

procedure TWinamp.SetVisible(AValue:boolean);
begin
 if AValue then Show else Hide;
end;

function TWinamp.FindExeFile:string;
const
 reg_Path='SOFTWARE\Microsoft\Windows\CurrentVersion\App Paths\Winamp.exe\';
 EXE_FILE='WINAMP.EXE';
 reg_ProgramFilesDir='SOFTWARE\Microsoft\Windows\CurrentVersion\ProgramFilesDir';
var
 s,sf:string;
 n,i:integer;
begin
 Result:=RegReadString(HKEY_LOCAL_MACHINE,reg_Path);
 if fileexists(Result)then exit;
 s:=RegReadString(HKEY_LOCAL_MACHINE,Reg_Path+'Path');
 Result:=NormalPath(s)+'Winamp.exe';
 if fileexists(Result)then exit;
 s:=RegReadString(HKEY_CLASSES_ROOT,'Winamp.File\DefaultIcon\');
 n:=pos(',',s);
 if n<>0 then begin
  delete(s,n,1000);
  s:=trim(s);
  sf:=Uppercase(extractfilename(s));
  if (sf=EXE_FILE)and(fileexists(s))then begin
   Result:=s;
   exit;
  end;
 end;
 s:=RegReadString(HKEY_CLASSES_ROOT,'Winamp.File\shell\open\command\');
 for i:=length(s) downto 1 do if s[i]='"' then delete(s,i,1);
 n:=pos('%',s);
 if n<>0 then begin
  delete(s,n,1000);
  s:=trim(s);
  sf:=Uppercase(extractfilename(s));
  if (sf=EXE_FILE)and(fileexists(s))then begin
   Result:=s;
   exit;
  end;
 end;
 s:=RegReadString(HKEY_LOCAL_MACHINE,reg_ProgramFilesDir);
 Result:=NormalPath(s)+'Winamp\Winamp.exe';
 if fileexists(Result)then exit;
 Result:='';
end;

function TWinamp.GetWND:HWND;
begin
 Result:=FindWindow(Winamp_ClassName,nil);
end;

function TWinamp.GetWNDEQ:HWND;
begin
 Result:=FindWindow(WinampEQ_ClassName,nil);
end;

function TWinamp.GetWNDPE:HWND;
begin
 Result:=FindWindow(WinampPE_ClassName,nil);
end;

function TWinamp.GetWNDMB:HWND;
begin
 Result:=FindWindow(WinampMB_ClassName,nil);
end;

procedure TWinamp.Open;
var
 i:integer;
 exe:THandle;
 serr:string;
begin
 if ExeFile='' then ExeFile:=FindExeFile;
 if WND<>0 then exit;
 if fileexists(ExeFile)=false then raise Exception.Create(Winamp_exe+' not found');
 exe:=ShellExecute(FAppHandle, nil,pchar(ExeFile),'','',SW_SHOW);
 serr:='';
 case exe of
 0:serr:='Out of memory or resources';
 ERROR_FILE_NOT_FOUND:serr:='File not found';
 ERROR_PATH_NOT_FOUND:serr:='Path not found';
 ERROR_BAD_FORMAT:serr:='EXE file is invalid';
 SE_ERR_ACCESSDENIED:serr:='Denied access';
 SE_ERR_OOM:serr:='Not enough memory';
 SE_ERR_SHARE:serr:='A sharing violation occurred';
 end;
 if (sErr='')and(exe>=32)and(exe<=32)then sErr:='Unknown error';
 if sErr<>'' then begin
  sErr:=Winamp_exe+': '+sErr;
  raise Exception.Create(sErr);
 end;
 for i:=1 to 30 do begin
  if WND<>0 then break;
  sleep(100);
 end;
end;

procedure TWinamp.Close;
begin
 SendMessage(WND,WM_SYSCOMMAND,SC_CLOSE,WM_Command);
end;

procedure TWinamp.Show;
begin
 ShowWindow(WND,SW_SHOW);
end;

procedure TWinamp.Hide;
begin
 ShowWindow(WND,SW_HIDE);
end;

procedure TWinamp.HideAll;//hide all it's windows
begin
 ShowWindow(WND,SW_HIDE);
// SendMessage(WNDPE,WM_SYSCOMMAND,SC_CLOSE,WM_Command);
// SendMessage(WNDEQ,WM_SYSCOMMAND,SC_CLOSE,WM_Command);
// SendMessage(WNDMB,WM_SYSCOMMAND,SC_CLOSE,WM_Command);
 ShowWindow(WNDPE,SW_HIDE);
 ShowWindow(WNDEQ,SW_HIDE);
 ShowWindow(WNDMB,SW_HIDE);
end;

function TWinamp.GetVersion:integer;
begin
 Result:=SendMessage(WND,WM_WA_IPC,0,IPC_GETVERSION);
end;

function TWinamp.GetVersionStr:string;
var
 iVersion:integer;
begin
 iVersion:=GetVersion;
 if iVersion<>0 then begin
  Result:=inttohex(iVersion,0);
  Result:=Result[1]+'.'+copy(Result,2,8);
 end else Result:='';
end;

procedure TWinamp.ListWrite;
begin
 SendMessage(WND,WM_WA_IPC,0,IPC_WRITEPLAYLIST);
//  IPC_WRITEPLAYLIST writes the current playlist to <winampdir>\\Winamp.m3u,
end;

procedure TWinamp.ListClear;
begin
 SendMessage(WND,WM_WA_IPC,0,IPC_DELETE);
end;

procedure TWinamp.AboutDlg;
begin
 SendMessage(WND,WM_COMMAND,WINAMP_HELP_ABOUT,0);
end;

procedure TWinamp.StartPlay;
begin
 SendMessage(WND,WM_WA_IPC,0,IPC_STARTPLAY);
end;

procedure TWinamp.SendCommand(Command:integer);
begin
 SendMessage(WND,WM_COMMAND,Command,0);
end;

procedure TWinamp.PlayFilesDlg;
begin
 SendMessage(WND,WM_COMMAND,WINAMP_BUTTON2_SHIFT,0);
end;

procedure TWinamp.OpenLocationDlg;
begin
 SendMessage(WND,WM_COMMAND,WINAMP_BUTTON2_CTRL,0);
end;

procedure TWinamp.PreferencesDlg;
begin
 SendMessage(WND,WM_COMMAND,WINAMP_OPTIONS_PREFS,0);
end;

procedure TWinamp.ToggleStayOnTop;
begin
 SendMessage(WND,WM_COMMAND,WINAMP_OPTIONS_AOT,0);
end;

procedure TWinamp.ToggleList;
begin
 SendMessage(WND,WM_COMMAND,WINAMP_OPTIONS_PLEDIT,0);
end;

procedure TWinamp.ToggleEQ;
begin
 SendMessage(WND,WM_COMMAND,WINAMP_OPTIONS_EQ,0);
end;

procedure TWinamp.Play;
begin
 SendMessage(WND,WM_COMMAND,WINAMP_BUTTON2,0);
end;

procedure TWinamp.Pause;
begin
 SendMessage(WND,WM_COMMAND,WINAMP_BUTTON3,0);
end;

procedure TWinamp.Stop;
begin
 SendMessage(WND,WM_COMMAND,WINAMP_BUTTON4,0);
end;

procedure TWinamp.StopFadeOut;
begin
 SendMessage(WND,WM_COMMAND,WINAMP_BUTTON4_SHIFT,0);
end;

procedure TWinamp.Prev;
begin
 SendMessage(WND,WM_COMMAND,WINAMP_BUTTON1,0);
end;

procedure TWinamp.Next;
begin
 SendMessage(WND,WM_COMMAND,WINAMP_BUTTON5,0);
end;

procedure TWinamp.Back;
begin
 SendMessage(WND,WM_COMMAND,WINAMP_BUTTON1_SHIFT,0);
end;

procedure TWinamp.Step;
begin
 SendMessage(WND,WM_COMMAND,WINAMP_BUTTON5_SHIFT,0);
end;

procedure TWinamp.VolumeUp;
begin
 SendMessage(WND,WM_COMMAND, WINAMP_VOLUMEUP,0);
end;

procedure TWinamp.VolumeDown;
begin
 SendMessage(WND,WM_COMMAND, WINAMP_VOLUMEDOWN,0);
end;

function TWinamp.GetMode:TAMPMode;
var
 wnd:HWND;
begin
 wnd:=GetWND;
 Result:=ampNotReady;
 if wnd<>0 then begin
  case SendMessage(WND,WM_WA_IPC,0,IPC_ISPLAYING) of
   0:Result:=ampStopped;
   1:Result:=ampPlaying;
   3:Result:=ampPaused;
  end;
 end;
end;

procedure TWinamp.SetVolume(AValue:integer);//0-255
begin
 SendMessage(WND,WM_WA_IPC,AValue,IPC_SETVOLUME);
end;

procedure TWinamp.SetPanning(AValue:integer);//0-255
begin
 SendMessage(WND,WM_WA_IPC,AValue,IPC_SETPANNING);
end;

procedure TWinamp.SetPosition(ms:integer);
begin
 SendMessage(WND,WM_WA_IPC,ms,IPC_JUMPTOTIME);
end;

function TWinamp.GetPosition:integer;
begin
 Result:=SendMessage(WND,WM_WA_IPC,0,IPC_GETOUTPUTTIME);
end;

procedure TWinamp.SetListPosition(Index:integer);
begin
 SendMessage(WND,WM_WA_IPC,Index,IPC_SETPLAYLISTPOS);
end;

function TWinamp.GetListPosition:integer;
begin
 Result:=SendMessage(WND,WM_WA_IPC,0,IPC_GETLISTPOS);
end;

function TWinamp.GetListCount:integer;
begin
 Result:=SendMessage(WND,WM_WA_IPC,0,IPC_GETLISTLENGTH);
end;

function TWinamp.GetTrackLength:integer;
begin
 Result:=SendMessage(WND,WM_WA_IPC,1,IPC_GETOUTPUTTIME);
end;

function TWinamp.GetPositionStr(TimeElapsed:boolean):string;
var
// ms,s,m:integer;
// strm,strs:string;
 ms,sec:integer;
begin
 ms:=Position;
 if ms>=0 then sec:=ms div 1000
 else sec:=-1;
 if (ms>=0) and (TimeElapsed=false) then begin
  sec:=TrackLength-sec;
 end;
 Result:=SecundTimeToStr(sec);
(* ms:=position;
 if ms<=0 then Result:='--:--' else begin
  if TimeElapsed=false then begin
   ms:=TrackLength*1000-ms;
  end;
  m:=ms div 60000;
  s:=ms div 1000-m*60;
  strm:=intToStr(m);
  if length(strm)=1 then strm:='0'+strm;
  strs:=intTostr(s);
  if length(strs)=1 then strs:='0'+strs;
  Result:=strm+':'+strs;
 end;*)
end;

function TWinamp.GetTrackInfo:TAMPTrackInfo;
var
 w:HWND;
begin
 w:=WND;
 Result.Length:=SendMessage(w,WM_WA_IPC,1,IPC_GETOUTPUTTIME);
 Result.Samplerate:=SendMessage(w,WM_WA_IPC,0,IPC_GETINFO);
 Result.Bitrate:=SendMessage(w,WM_WA_IPC,1,IPC_GETINFO);
 Result.Channels:=SendMessage(w,WM_WA_IPC,2,IPC_GETINFO);
end;

function TWinamp.GetEQData:TAMPEQData;
var
 i:integer;
 w:HWND;
begin
 w:=WND;
 for i:=0 to 9 do Result.Bands[i]:=SendMessage(w,WM_WA_IPC,i,IPC_GETEQDATA);
 Result.Preamp:=SendMessage(w,WM_WA_IPC,10,IPC_GETEQDATA);
 Result.Enabled:=boolean(SendMessage(w,WM_WA_IPC,11,IPC_GETEQDATA));
 Result.AutoLoad:=boolean(SendMessage(w,WM_WA_IPC,12,IPC_GETEQDATA));
end;

procedure TWinamp.SetEQData(const Data:TAMPEQData);
var
 i:integer;
 w:HWND;
begin
 w:=WND;
 for i:=0 to 9 do begin
  SendMessage(w,WM_WA_IPC,i,IPC_GETEQDATA);
  SendMessage(w,WM_WA_IPC,Data.Bands[i],IPC_SETEQDATA);
 end;
 SendMessage(w,WM_WA_IPC,10,IPC_GETEQDATA);
 SendMessage(w,WM_WA_IPC,Data.Preamp,IPC_SETEQDATA);
 SendMessage(w,WM_WA_IPC,11,IPC_GETEQDATA);
 SendMessage(w,WM_WA_IPC,integer(Data.Enabled),IPC_SETEQDATA);
 SendMessage(w,WM_WA_IPC,12,IPC_GETEQDATA);
 SendMessage(w,WM_WA_IPC,integer(Data.AutoLoad),IPC_SETEQDATA);
end;

procedure TWinamp.AddFile(FileName:string);
var
 cds:TCopyDataStruct;
 Buffer: array[0..MAX_PATH+1] of Char;
begin
 FileName:=FileName+#0; // include null
 StrCopy(Buffer,pchar(FileName));
 cds.dwData:=IPC_PLAYFILE;
 cds.lpData:=@Buffer;
 cds.cbData:=length(FileName);
 SendMessage(WND,WM_COPYDATA,FAppHandle,integer(@cds));
end;

procedure TWinamp.AddFiles(FileNames:TStrings);
var
 i:integer;
begin
 for i:=0 to FileNames.Count-1 do begin
  AddFile(FileNames[i]);
 end;
end;

procedure TWinamp.ChangeDir(Dir:string);
var
 cds:TCopyDataStruct;
 Buffer: array[0..MAX_PATH+1] of Char;
begin
 Dir:=Dir+#0; // include null
 StrCopy(Buffer,pchar(Dir));
 cds.dwData:=IPC_CHDIR;
 cds.lpData:=@Buffer;
 cds.cbData:=length(Dir);
 SendMessage(WND,WM_COPYDATA,FAppHandle,integer(@cds));
end;

procedure TWinamp.PlayFile(FileName:string);
begin
 ListClear;
 AddFile(FileName);
 Play;
end;

end.
