/* mciplay.cpp

   A simple function to send the playback time back to the main window and
   handle seeks. This operates using the MCI command strings, and assumes
   that a sound file aliased "sounder" is already loaded. Written by
   Jeff Tsay (ctsay@pasteur.eecs.berkeley.edu).

   Added CD audio support.

   Last modified : 11/04/97 */

#define  STRICT
#include <windows.h>

#include "w32_util.h"
#include "str_lib.h"
#include "args.h"
#include "mpw_msg.h"

char mci_buffer[1024];
char mci_command[256];

HWND hWndRep = NULL;
BOOL CDMode = FALSE;
BOOL final_track = FALSE;
int  this_track_num = 1;
int  next_track_num = 2;
char *this_track_str = NULL;
char *next_track_str = NULL;

BOOL send_play_string(BOOL from_track_begin=FALSE)
{
   DWORD error;

   lstrcpy(mci_command, "play sounder");

   if (CDMode && from_track_begin) {
      lstrcat(mci_command, " from ");
      lstrcat(mci_command, this_track_str);
   }

   if (CDMode && !final_track) {
		lstrcat(mci_command, " to ");
      lstrcat(mci_command, next_track_str);
   }
   lstrcat(mci_command, " notify");

   error = mciSendString(mci_command, mci_buffer, 1024, hWndRep);

   if (!CDMode && (error != 0)) {
      report_mci_error(hWndRep, error, "Could not play");
      return FALSE;
   }
   return TRUE;
}

DWORD mciplay (MCI_Args *args)
{
	char ms_buffer[32];
   char minute_str[4];
   char second_str[4];
   char cs_str[4];
   int  skip_time = 0;

   DWORD error;

   // These should not change while playing
   HANDLE mutex = args->mutex;
   int32 length = args->length;
   hWndRep      = args->hWnd;
	CDMode       = args->CDMode;
   final_track  = args->final_track;

   if (CDMode) {
	   this_track_str = args->this_track_str;
      this_track_num = atoi(this_track_str);
      next_track_str = args->next_track_str;
      if (!final_track) {
         next_track_num = atoi(next_track_str);
         skip_time = 100;
      } else {
         skip_time = 200;
      }
   }

   if (!send_play_string(TRUE))
      ExitThread(-1);

	while(true) {

   	// lock the mutex
   	WaitForSingleObject(mutex, INFINITE);

      // Check if the user hit stop
   	if (args->stop) {

		   error = mciSendString("stop sounder wait", mci_buffer, 1024, NULL);

         if (error != 0) {
	         report_mci_error(hWndRep, error, "Could not stop");
         }

         args->done = true;
         ReleaseMutex(mutex);
      	ExitThread(0);
      }

      // Update the scroll bar
		error = mciSendString("status sounder position wait",
                            mci_buffer, 1024, NULL);

      if (error != 0) {
	      report_mci_error(hWndRep, error,
                          "Could not get position");
         ReleaseMutex(mutex);
         ExitThread(-1);
      }

      if (CDMode) {
         char temp[32];
			char *ptr = mci_buffer;
         int  total_ms;
         int  colon_pos=0;
         int  start_at=0;
         BOOL non_zero_found = FALSE;

         while (*ptr != ':') {
            if ((*ptr == '0') && !non_zero_found) start_at++;
            else non_zero_found = TRUE;
         	colon_pos++;
            ptr++;
         }
         ptr++;

         lstrcpyn(temp, mci_buffer, colon_pos + 1);

         if (lstrcmp(temp + start_at, this_track_str) != 0) {
            total_ms = length;
         } else {

           lstrcpyn(minute_str, ptr, 3);
           total_ms = atoi(minute_str) * 60000;

           while (*ptr++ != ':');
           lstrcpyn(second_str, ptr, 3);
           total_ms += atoi(second_str) * 1000;

			  while (*ptr++ != ':');
			  lstrcpyn(cs_str, ptr, 3);
           total_ms += atoi(cs_str) * 10;
         }

			PostMessage(hWndRep, SCROLL_POS, total_ms, 0);

         // This hack is necessary because CD's are screwy

         if ((total_ms+skip_time) >= length) {

  			   error = mciSendString("stop sounder", mci_buffer, 1024, NULL);

			   ReleaseMutex(mutex);

            if (error != 0) {
	   		   report_mci_error(hWndRep, error,
                                "Could not stop device at end");
               ExitThread(-1);
            }

            PostMessage(hWndRep, MM_MCINOTIFY, MCI_NOTIFY_SUCCESSFUL, 0);
            ExitThread(0);
         }

      } else {

			PostMessage(hWndRep, SCROLL_POS, atoi(mci_buffer), 0);
      }

		// Handle seeks

		if (args->position_change) {

			error = mciSendString("stop sounder", mci_buffer, 1024, NULL);

         if (error != 0) {
			   report_mci_error(hWndRep, error,
                             "Could not stop device before seek");
			   ReleaseMutex(mutex);
            ExitThread(-1);
         }

			lstrcpy(mci_command, "seek sounder to ");

         if (CDMode) {

            if (args->desired_position > (length - skip_time))
               args->desired_position = length - skip_time;

         	lstrcat(mci_command, this_track_str);
            lstrcat(mci_command, ":");

            int minutes = args->desired_position / 60000;
            int seconds = (args->desired_position / 1000) -
                          (minutes * 60);
				int cs      = (args->desired_position / 10) -
                          (minutes * 6000) -
                          (seconds * 100);
            itoa(minutes, minute_str, 10);
            itoa(seconds, second_str, 10);
            itoa(cs, cs_str, 10);

            if (minutes < 10)
            	lstrcat(mci_command, "0");
            lstrcat(mci_command, minute_str);

            lstrcat(mci_command, ":");

            if (seconds < 10)
            	lstrcat(mci_command, "0");
            lstrcat(mci_command, second_str);

            lstrcat(mci_command, ":");

            if (cs < 10)
            	lstrcat(mci_command, "0");
            lstrcat(mci_command, cs_str);

         } else {

				lstrcat(mci_command, itoa(args->desired_position,
                                      ms_buffer, 10));
			}

			lstrcat(mci_command, " wait");

			error = mciSendString(mci_command, mci_buffer, 1024, NULL);

         if (error != 0) {
			   report_mci_error(hWndRep, error,
                             "Could not seek");
			   ReleaseMutex(mutex);
            ExitThread(-1);
         }

			if (args->playing) {
            if (!send_play_string(FALSE)) {
               ReleaseMutex(mutex);
               ExitThread(-1);
            }
         }

			args->position_change = false;

			PostMessage(hWndRep, SEEK_ACK, 0, 0);
			ReleaseMutex(mutex);

		} else {
			ReleaseMutex(mutex);
			Sleep(100);
      }
	}
}


