/* mp2win.cpp

	Interface for maplay written by Jeff Tsay. E-mail address:
	ctsay@pasteur.eecs.berkeley.edu. Basic sound file player with
	file open, command line capability, pause, stop, and MPEG
	properties. Also added play list and seeking functionality. Uses
	threads, so will NOT work on Win32s!

   Adapted from the Sounder sample application distributed on the
	Borland C++ CD */

/* Version 1.80 :

	Changes since last version:

   Seeking calls functions within the args class instead of directly reading
   or writing variables internal to the class.

   Tried to signal the thread procedures to terminate themselves so that
   memory will not be lost. However, this does not always work, so after
   a waiting period TerminateThread is called.

   Eliminated passing the crc to maplay, instead a dummy crc is used to read
   the header.

   Increased the maximum size of the playlist to 512 entries.

   Last edit : 02/01/97 */

/*	Version 1.81:

	MIDI and WAV files with filenames that contain spaces can now be loaded
	and played. This was due to a suggestion by Alexander Grigoriev. */

/* Version 1.90:

   - Added lots of options, including minimizing into the systray,
     playing a file immediately after it has been opened, prompting
     for another file after a file is played, exiting after being
     called on the commandline, setting the priority of the decoding
     process, and outputing to a WAV file. The settings are loaded
     and saved in the Windows registry. The options can be changed
     with the new options property sheet.
   - If maplay is already open, opening it again brings up the original
     window, and forwards the file from the command-line to the original
     instance.
   - Added ability to save and delete a file. MPEG files may be saved as
     copies or as decoded WAV files.
   - Right-clicking will display a convenient popup menu.
   - Added recently opened files and lists to the File menu, as well as
     the new popup menu.
   - If a bad file is opened, the state is reset to the initial state,
     except for the list status. I will fix that when I redo the
     playlists.
   - More file types are supported with MCI commands, including Sun audio
     files, Apple AIFF files, and Quicktime, AVI, and MPEG-1 movies (!).
     This relies on ActiveMovie being installed on the computer.
   - The MPEG..Properties dialog box was changed a bit, but it's still
     ugly. The playtime has been added.
   - Preliminary support for CD audio has been added, but it's still
     in the development phase. */

#define  STRICT
#include <windows.h>
#include <commctrl.h>
#include <commdlg.h>
#include <shlobj.h>

#include "all.h"
#include "crc.h"
#include "ibitstr.h"
#include "header.h"
#include "args.h"
#include "str_lib.h"
#include "rfu_list.h"
#include "playlist.h"
#include "w32_common.h"
#include "w32_util.h"
#include "w32_lview.h"
#include "w32_option.h"
//#include "midasdll.h"

#include "mp2win.h"
#include "mpw_msg.h"

#include <math.h>
//#include <stdio.h>

#define NUM_RFL_ENTRIES 4
#define REMOTE_THRESH   5

// Argument containers
MPEG_Args _maplay_args;
MPEG_Args *maplay_args = &_maplay_args;
MCI_Args  _mci_args;
MCI_Args  *mci_args    = &_mci_args;
Args *args;

// Copies of header and stream
Header     header_copy;
Ibitstream stream_copy;

// Thread functions
uint32 maplay(MPEG_Args *);
DWORD  mciplay(MCI_Args *);

// data initialized by first instance

char szAppName[20];

// Data that can be referenced throughout the
// program but not passed to other instances

char szName[MAX_FILENAME_LENGTH * 64]; // Name of sound file(s)
char FileTitle[MAX_PATH];

char SaveAsFilename[MAX_PATH] = "";	  // Name of PCM wave file to save as
char ListSaveFile[MAX_PATH] = "";
char TitleText[MAX_PATH + 16] = "";   // Title of window
char command[384];            	     // MCI command string
char scratch[1024];
char Buffer[1024];    	      	     // MCI response string

// CD Track Numbers
char this_track_str[8];
char next_track_str[8];

OPENFILENAME ofnSaveAs;
char SaveAsFilter[128];

unsigned int scroll_range = 0;
// int predecode=0;		// not implemented yet

// Dummy CRC pointer
Crc16 *crc;

// WaveOut handle and its address
HWAVEOUT hwo;
HWAVEOUT *phwo = &hwo;

// Threads
HANDLE MPEG_Thread = NULL;
HANDLE MCI_Thread  = NULL;
DWORD dwThreadId;

// Buttons in main window
HWND openbut, pausebut, stopbut, aboutbut;
HWND rewbut, ffbut, prevbut, nextbut;

// Track bar
int line_size;
HWND tracker;

// Status bar
int panes[4]= {68, 213, 273, -1};
char playlist_display[MAX_PATH];
HWND status_bar;

// Menus
HMENU mainmenu;
HMENU rfl_file_menu;
HMENU rfl_list_menu;

// Recent File Lists
RecentFileList *rfl_f;
RecentFileList *rfl_l;

HDC hdc;
DRAWITEMSTRUCT *dis;

// Function prototypes

int  PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
                    LPSTR lpszCmdLine, int cmdShow);
void InitApp(HINSTANCE hInstance, int cmdShow);

// old window procedure.  for subclassing
WNDPROC OldPrevButWndProc	= NULL;
WNDPROC OldNextButWndProc	= NULL;
WNDPROC OldPlayButWndProc	= NULL;

// new window procedure.
BOOL CALLBACK NewPrevButWndProc(HWND, UINT, WPARAM, LPARAM);
BOOL CALLBACK NewNextButWndProc(HWND, UINT, WPARAM, LPARAM);
BOOL CALLBACK NewPlayButWndProc(HWND, UINT, WPARAM, LPARAM);

// Mini-control window procedure
BOOL CALLBACK MiniWndProc(HWND hDlg, UINT message,
                          WPARAM wParam, LPARAM lParam);

// tooltip for the jump window
HWND tooltip_control;

// Helper procedures
BOOL AppFileOpen();
BOOL file_load(BOOL dir_inc);
void clear_sound();
void reset();
void ok_play();
void no_play();
void no_stop();
BOOL reinit_MPEG();
BOOL reinit_MCI();
void handle_bad_file(LPSTR filename, BOOL fix_list);
BOOL WaveP(LPSTR);
BOOL CDAP(LPSTR);
BOOL MPEG_play();
BOOL MCI_play();

void update_timewin(int num);
void DisplayTitle(LPSTR name, BOOL reset);

BOOL list_load();
void Init_List();
void no_list();

void set_list_display();
void List_Advance(BOOL list_play_after_load, BOOL user_hit);

// Displays the menu when the mouse is right-clicked
void DisplayContextMenu();

// Menu procedures
void file_open();
void file_save_as();
void file_delete();
void file_recent(RecentFileList *rfl, int pos);
void audio_play();
void audio_pause();
void audio_stop();
void audio_properties();
BOOL open_mini_win();
void help_web_site();
void about();
void leave();

LRESULT APIENTRY SounderWndProc(HWND,UINT,WPARAM,LPARAM);

#pragma argsused
int APIENTRY WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
		 					LPSTR lpszCmdLine, int cmdShow)
{
	 // Read the settings from the registry
    settings(FALSE);

    // Get string from resource with application name.
	 LoadString(hInstance, IDS_NAME, (LPSTR) szAppName, 20);

    if (!allow_multiple) {
		 HWND hWnd_old = FindWindow(szAppName, NULL);
		 if (hWnd_old) {

	    	if (lstrlen(lpszCmdLine) > 1)
	      	SendMessage(hWnd_old, WM_CMDLINEFILE, 0, (LPARAM) lpszCmdLine);

	  		ShowWindow(hWnd_old, SW_SHOWNORMAL);
	    	SetForegroundWindow(hWnd_old);
	      return 0;
		 }
    }

	 MSG   msg;

	 // Go init this application.
	 InitApp(hInstance, cmdShow);

	 if (lstrlen(lpszCmdLine)>1) {
		 CMDLINE = TRUE;
       SendMessage(hWnd, WM_CMDLINEFILE, 0, (LPARAM) lpszCmdLine);
	 }

	 // Get and dispatch messages for this applicaton.
	 while (GetMessage(&msg, NULL, 0, 0))
	 {
		  TranslateMessage(&msg);
		  DispatchMessage(&msg);
	 }
	 return msg.wParam;
}

void InitApp(HINSTANCE hInstance, int cmdShow)
{
	 WNDCLASS wcSoundClass;
    HMENU    file_menu;

	 // Define the window class for this application.
	 wcSoundClass.lpszClassName = szAppName;
	 wcSoundClass.hInstance     = hInstance;
	 wcSoundClass.lpfnWndProc   = SounderWndProc;
	 wcSoundClass.hCursor       = LoadCursor(NULL, IDC_ARROW);
	 wcSoundClass.hIcon         = LoadIcon(hInstance,
														MAKEINTRESOURCE(ICON_MP));
	 wcSoundClass.lpszMenuName  = MAKEINTRESOURCE(MENU_E);
	 wcSoundClass.hbrBackground = (HBRUSH) GetStockObject(BLACK_BRUSH);
	 wcSoundClass.style         = CS_HREDRAW | CS_VREDRAW;
	 wcSoundClass.cbClsExtra    = 0;
	 wcSoundClass.cbWndExtra    = 0;

	 // Register the class
	 RegisterClass(&wcSoundClass);

	 hInst = hInstance;       // save for use by window procs

	 InitCommonControls();

	 // Create applications main window.
	 hWnd = 	CreateWindowEx(
				      WS_EX_ACCEPTFILES | WS_EX_CONTROLPARENT,
						szAppName,         // window class name
						szAppName,         // window title
						WS_OVERLAPPED | WS_MINIMIZEBOX | WS_CAPTION | WS_SYSMENU,
                                     // type of window
						200,               // x  window location
						5,                 // y
						310,               // cx and size
						145,               // cy
						NULL,              // no parent for this window
						NULL,              // use the main menu
						hInstance,         // who created this window
						NULL);             // no parms to pass on


	 // Update display of main window.
	 mainmenu = GetMenu(hWnd);

	 ShowWindow(hWnd, cmdShow);
	 UpdateWindow(hWnd);

    // Set topness of the main window
    SetWindowPos(hWnd, topness, 0, 0, 0, 0,
                 SWP_NOMOVE | SWP_NOSIZE | SWP_SHOWWINDOW);

//    MIDASstartup();

    // Initialize playlist object
    fl.set_warn_window(hWnd);

    // Initialize the recent file used lists
    rfl_f = new RecentFileList(NUM_RFL_ENTRIES, RFL_FILE_TYPE);
    rfl_f->read_from_registry();
    rfl_l = new RecentFileList(NUM_RFL_ENTRIES, RFL_LIST_TYPE);
    rfl_l->read_from_registry();

    file_menu = GetSubMenu(mainmenu, 0);

    rfl_file_menu = GetSubMenu(file_menu, 5);
    rfl_f->build_menu(rfl_file_menu, hWnd);

    rfl_list_menu = GetSubMenu(file_menu, 6);
    rfl_l->build_menu(rfl_list_menu, hWnd);

    // Setup open file dialog box structure
	 ofnOpen.lStructSize       = sizeof(OPENFILENAME);
    ofnOpen.hwndOwner         = hWnd;
    ofnOpen.hInstance         = 0;
    ofnOpen.lpstrFilter       = (LPSTR) OpenFilter;
    ofnOpen.lpstrCustomFilter = NULL;
    ofnOpen.nMaxCustFilter    = 0;
    ofnOpen.nFilterIndex      = 1;
    ofnOpen.lpstrFile         = (LPSTR)szName;
    ofnOpen.nMaxFile          = sizeof(szName);
    ofnOpen.lpstrFileTitle    = FileTitle;
    ofnOpen.nMaxFileTitle     = sizeof(FileTitle);
    ofnOpen.lpstrInitialDir   = NULL;
    ofnOpen.lpstrTitle        = "Open Media File(s)";
    ofnOpen.Flags             = OFN_FILEMUSTEXIST | OFN_PATHMUSTEXIST |
  	     				   		     OFN_HIDEREADONLY  | OFN_ALLOWMULTISELECT |
                                OFN_EXPLORER; // fix this
    ofnOpen.nFileOffset       = 0;
    ofnOpen.nFileExtension    = 0;
    ofnOpen.lpstrDefExt       = "MP*";
    ofnOpen.lCustData         = 0;
    ofnOpen.lpfnHook          = NULL;
    ofnOpen.lpTemplateName    = NULL;

    // Setup output to wavefile dialog box structure
	 ofnOutput.lStructSize       = sizeof(OPENFILENAME);
    ofnOutput.hwndOwner         = hWnd;
    ofnOutput.hInstance         = 0;
    ofnOutput.lpstrFilter       = (LPSTR) OutputFilter;
    ofnOutput.lpstrCustomFilter = NULL;
    ofnOutput.nMaxCustFilter    = 0;
    ofnOutput.nFilterIndex      = 1;
    ofnOutput.lpstrFile         = (LPSTR)OutputFilename;
    ofnOutput.nMaxFile          = sizeof(OutputFilename);
    ofnOutput.lpstrFileTitle    = NULL;
    ofnOutput.nMaxFileTitle     = 0;
    ofnOutput.lpstrInitialDir   = NULL;
    ofnOutput.lpstrTitle        = "Decode to PCM WAV File";
    ofnOutput.Flags             = OFN_PATHMUSTEXIST | OFN_OVERWRITEPROMPT |
  	     				   		     	 OFN_HIDEREADONLY  | OFN_EXPLORER; // fix this
    ofnOutput.nFileOffset       = 0;
    ofnOutput.nFileExtension    = 0;
    ofnOutput.lpstrDefExt       = "WAV";
    ofnOutput.lCustData         = 0;
    ofnOutput.lpfnHook          = NULL;
    ofnOutput.lpTemplateName    = NULL;

    // Setup Save As dialog box structure (not complete)
	 ofnSaveAs.lStructSize       = sizeof(OPENFILENAME);
    ofnSaveAs.hwndOwner         = hWnd;
    ofnSaveAs.hInstance         = 0;
    ofnSaveAs.lpstrFilter       = NULL;
    ofnSaveAs.lpstrCustomFilter = NULL;
    ofnSaveAs.nMaxCustFilter    = 0;
    ofnSaveAs.nFilterIndex      = 1;
    ofnSaveAs.lpstrFile         = (LPSTR)SaveAsFilename;
    ofnSaveAs.nMaxFile          = sizeof(SaveAsFilename);
    ofnSaveAs.lpstrFileTitle    = NULL;
    ofnSaveAs.nMaxFileTitle     = 0;
    ofnSaveAs.lpstrInitialDir   = NULL;
    ofnSaveAs.lpstrTitle        = "Save As";
    ofnSaveAs.Flags             = OFN_PATHMUSTEXIST | OFN_OVERWRITEPROMPT |
  	     				   		     	 OFN_HIDEREADONLY  | OFN_EXPLORER; // fix this
    ofnSaveAs.nFileOffset       = 0;
    ofnSaveAs.nFileExtension    = 0;
    ofnSaveAs.lpstrDefExt       = NULL;
    ofnSaveAs.lCustData         = 0;
    ofnSaveAs.lpfnHook          = NULL;
    ofnSaveAs.lpTemplateName    = NULL;

    // Setup list edit dialog box
	 ofnListEdit.lStructSize       = sizeof(OPENFILENAME);
    ofnListEdit.hwndOwner         = hWnd;
    ofnListEdit.hInstance         = hInst;
    ofnListEdit.lpstrFilter       = (LPSTR) OpenFilter;
    ofnListEdit.lpstrCustomFilter = NULL;
    ofnListEdit.nMaxCustFilter    = 0;
    ofnListEdit.nFilterIndex      = 1;
    ofnListEdit.lpstrFile         = (LPSTR) ListEditAddName;
    ofnListEdit.nMaxFile          = sizeof(ListEditAddName);
    ofnListEdit.lpstrFileTitle    = ListEditFileTitle;
    ofnListEdit.nMaxFileTitle     = sizeof(ListEditFileTitle);
    ofnListEdit.lpstrInitialDir   = NULL;
    ofnListEdit.lpstrTitle        = "Edit Playlist";
    ofnListEdit.Flags             = OFN_FILEMUSTEXIST  | OFN_PATHMUSTEXIST |
	  	     				   		      OFN_HIDEREADONLY   | OFN_ALLOWMULTISELECT |
                                    OFN_ENABLETEMPLATE | OFN_ENABLEHOOK |
                                    OFN_EXPLORER;
    ofnListEdit.nFileOffset       = 0;
    ofnListEdit.nFileExtension    = 0;
    ofnListEdit.lpstrDefExt       = "MP*";
    ofnListEdit.lCustData         = 0;
    ofnListEdit.lpfnHook          = (LPOFNHOOKPROC) ListEditProc;
    ofnListEdit.lpTemplateName    = MAKEINTRESOURCE(IDD_LISTDLG);

    // Setup List Save As dialog box
	 ofnListSaveAs.lStructSize       = sizeof(OPENFILENAME);
    ofnListSaveAs.hwndOwner         = hWnd;
    ofnListSaveAs.hInstance         = hInst;
    ofnListSaveAs.lpstrFilter       = (LPSTR) ListSaveAsFilter;
    ofnListSaveAs.lpstrCustomFilter = NULL;
    ofnListSaveAs.nMaxCustFilter    = 0;
    ofnListSaveAs.nFilterIndex      = 1;
    ofnListSaveAs.lpstrFile         = (LPSTR) ListSaveFile;
    ofnListSaveAs.nMaxFile          = sizeof(ListSaveFile);
    ofnListSaveAs.lpstrFileTitle    = NULL;
    ofnListSaveAs.nMaxFileTitle     = 0;
    ofnListSaveAs.lpstrInitialDir   = NULL;
    ofnListSaveAs.lpstrTitle        = "Save List As";
    ofnListSaveAs.Flags             = OFN_PATHMUSTEXIST | OFN_OVERWRITEPROMPT |
	  	     				   		     	  OFN_HIDEREADONLY  | OFN_EXPLORER;
    ofnListSaveAs.nFileOffset       = 0;
    ofnListSaveAs.nFileExtension    = 0;
    ofnListSaveAs.lpstrDefExt       = "LST";
    ofnListSaveAs.lCustData         = 0;
    ofnListSaveAs.lpfnHook          = NULL;
    ofnListSaveAs.lpTemplateName    = NULL;

    InitIconList();
    setup_lview_ofn();
}

BOOL AppFileOpen()
{
  openbox_open = TRUE;
  if (!GetOpenFileName((LPOPENFILENAME) &ofnOpen))
  {
	 if (CommDlgExtendedError() != 0) // 0 value means user selected Cancel
		MessageBox(hWnd, "Could not open file.", "Warning", MB_OK | MB_ICONSTOP);

    openbox_open = FALSE;
	 return FALSE;
  }

  openbox_open = FALSE;
  return TRUE;
}

void clear_sound()
{
	scrolling = FALSE;
   playing = FALSE;

	if (MPEG) {

   	if (MPEG_Thread) {

         SetThreadPriority(MPEG_Thread, THREAD_PRIORITY_HIGHEST);

			if (paused) {
				paused      = FALSE;
            waveOutReset(*phwo);
			   waveOutRestart(*phwo);
				ResumeThread(MPEG_Thread);
 			}

			WaitForSingleObject(maplay_args->mutex, INFINITE);

         if (!maplay_args->done) {
				maplay_args->stop = true;

				ReleaseMutex(maplay_args->mutex);

	         if (WaitForSingleObject(MPEG_Thread, 5000) == WAIT_TIMEOUT) {
	         	TerminateThread(MPEG_Thread, 0);
			  		waveOutReset(*phwo);
					waveOutClose(*phwo);
            }
	      } else {
				ReleaseMutex(maplay_args->mutex);
         }

         CloseHandle(MPEG_Thread);
      }

		MPEG_Thread = NULL; // ensures next call to clear_sound() will not
      					     // attempt to kill thread

		delete maplay_args->stream;
		maplay_args->stream = NULL;

		delete maplay_args->MPEGheader;
      maplay_args->MPEGheader = NULL;

      if (maplay_args->mutex) {
	      CloseHandle(maplay_args->mutex);
         maplay_args->mutex = NULL;
      }

	} else {

   	if (MCI_Thread) {

         SetThreadPriority(MCI_Thread, THREAD_PRIORITY_HIGHEST);

			if (paused) {
				paused      = FALSE;
				ResumeThread(MCI_Thread);
 			}

      	WaitForSingleObject(mci_args->mutex, INFINITE);

			mci_args->stop = true;
         ReleaseMutex(mci_args->mutex);

	      if(WaitForSingleObject(MCI_Thread, 20000) == WAIT_TIMEOUT)
	        	TerminateThread(MCI_Thread, 0);

         CloseHandle(MCI_Thread);
      }

		MCI_Thread = NULL; // ensures next call to clear_sound() will not
                         // attempt to kill thread

		mciSendString((LPSTR)"close sounder wait", scratch, 1024, NULL);

      if (mci_args->mutex) {
	      CloseHandle(mci_args->mutex);
         mci_args->mutex = NULL;
      }
	}

	SendMessage(status_bar, SB_SETTEXT, 0, (LPARAM) "Stopped");
	SendMessage(status_bar, SB_SETTEXT, 2, (LPARAM) "000:00");
}

void reset()
{
	clear_sound();

   MPEG      = FALSE;
   save_mode = FALSE;
   playing   = FALSE;
   paused    = FALSE;
   scrolling = FALSE;
	scroll_range = 0;
   can_play  = FALSE;
   can_pause = FALSE;
   can_stop  = FALSE;

   szName[0] = '\0';

   DisplayTitle(NULL, TRUE);

	EnableMenuItem(mainmenu,CM_AUDIOPLAY, MF_GRAYED);
	EnableWindow(playbut, FALSE);
 	EnableMenuItem(mainmenu,CM_AUDIOPAUSE, MF_GRAYED);
	EnableWindow(pausebut, FALSE);
	EnableMenuItem(mainmenu,CM_AUDIOSTOP, MF_GRAYED);
	EnableWindow(stopbut, FALSE);

   SendMessage(tracker, TBM_CLEARTICS, FALSE, 0);
	SendMessage(tracker, TBM_SETPOS, TRUE, 0);
	EnableWindow(tracker, FALSE);

   EnableMenuItem(mainmenu,CM_FILEDELETE,  MF_GRAYED);
   EnableMenuItem(mainmenu,CM_FILESAVE_AS, MF_GRAYED);
	EnableMenuItem(mainmenu,CM_AUDIOREPEAT, MF_GRAYED);
	EnableWindow(rewbut, FALSE);
	EnableMenuItem(mainmenu, CM_AUDIOREWIND, MF_GRAYED);
	EnableWindow(ffbut, FALSE);
	EnableMenuItem(mainmenu, CM_AUDIOFAST_FORWARD, MF_GRAYED);
   EnableMenuItem(mainmenu, CM_MPEGPROPERTIES, MF_GRAYED);

	SendMessage(status_bar, SB_SETTEXT, 0, (LPARAM) "No file");
}

void ok_play()
{
	can_play  = TRUE;
	can_pause = FALSE;
	playing   = FALSE;
	EnableMenuItem(mainmenu,CM_AUDIOPLAY, MF_ENABLED);
	EnableWindow(playbut, TRUE);
	EnableMenuItem(mainmenu,CM_AUDIOPAUSE, MF_GRAYED);
	EnableWindow(pausebut, FALSE);
}

void no_play()
{
	can_play  = FALSE;
	can_pause = TRUE;
	can_stop  = TRUE;
	EnableMenuItem(mainmenu,CM_AUDIOPLAY, MF_GRAYED);
	EnableWindow(playbut, FALSE); 
	EnableMenuItem(mainmenu,CM_AUDIOPAUSE, MF_ENABLED);
	EnableWindow(pausebut, TRUE);
	EnableMenuItem(mainmenu,CM_AUDIOSTOP, MF_ENABLED);
	EnableWindow(stopbut, TRUE);
}

void no_stop()
{
	can_stop = FALSE;
	EnableMenuItem(mainmenu,CM_AUDIOSTOP, MF_GRAYED);
	EnableWindow(stopbut, FALSE);
}

BOOL reinit_MPEG()
{
	MPEG = TRUE;
	args = maplay_args;

	args->hWnd  = hWnd;
	args->mutex = CreateMutex(NULL, FALSE, "m");
   args->stop  = false;
   args->done  = false;

	args->position_change  = false;
   args->desired_position = 0;

	maplay_args->stream      = new Ibitstream(szName);
	maplay_args->MPEGheader  = new Header;
	maplay_args->phwo        = phwo;
	maplay_args->which_c     = mode;
   maplay_args->stdout_mode = false;
   maplay_args->use_own_scalefactor = use_own_scalefac;
   maplay_args->scalefactor = scalefactor;

   if (save_mode) {
	   maplay_args->output_mode = O_WAVEFILE;
   } else {
	   maplay_args->output_mode = output_mode;
   }

   if (save_mode) {
		lstrcpy(maplay_args->output_filename, SaveAsFilename);
   } else if (output_mode == O_WAVEFILE) {
	   lstrcpy(maplay_args->output_filename, OutputFilename);
   }

	if (!maplay_args->MPEGheader->read_header(maplay_args->stream, &crc)){
		MessageBox(hWnd, "No header found!",
		"File format error", MB_OK | MB_ICONSTOP);
		return FALSE;
	}

   header_copy = *(maplay_args->MPEGheader);
   stream_copy = *(maplay_args->stream);

	scroll_range = maplay_args->MPEGheader->min_number_of_frames(maplay_args->stream);
	line_size = scroll_range >> 3;

   EnableMenuItem(mainmenu, CM_MPEGPROPERTIES, MF_ENABLED);

	return TRUE;
}

BOOL reinit_MCI()
{
  	DWORD error;
   int length;

	MPEG   = FALSE;
   CDMode = CDAP(szName);

	EnableMenuItem(mainmenu,CM_MPEGPROPERTIES, MF_GRAYED);

   if (CDMode) {

      int this_track = CDTrack_number(this_track_str, next_track_str, szName);

      lstrcpy(command, "open cdaudio alias sounder shareable wait");
    	error = mciSendString ((LPSTR)command, Buffer, 1024, NULL);

      if (error != 0) {
         report_mci_error(hWnd, error, "Error opening CD audio device");
         return FALSE;
      }

      error = mciSendString("status sounder number of tracks wait",
                            Buffer, 1024, NULL);

      if (error != 0) {
         report_mci_error(hWnd, error, "Error getting number of CD tracks");
         return FALSE;
      }

      final_track = (BOOL) (atoi(Buffer) == this_track);

      error = mciSendString("set sounder time format ms wait", Buffer,
                            1024, NULL);

      if (error != 0) {
         report_mci_error(hWnd, error, "Error setting CD time format to ms");
         return FALSE;
      }

      lstrcpy(command, "status sounder length track ");
      lstrcat(command, this_track_str);
      lstrcat(command, " wait");
      error = mciSendString((LPSTR)command, Buffer, 1024, NULL);

      if (error != 0) {
         report_mci_error(hWnd, error, "Error getting length of track");
         return FALSE;
      }

      error = mciSendString("set sounder time format tmsf", scratch,
                            1024, NULL);

      if (error != 0) {
         report_mci_error(hWnd, error, "Error setting time format to tmsf");
         return FALSE;
      }

   } else {
		lstrcpy(command, "open \"");
		lstrcat(command, szName);
      lstrcat(command, "\" alias sounder wait");

      error = mciSendString ((LPSTR)command, Buffer, 1024, NULL);

      if (error != 0) {
         report_mci_error(hWnd, error, "Error opening multimedia file");
         return FALSE;
      }

		error = mciSendString("set sounder time format ms wait", Buffer,
                            1024, NULL);

      if (error != 0) {
         report_mci_error(hWnd, error, "Error setting time format to ms");
         return FALSE;
      }

		error = mciSendString("status sounder length wait", Buffer,
                            1024, NULL);

      if (error != 0) {
         report_mci_error(hWnd, error,
                          "Error getting length of multimedia file");
         return FALSE;
      }
   }

   length       = atoi(Buffer);

	scroll_range = length;
	line_size    = scroll_range >> 3;

   args         = mci_args;

 	args->hWnd  = hWnd;
   args->mutex = CreateMutex(NULL, FALSE, "m");
   args->stop  = false;
   args->done  = false;

	args->position_change  = false;
	args->desired_position = 0;

	mci_args->playing = false;
   mci_args->CDMode  = CDMode;

   if (CDMode) {
   	lstrcpy(mci_args->this_track_str, this_track_str);
		lstrcpy(mci_args->next_track_str, next_track_str);
      mci_args->final_track = final_track;
      mci_args->length = length;
   }

	return TRUE;
}

BOOL WaveP(LPSTR t)
// Check if the file is a format the Windows multimedia system supports.
{
	return(b_strcmpi(t + lstrlen(t)-3, "WAV") ||
			 b_strcmpi(t + lstrlen(t)-3, "MID") ||
			 b_strcmpi(t + lstrlen(t)-3, "RMI") ||
			 b_strcmpi(t + lstrlen(t)-3, "CDA") ||
          b_strcmpi(t + lstrlen(t)-2, "AU")  ||
          b_strcmpi(t + lstrlen(t)-3, "AIF") ||
          b_strcmpi(t + lstrlen(t)-4, "AIFF")||
          b_strcmpi(t + lstrlen(t)-3, "AVI") ||
          b_strcmpi(t + lstrlen(t)-3, "MOV") ||
          b_strcmpi(t + lstrlen(t)-3, "QT")  ||
          b_strcmpi(t + lstrlen(t)-3, "DAT") ||
          b_strcmpi(t + lstrlen(t)-3, "MPG") ||
			 b_strcmpi(t + lstrlen(t)-4, "MPEG")||
          b_strcmpi(t + lstrlen(t)-3, "MPV"));
}

BOOL CDAP(LPSTR t)
{
	return b_strcmpi(t + lstrlen(t)-3, "CDA");
}

void DisplayTitle(LPSTR name, BOOL reset)
{

	LoadString(hInst, IDS_NAME, (LPSTR) TitleText, MAX_PATH+16);

   if (!reset) {
		lstrcat(TitleText, " - [");

      if (ListMode && pl.get_list_name()) {
      	lstrcat(TitleText, pl.get_list_name());
      } else {
			if (name)
				lstrcat(TitleText, Proper_Filename(name));
			else
				lstrcat(TitleText, FileTitle);
   	} // ListMode

		lstrcat(TitleText, "]");
   }

   SetWindowText(hWnd, (LPCTSTR) TitleText);
}

BOOL MPEG_play()
// Assumes stream, MPEGheader are already initialized
{
	BOOL t;
	MPEG_Thread = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE) maplay,
										maplay_args, 0, &dwThreadId);
	t = SetThreadPriority(MPEG_Thread, priority);

   playing = TRUE;
	return t;
}

BOOL MCI_play()
{
	mci_args->playing = true;

	MCI_Thread = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE) mciplay,
									  mci_args, 0, &dwThreadId);

   playing = TRUE;
	return (MCI_Thread != NULL);
}

void set_list_display()
{
	char *img_n, *lyr_n;

   pl.get_track_display_at(playlist_display, pl.get_current_track());
	SendMessage(status_bar, SB_SETTEXT, 1, (LPARAM) playlist_display);

   // hilite the current song in the playlist window
   PlayListWndHiliteCurrent();

   img_n = pl.get_current_image_name();
   lyr_n = pl.get_current_lyric_name();

   if (display_images && img_n)
   	launch_file(img_n, hWnd, iv_cmdline,
                  custom_img_viewer ? civ_name : NULL);

   if (display_lyric_tf && lyr_n)
   	launch_file(lyr_n, hWnd);
}

void handle_bad_file(LPSTR filename, BOOL fix_list)
{
   reset();

   rfl_f->remove(filename);
   rfl_f->build_menu(rfl_file_menu, hWnd);

   if (ListMode) {
     	pl.remove_all_occurences(filename);

      if (pl.get_number_of_entries() <= 0) {
        	no_list();
      } else {
	      UpdatePlayListWnd();
      	if (fix_list)
	         List_Play_Current(FALSE);
      }
   }
}

BOOL File_Init(BOOL fix_list)
{
	BOOL result = (WaveP(szName) ? reinit_MCI() : reinit_MPEG());

   if (!result) {
   	handle_bad_file(szName, fix_list);
   	return FALSE;
   }

   EnableMenuItem(mainmenu,CM_FILEDELETE,  MF_ENABLED);
   EnableMenuItem(mainmenu,CM_FILESAVE_AS, MF_ENABLED);

	EnableMenuItem(mainmenu,CM_AUDIOREPEAT, MF_ENABLED);
	EnableWindow(rewbut, TRUE);
	EnableMenuItem(mainmenu, CM_AUDIOREWIND, MF_ENABLED);
	EnableWindow(ffbut, TRUE);
	EnableMenuItem(mainmenu, CM_AUDIOFAST_FORWARD, MF_ENABLED);

	SendMessage(tracker, TBM_SETRANGEMIN, FALSE, 0);
	SendMessage(tracker, TBM_SETRANGEMAX, FALSE, scroll_range);
	SendMessage(tracker, TBM_SETPOS, TRUE, 0);
	SendMessage(tracker, TBM_SETTICFREQ, scroll_range >> 4, 1);
	SendMessage(tracker, TBM_SETLINESIZE, 0, line_size);
	EnableWindow(tracker, TRUE);

	SendMessage(status_bar, SB_SETTEXT, 0, (LPARAM) "Ready");
	SendMessage(status_bar, SB_SETTEXT, 2, (LPARAM) "000:00");

   return TRUE;
}

void List_Advance(BOOL list_play_after_load, BOOL user_hit)
{
	if (ListMode) {
		if (pl.last_song()) {
			if (CMDLINE && cmdline_exit && !user_hit) {
				leave();
	      } else {
	      	pl.set_current_track(0);
	         List_Play_Current((Repeat || user_hit) && list_play_after_load);
	      }
	   } else {
	   	pl.goto_next();
	   	List_Play_Current(list_play_after_load);
	   }
   }
}

void List_Play_Current(BOOL list_play_after_load)
{
	BOOL success = FALSE;

	CMDLINE = FALSE;

   if (ListMode) {
		clear_sound();

	   while (!success && (pl.get_number_of_entries() > 0)) {
			lstrcpy(szName, pl.get_current_song_name());
	   	success = File_Init(FALSE);
		}

		ok_play();
		no_stop();

	   set_list_display();

	  	if (list_play_after_load)
			audio_play();
   }
}

BOOL list_load()
{
	clear_sound();
	Init_List();

	set_list_display();

	lstrcpy(szName, pl.get_current_song_name());
   if (!File_Init(TRUE)) {
		return FALSE;
   }

	ok_play();
	no_stop();

   return TRUE;
}

BOOL cd_list(BOOL hybrid)
{
   char num_track_str[16];
   DWORD error;
   int tracks;

   clear_sound();

 	error = mciSendString ("open cdaudio alias cl shareable wait",
                          Buffer, 1024, NULL);

   if (error != 0) {
      report_mci_error(hWnd, error, "Error opening CD audio device");
      return FALSE;
   }

   error = mciSendString("status cl number of tracks wait",
                         num_track_str, 16, NULL);

   if (error != 0) {
      report_mci_error(hWnd, error, "Error getting number of CD tracks");
      return FALSE;
   }


   mciSendString ("close cl wait", Buffer, 1024, NULL);

   tracks = atoi(num_track_str);

   if (tracks > 0) {

      fl.reset();

      char track_str[32];
      char num_str[16];

      int i = hybrid ? 1 : 0;

      for (; i<tracks; i++) {
          lstrcpy(track_str, "Track");
          itoa(i+1, num_str, 10);
          lstrcat(track_str, num_str);
          lstrcat(track_str, ".cda");

          fl.append_to_list(track_str);
      }
      fl.set_list_name("CD List");
      pl = fl;
      return list_load();

   } else {
      MessageBox(hWnd, "CD has no audio tracks",
                 "Error opening CD list", MB_ICONEXCLAMATION | MB_OK);
      return FALSE;
   }
}

void Init_List()
{
	char *list_name = pl.get_list_name();

	ListMode = TRUE;

   DisplayTitle(list_name, (BOOL) (list_name == NULL));

	EnableMenuItem(mainmenu,CM_LISTEDIT, MF_ENABLED);
	EnableMenuItem(mainmenu,CM_LISTSAVE, MF_ENABLED);
	EnableMenuItem(mainmenu,CM_LISTRAND, MF_ENABLED);
	EnableMenuItem(mainmenu,CM_LISTPREV, MF_ENABLED);
	EnableWindow(prevbut, TRUE);
	EnableMenuItem(mainmenu,CM_LISTREPEAT, MF_ENABLED);
	EnableMenuItem(mainmenu,CM_LISTNEXT, MF_ENABLED);
	EnableWindow(nextbut, TRUE);

	EnableMenuItem(mainmenu,CM_LISTJUMP, MF_ENABLED);
   UpdatePlayListWnd();
}

void no_list()
{
	pl.reset();

	ListMode = FALSE;
	EnableMenuItem(mainmenu,CM_LISTEDIT, MF_GRAYED);
	EnableMenuItem(mainmenu,CM_LISTSAVE, MF_GRAYED);
	EnableMenuItem(mainmenu,CM_LISTRAND, MF_GRAYED);
	EnableMenuItem(mainmenu,CM_LISTPREV, MF_GRAYED);
	EnableWindow(prevbut, FALSE);
	EnableMenuItem(mainmenu,CM_LISTREPEAT, MF_GRAYED);
	EnableMenuItem(mainmenu,CM_LISTNEXT, MF_GRAYED);
	EnableWindow(nextbut, FALSE);
	EnableMenuItem(mainmenu,CM_LISTJUMP, MF_GRAYED);

	SendMessage(status_bar,SB_SETTEXT, 1, NULL);
}

BOOL list_save(FileList *plsfl, HWND hWnd0)
{
   if (ListMode) {
     ofnListSaveAs.hwndOwner = hWnd0;

     if (!GetSaveFileName((LPOPENFILENAME) &ofnListSaveAs)) {
   	  if (CommDlgExtendedError() != 0) {
		     MessageBox(hWnd0, "Could not save list.", "Warning",
   	         		 MB_OK | MB_ICONSTOP);
        }
        return FALSE;
     } else {
     	  if (plsfl->save(ListSaveFile)) {
           rfl_l->push(ListSaveFile);
		     rfl_l->build_menu(rfl_list_menu, hWnd);
        } else {
		    MessageBox(hWnd0, "Could not save list.", "Warning",
   	        		   MB_OK | MB_ICONSTOP);
      	 return FALSE;
        }
     }
     return TRUE;
   }
   return FALSE;
}

void update_timewin(int num)
{
	if (MPEG) {
		SendMessage(status_bar, SB_SETTEXT, 2,
					   (LPARAM) time_string(
                   (int) (num * maplay_args->MPEGheader->ms_per_frame()),
                    scratch));
	} else {
		SendMessage(status_bar, SB_SETTEXT, 2,
					   (LPARAM) time_string(num, scratch));
   }
}

BOOL file_load(BOOL dir_inc)
{
	BOOL valid_list = FALSE;

   no_list();

   if (multi_file_select(szName)) {

		valid_list = fl.read_from_string(szName, dir_inc);

      if (valid_list) {
      	ListMode = TRUE;
      	pl = fl;
         if (rand_list_on_open)
            pl.randomize_order();

      	DisplayTitle(NULL, TRUE);
      } else {
      	reset();
      	return FALSE;
      }

	} else if (is_list(szName)) {

		valid_list = fl.read_from_file(szName, false);

      if (valid_list) {
      	ListMode = TRUE;
      	pl = fl;
         if (rand_list_on_open)
            pl.randomize_order();

         DisplayTitle(Proper_Filename(szName), FALSE);

	      rfl_l->push(szName);
	  	   rfl_l->build_menu(rfl_list_menu, hWnd);
      } else {
      	reset();
   		return FALSE;
      }
   }

   if (valid_list) {

      if (!list_load())
      	return FALSE;

   } else {

		clear_sound();

     	if (!File_Init(TRUE)) {
			return FALSE;
      }

      rfl_f->push(szName);
      rfl_f->build_menu(rfl_file_menu, hWnd);

      DisplayTitle(Proper_Filename(szName), FALSE);

	   ok_play();
	   no_stop();
      // this would destroy the playlist window if it exists
      UpdatePlayListWnd();

   } // if (valid_list)

   return TRUE;
}


void file_open()
{
	CMDLINE = FALSE;

	if (AppFileOpen()) {
      if (file_load(TRUE) && play_after_open)
	   	audio_play();
	}
}

void file_save_as()
{
   CMDLINE = FALSE;
   char SaveAsFilter[256] = "\0";
   LPSTR temp_str = szName + lstrlen(szName) - 1;

   while ((*temp_str != '.') && (temp_str > szName))
   	temp_str--;

   if (temp_str != szName) {

   	temp_str++;

      if (b_strcmpi(temp_str, "MP1") || b_strcmpi(temp_str, "MP2") ||
          b_strcmpi(temp_str, "MP3") || b_strcmpi(temp_str, "MPP")) {
 	      strcpy_2n(SaveAsFilter,
		             "MPEG Audio Files (.mp1, .mp2, .mp3, .mpp)\0"
                   "*.mp1;*.mp2;*.mp3;*.mpp\0"
         		    "PCM Wave Files (.wav)\0*.wav\0");
      } else if (b_strcmpi(temp_str, "SND")) {
      	strcpy_2n(SaveAsFilter,
					    "DALET Sound Files (.snd)\0*.snd\0"
         		    "PCM Wave Files (.wav)\0*.wav\0");
      } else if (b_strcmpi(temp_str, "WAV")) {
      	strcpy_2n(SaveAsFilter, "PCM Wave Files (*.wav)\0*.wav\0");
      } else if (b_strcmpi(temp_str, "MID") || b_strcmpi(temp_str, "RMI")) {
			strcpy_2n(SaveAsFilter, "MIDI Sequencer (.mid, .rmi)\0*.mid;*.rmi\0");
      } else if (b_strcmpi(temp_str, "AU")) {
      	strcpy_2n(SaveAsFilter, "Sun/NeXT Sound (.au)\0*.au\0");
      } else if (b_strcmpi(temp_str, "AIFF") || b_strcmpi(temp_str, "AIF")) {
			strcpy_2n(SaveAsFilter, "Apple AIFF (.aif, .aiff)\0*.aif;*.aiff\0");
      } else if (b_strcmpi(temp_str, "MOV") || b_strcmpi(temp_str, "QT")) {
	      strcpy_2n(SaveAsFilter, "Quicktime Movie (.mov, .qt)\0*.mov;*.qt\0");
      } else if (b_strcmpi(temp_str, "DAT")) {
	      strcpy_2n(SaveAsFilter, "Video CD (.dat)\0*.dat\0");
      } else if (b_strcmpi(temp_str, "MPG") || b_strcmpi(temp_str, "MPEG")
      			  || b_strcmpi(temp_str, "MPV")) {
	      strcpy_2n(SaveAsFilter, "MPEG Video (.mpg, .mpeg, .mpv)\0"
         							 "*.mpg;*.mpeg;*.mpv\0");
      } else if (b_strcmpi(temp_str, "AVI")) {
	      strcpy_2n(SaveAsFilter, "Video for Windows (.avi)\0*.avi\0");
      }
   }

   strcat_2n(SaveAsFilter, "All Files (*.*)\0*.*\0");

   ofnSaveAs.lpstrFilter = (LPSTR) SaveAsFilter;
   ofnSaveAs.lpstrDefExt = (LPSTR) temp_str;

   if (!GetSaveFileName((LPOPENFILENAME) &ofnSaveAs)) {
	  if (CommDlgExtendedError() != 0) // 0 value means user selected Cancel
        MessageBox(hWnd, "Could not open file.", "Warning", MB_OK | MB_ICONSTOP);

	  return;
   }

   if (MPEG && b_strcmpi(SaveAsFilename + lstrlen(SaveAsFilename) - 3, "WAV")) {

   	save_mode = TRUE;

      file_load(FALSE);
      audio_play();

	} else {

		BOOL was_paused = FALSE;

   	save_mode = FALSE;

      if (playing && !paused) {
      	was_paused = TRUE;
      	audio_pause();
      }

	   if (!CopyFile(szName, SaveAsFilename, FALSE)) {
	   	MessageBox(hWnd, "Error copying file", "Save error",
	      			  MB_OK | MB_ICONSTOP);
	      return;
   	}

      if (was_paused)
      	audio_play();
   }
}

void file_delete()
{
	CMDLINE = FALSE;

   if (szName[0] == '\0') return;

	lstrcpy(scratch, "Are you sure you want to delete '");
   lstrcat(scratch, szName);
   lstrcat(scratch, "'?");

   if (MessageBox(hWnd, scratch, "Confirm File Delete",
		            MB_YESNO | MB_ICONQUESTION) == IDYES) {

   /*   LPITEMIDLIST pidl;
      char path[MAX_PATH];

		if (SHGetSpecialFolderLocation(hWnd, CSIDL_BITBUCKET, &pidl)
          != NOERROR) {
          MessageBox(hWnd, "Could not find location of Recycle Bin.",
          				"File Delete Error", MB_ICONEXCLAMATION | MB_OK);
          return;
      }

      if (!SHGetPathFromIDList((LPCITEMIDLIST) pidl, path)) {
          MessageBox(hWnd, "Could not convert location of Recycle Bin "
          				"to a pathname.",
          				"File Delete Error", MB_ICONEXCLAMATION | MB_OK);
          return;
      }

		lstrcat(path, Proper_Filename(szName));
   	if(!MoveFileEx(szName, path, MOVEFILE_COPY_ALLOWED)) {
      	 MessageBox(hWnd, "Could not move file to the Recycle Bin.",
          				"File Delete Error", MB_ICONEXCLAMATION | MB_OK);
      } */
      char del_filename[MAX_PATH];
      lstrcpy(del_filename, szName);

      reset();
      if (!DeleteFile(del_filename)) {
      	 MessageBox(hWnd, "Could not delete file.",
          				"File Delete Error", MB_ICONEXCLAMATION | MB_OK);
          return;
      }

      handle_bad_file(del_filename, TRUE);
   }
}

void file_recent(RecentFileList *rfl, int pos)
{
	CMDLINE = FALSE;

	rfl->get_name_at_pos(szName, pos);

   stradd_2n(szName);

   if (file_load(FALSE) && play_after_open)
   	audio_play();
}

void leave()
{
	clear_sound();
   no_list();

   rfl_f->write_to_registry();
   delete rfl_f;

   rfl_l->write_to_registry();
   delete rfl_l;

   if (systray_mode)
      TrayMessage(NIM_DELETE, hWnd);

   CleanupPlayListWnd();
	DestroyWindow(hWnd);
}

void audio_play()
{
	if (can_play) {
		no_play();

		if (MPEG) {
			if (paused && MPEG_Thread) {
         	playing = TRUE;
				paused  = FALSE;
				waveOutRestart(*phwo);
				ResumeThread(MPEG_Thread);
			} else {
				MPEG_play();
         }
		} else {
			if (paused && MCI_Thread) {
         	playing = TRUE;
				paused  = FALSE;
				mci_args->playing = true;
			   if (CDMode) {
			      lstrcpy(command, "play sounder");
               if (!final_track) {
	               lstrcat(command, " to ");
				      lstrcat(command, next_track_str);
               }
			      lstrcat(command, " notify");
			      mciSendString ((LPSTR)command, Buffer, 1024, hWnd);
				} else {
			   	mciSendString ((LPSTR)"play sounder notify", Buffer, 1024, hWnd);
			   }
				ResumeThread(MCI_Thread);
			} else {
				MCI_play();
         }
		}
      if (save_mode) {
	      SendMessage(status_bar,SB_SETTEXT, 0,(LPARAM) "Saving...");
      } else {
			SendMessage(status_bar,SB_SETTEXT, 0,(LPARAM) "Playing...");
      }
	}
}

void audio_pause()
{
	if (can_pause) {

      can_pause = FALSE;
		paused    = TRUE;
		CMDLINE   = FALSE;

		if (MPEG) {
			SuspendThread(MPEG_Thread);
			waveOutPause(*phwo);
		} else {
         DWORD error;

  			SuspendThread(MCI_Thread);
			error = mciSendString ((LPSTR)"pause sounder", NULL, 0, NULL);

         if (error != 0) {
         	report_mci_error(hWnd, error, "Could not pause");
            return;
         }
		}
		ok_play();
		SendMessage(status_bar, SB_SETTEXT, 0,(LPARAM) "Paused");
   }
}

void audio_stop()
{
	if (can_stop) {

		clear_sound();

      can_stop  = FALSE;
	   CMDLINE   = FALSE;
      paused    = FALSE;
      save_mode = FALSE;

		File_Init(TRUE);
		ok_play();
		no_stop();
   }
}

void audio_properties()
{
	 char MPEGinfo[1024];

	 lstrcpy(MPEGinfo, "Version: ");
	 lstrcat(MPEGinfo, header_copy.version_string());
	 lstrcat(MPEGinfo, "\nLayer :  ");
	 lstrcat(MPEGinfo, header_copy.layer_string());
	 lstrcat(MPEGinfo, "\nMode : ");
	 lstrcat(MPEGinfo, header_copy.mode_string());
	 lstrcat(MPEGinfo, "\nSample Frequency : ");
	 lstrcat(MPEGinfo, header_copy.sample_frequency_string());
	 lstrcat(MPEGinfo, "\nBitrate: ");
	 lstrcat(MPEGinfo, header_copy.bitrate_string());

	 lstrcat(MPEGinfo, "\nChecksums? : ");
	 lstrcat(MPEGinfo, header_copy.checksums() ?
							 "Yes" : "No");
	 lstrcat(MPEGinfo, "\nOriginal?        : ");
	 lstrcat(MPEGinfo, header_copy.original() ?
							 "Yes" : "No");
	 lstrcat(MPEGinfo, "\nCopyright?     : ");
	 lstrcat(MPEGinfo, header_copy.copyright() ?
							 "Yes" : "No");

	 lstrcat(MPEGinfo, "\n\nFrames  :  ");
	 lstrcat(MPEGinfo, my_itoa(scroll_range, scratch, 10));
    lstrcat(MPEGinfo, "\nLength  : ");
	 lstrcat(MPEGinfo,
     time_string(header_copy.total_ms(&stream_copy),
     			     scratch));
	 lstrcat(MPEGinfo, "\nFile Size : ");
	 lstrcat(MPEGinfo, my_itoa(stream_copy.file_size() >> 10,
							 scratch, 10));
	 lstrcat(MPEGinfo, " KB");

	 MessageBox(hWnd, MPEGinfo, "MPEG Properties", MB_OK | MB_ICONINFORMATION);
}

void help_web_site()
{
   SHELLEXECUTEINFO sei;
   sei.cbSize = sizeof(sei);
   sei.fMask  = SEE_MASK_NOCLOSEPROCESS | SEE_MASK_CLASSNAME;
   sei.hwnd   = hWnd;
   sei.lpVerb = "open";
   sei.lpFile = "C:\\Program Files\\Netscape\\Communicator\\Program\\netscape.exe";
   sei.lpParameters = "http://www-inst.eecs.berkeley.edu/~ctsay/mp2win32.html";
   sei.lpDirectory  = "";
   sei.nShow        = SW_SHOWNORMAL;
	sei.lpClass      = ".html";

   ShellExecuteEx(&sei);
}

void about()
{
   aboutbox_open = TRUE;
	MessageBox(hWnd,
"maplay 1.2+ for Win32 version 1.A (12/16/97) - "
#ifdef PENTIUM
"Pentium"
#else
"i486"
#endif
" build.\n\n"
"A Full Quality MPEG-1 and MPEG-2 LSF (Layers I, II, & III) Audio Decoder\n"
"Copyright  1996, 1997 Jeff Tsay (ctsay@pasteur.eecs.berkeley.edu)\n\n"
"Based on maplay 1.2, Copyright  1993, 1994 "
"Tobias Bading (bading@cs.tu-berlin.de)\n\n"
"Layer III code adapted from the ISO MPEG Audio Subgroup Software "
"Simulation Group.\n\n"
"You may find information and the latest version at:\n"
"http://www-inst.eecs.berkeley.edu/~ctsay/mp2win32.html\n\n"
"Compiled with Borland C++ 5.01A courtesy of Borland International.\n\n"
"This program is free software. See the GNU General Public License in "
"the file\n"
"COPYING for more details.",
"About maplay 1.2+ for Win32", MB_OK);
   aboutbox_open = FALSE;
}

static TOOLINFO prevbut_ti, nextbut_ti;

#pragma argsused
LRESULT APIENTRY SounderWndProc(HWND hWnd, UINT message,
										  WPARAM wParam, LPARAM lParam)
{
    DWORD command;
	 BOOL disabled, selected;
	 int new_pos;

    static int center_x, center_y;
    static HCURSOR orig_cursor, remote_cursor;

	 switch (message)
	 {
		  case WM_CREATE:
		  openbut= CreateWindow ("BUTTON", NULL,
					  WS_CHILD| WS_VISIBLE | WS_TABSTOP |
					  BS_OWNERDRAW | BS_PUSHBUTTON,
					  5, 5, 24, 24,
					  hWnd, (HMENU)CM_FILEOPEN, hInst, NULL);

		  playbut= CreateWindow ("BUTTON", NULL,
					  WS_CHILD | WS_VISIBLE | WS_DISABLED |
					  WS_TABSTOP | BS_OWNERDRAW | BS_PUSHBUTTON,
					  45, 5, 24, 24,
					  hWnd, (HMENU)CM_AUDIOPLAY, hInst,
					  NULL);

        OldPlayButWndProc =
        		(WNDPROC) SetWindowLong(playbut,GWL_WNDPROC,
            								(LONG) NewPlayButWndProc);

		  pausebut= CreateWindow ("BUTTON", NULL,
						WS_CHILD| WS_VISIBLE| WS_DISABLED |
						WS_TABSTOP | BS_OWNERDRAW | BS_PUSHBUTTON,
						69, 5, 24, 24,
						hWnd, (HMENU)CM_AUDIOPAUSE, hInst, NULL);

		  stopbut= CreateWindow ("BUTTON", NULL,
					  WS_CHILD| WS_VISIBLE | WS_DISABLED |
					  WS_TABSTOP | BS_OWNERDRAW | BS_PUSHBUTTON,
						93, 5, 24, 24,
					  hWnd, (HMENU)CM_AUDIOSTOP, hInst, NULL);

		  rewbut=  CreateWindow ("BUTTON", NULL,
					  WS_CHILD| WS_VISIBLE | WS_DISABLED |
					  WS_TABSTOP | BS_PUSHBUTTON | BS_OWNERDRAW,
					  157, 5, 24, 24,
					  hWnd, (HMENU)CM_AUDIOREWIND, hInst, NULL);

			 ffbut= CreateWindow ("BUTTON", NULL,
					  WS_CHILD| WS_VISIBLE | WS_DISABLED |
					  WS_TABSTOP | BS_PUSHBUTTON | BS_OWNERDRAW,
					  181, 5, 24, 24,
					  hWnd, (HMENU)CM_AUDIOFAST_FORWARD, hInst, NULL);

		  prevbut= CreateWindow ("BUTTON", NULL,
					  WS_CHILD| WS_VISIBLE | WS_DISABLED |
					  WS_TABSTOP | BS_PUSHBUTTON | BS_OWNERDRAW,
					  133, 5, 24, 24,
					  hWnd, (HMENU)CM_LISTPREV, hInst, NULL);

        // subclassing the prevbut
        OldPrevButWndProc =
        		(WNDPROC) SetWindowLong(prevbut,GWL_WNDPROC,
            								(LONG) NewPrevButWndProc);

		  nextbut= CreateWindow ("BUTTON", NULL,
					  WS_CHILD| WS_VISIBLE | WS_DISABLED |
					  WS_TABSTOP | BS_PUSHBUTTON | BS_OWNERDRAW,
					  205, 5, 24, 24,
					  hWnd, (HMENU)CM_LISTNEXT, hInst, NULL);

        // subclassing the nextbut
        OldNextButWndProc =
        		(WNDPROC) SetWindowLong(nextbut,GWL_WNDPROC,
            								(LONG) NewNextButWndProc);

        aboutbut= CreateWindow ("BUTTON", NULL,
					  WS_CHILD| WS_VISIBLE | WS_TABSTOP |
					  BS_PUSHBUTTON | BS_OWNERDRAW,
					  245, 5, 24, 24,
					  hWnd, (HMENU)CM_HELPABOUT, hInst, NULL);

		  tracker= CreateWindow(TRACKBAR_CLASS, NULL,
					  WS_CHILD | WS_VISIBLE | WS_TABSTOP |
					  WS_DISABLED | TBS_AUTOTICKS | TBS_ENABLESELRANGE,
					  5, 34, 263, 30, hWnd, (HMENU) 8, hInst,  NULL);

   	  status_bar= CreateStatusWindow(WS_CHILD | WS_VISIBLE,
					                        "No file", hWnd, 0);

		  SendMessage(status_bar, SB_SETPARTS, 4, (LPARAM) panes);

        tooltip_control=CreateWindow(TOOLTIPS_CLASS, (LPSTR) NULL,
        						TTS_ALWAYSTIP, CW_USEDEFAULT, CW_USEDEFAULT,
                        CW_USEDEFAULT, CW_USEDEFAULT, hWnd,
                        (HMENU) NULL, hInst, NULL);

        // setup the tooltip control for prevbut and nextbut
        prevbut_ti.cbSize			= sizeof(TOOLINFO);
        prevbut_ti.uFlags			= TTF_IDISHWND;
        prevbut_ti.hwnd				= prevbut;
        prevbut_ti.hinst			= hInst;
        prevbut_ti.uId				= (UINT) prevbut;
        prevbut_ti.lpszText		= LPSTR_TEXTCALLBACK;
        SendMessage(tooltip_control, TTM_ADDTOOL,
        					0, (LPARAM) (LPTOOLINFO) &prevbut_ti);

        nextbut_ti.cbSize			= sizeof(TOOLINFO);
        nextbut_ti.uFlags			= TTF_IDISHWND;
        nextbut_ti.hwnd				= nextbut;
        nextbut_ti.hinst			= hInst;
        nextbut_ti.uId				= (UINT) nextbut;
        nextbut_ti.lpszText		= LPSTR_TEXTCALLBACK;
        SendMessage(tooltip_control, TTM_ADDTOOL,
        					0, (LPARAM) (LPTOOLINFO) &nextbut_ti);

        SendMessage(tooltip_control, TTM_SETDELAYTIME, TTDT_AUTOMATIC, 200);

        // load our beautiful remote cursor
        orig_cursor = GetCursor();
        remote_cursor = LoadCursor(hInst, MAKEINTRESOURCE(IDC_REMOTE));

        if (systray_mode)
        	  TrayMessage(NIM_ADD, hWnd);

        DragAcceptFiles(hWnd, TRUE);

		  return DefWindowProc(hWnd, message, wParam, lParam);

		  case WM_DRAWITEM:
		  hdc = GetDC(hWnd);
		  dis = (DRAWITEMSTRUCT *) lParam;
		  disabled = (dis->itemState & ODS_DISABLED) == ODS_DISABLED;
		  selected = (dis->itemState & ODS_SELECTED) == ODS_SELECTED;

		  switch (dis->CtlType) {

		  case ODT_BUTTON:
		  if (dis->hwndItem == openbut)
			if (selected)
            DrawIcon(hdc, 5, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_9)));
			else
				DrawIcon(hdc, 5, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_12)));
		  else if (dis->hwndItem == playbut)
			if (disabled)
				DrawIcon(hdc, 45, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_14)));
			else
				if (selected)
					DrawIcon(hdc, 45, 5, LoadIcon(hInst,MAKEINTRESOURCE(ICON_18)));
				else
					DrawIcon(hdc, 45, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_13)));
		  else if (dis->hwndItem == pausebut)
			if (disabled)
			  DrawIcon(hdc, 69, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_4)));
			else
				if (selected)
				  DrawIcon(hdc, 69, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_24)));
				else
				  DrawIcon(hdc, 69, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_6)));
		  else if (dis->hwndItem == stopbut)
			if (disabled)
			  DrawIcon(hdc, 93, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_3)));
			else
			  if (selected)
				  DrawIcon(hdc, 93, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_10)));
			  else
				  DrawIcon(hdc, 93, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_5)));
		  else if (dis->hwndItem == aboutbut)
			  if (selected)
				DrawIcon(hdc, 245, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_21)));
			  else
				DrawIcon(hdc, 245, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_15)));
		  else if (dis->hwndItem == rewbut)
			if (disabled)
				DrawIcon(hdc, 157, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_8)));
			else
				if (selected)
					DrawIcon(hdc, 157, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_22)));
				else
					DrawIcon(hdc, 157, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_7)));
		  else if (dis->hwndItem == ffbut)
			if (disabled)
				DrawIcon(hdc, 181, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_17)));
			else
				if (selected)
					DrawIcon(hdc, 181, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_23)));
				else
					DrawIcon(hdc, 181, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_16)));

		  else if (dis->hwndItem == prevbut)
			if (disabled)
				DrawIcon(hdc, 133, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_26)));
			else
				if (selected)
					DrawIcon(hdc, 133, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_27)));
				else
					DrawIcon(hdc, 133, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_20)));
		  else if (dis->hwndItem == nextbut)
			if (disabled)
				DrawIcon(hdc, 205, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_29)));
			else
				if (selected)
					DrawIcon(hdc, 205, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_30)));
				else
					DrawIcon(hdc, 205, 5, LoadIcon(hInst, MAKEINTRESOURCE(ICON_28)));
		  break;
		  }

		  ReleaseDC(hWnd, hdc);
		  return DefWindowProc(hWnd, message, wParam, lParam);

		  case WM_COMMAND:
        command = GET_WM_COMMAND_ID(wParam, lParam);
		  switch (command)
  		  {
				 case CM_FILEOPEN:
				 file_open();
				 break;

             case CM_FILESAVE_AS:
             file_save_as();
             break;

             case CM_FILEDELETE:
             file_delete();
             break;

             case CM_FILERECENTF1:
             case (CM_FILERECENTF1+1):
             case (CM_FILERECENTF1+2):
             case (CM_FILERECENTF1+3):
             file_recent(rfl_f, command - CM_FILERECENTF1);
             break;

             case CM_FILERECENTL1:
             case (CM_FILERECENTL1+1):
             case (CM_FILERECENTL1+2):
             case (CM_FILERECENTL1+3):
             file_recent(rfl_l, command - CM_FILERECENTL1);
             break;

				 case CM_FILEEXIT:
				 leave();
				 break;

				 case CM_AUDIOPLAY:
				 audio_play();
				 break;

				 case CM_AUDIOPAUSE:
				 audio_pause();
				 break;

				 case CM_AUDIOSTOP:
				 audio_stop();
				 break;

				 case CM_AUDIOREWIND:
				 CMDLINE   = FALSE;
				 scrolling = TRUE;

				 new_pos = SendMessage(tracker, TBM_GETPOS, 0, 0);
				 if ((new_pos -= line_size) < 0)
						new_pos = 0;

				 WaitForSingleObject(args->mutex, INFINITE);
				 args->desired_position = new_pos;

				 if (!args->position_change) {
					args->position_change = true;
					if (!paused && playing)
						PostMessage(status_bar,SB_SETTEXT, 0,
                                 (LPARAM) "Seeking...");
				 }
				 ReleaseMutex(args->mutex);
				 SendMessage(tracker, TBM_SETPOS, (WPARAM) TRUE, new_pos);
				 break;

				 case CM_AUDIOFAST_FORWARD:
				 CMDLINE   = FALSE;

				 new_pos = SendMessage(tracker, TBM_GETPOS, 0, 0);
				 if ((new_pos += line_size) < scroll_range) {

	             scrolling = TRUE;
				    WaitForSingleObject(args->mutex, INFINITE);
					 args->desired_position = new_pos;
                scrolling = TRUE;

					 if (!args->position_change) {
                    args->position_change = TRUE;

						  if (!paused && playing)
								PostMessage(status_bar,SB_SETTEXT, 0,
                        				(LPARAM) "Seeking...");
                }
					 ReleaseMutex(args->mutex);

					 SendMessage(tracker, TBM_SETPOS, (WPARAM) TRUE, new_pos);
             }
				 break;


				 case CM_AUDIOREPEAT:
				 CMDLINE = FALSE;

             Repeat = !Repeat;
				 if (Repeat) {
				 	CheckMenuItem(mainmenu, CM_AUDIOREPEAT, MF_CHECKED);
				 	SendMessage(status_bar,SB_SETTEXT, 3,(LPARAM) "R");
				 } else {
				 	CheckMenuItem(mainmenu, CM_AUDIOREPEAT, MF_UNCHECKED);
				 	SendMessage(status_bar,SB_SETTEXT, 3,(LPARAM) "");
				 }
				 break;

				 case CM_MPEGPROPERTIES:
				 audio_properties();
				 break;

             case CM_LISTNEW:
				 edit_fl.reset();

             listedit_command = LIST_NEW;
             ofnListEdit.lpstrTitle = "New Playlist";
             GetOpenFileName(&ofnListEdit);

             if (listedit_ok) {
	            reset();
             	if (pl.empty()) {
             		no_list();
               } else {
               	list_load();
               }
             }
             break;

             case CM_LISTEDIT:
             if (ListMode) {

               if (pl.has_filelist())
    					edit_fl = pl.get_filelist();
               else
                  edit_fl.reset();

               listedit_command = LIST_EDIT;
               ofnListEdit.lpstrTitle = "Edit Playlist";
               GetOpenFileName(&ofnListEdit);

               if (listedit_ok) {
	              reset();
             	  if (pl.empty()) {
                    no_list();
                 } else {
                    list_load();
                 }
               }
             }
             break;

             case CM_LISTSAVE:
             list_save(&pl, hWnd);
             break;

             case CM_LISTRAND:
             if (ListMode) {
                pl.randomize_order();
                list_load();
             }
             break;

				 case CM_LISTPREV:
             if (ListMode) {
	             pl.goto_previous();
	             List_Play_Current(playing);
	             SendMessage(tooltip_control, TTM_UPDATETIPTEXT, 0,
	                         (LPARAM) (LPTOOLINFO) &prevbut_ti);
             }
				 break;

				 case CM_LISTREPEAT:
             if (ListMode) {
					 List_Play_Current(playing);
             }
				 break;

				 case CM_LISTNEXT:
             if (ListMode) {
					 List_Advance(playing, TRUE);
	             SendMessage(tooltip_control, TTM_UPDATETIPTEXT, 0,
	                         (LPARAM) (LPTOOLINFO) &nextbut_ti);
             }
				 break;

             case CM_LISTJUMP:
             CreatePlayListWnd();
             break;

  				 case CM_OPTIONS:
             CreatePropertySheet();
				 break;

             case CM_CDLIST:
             cd_list(FALSE);
             break;

             case CM_CDLISTHYBRID:
             cd_list(TRUE);
             break;

             case CM_MINICONTROL:
             open_mini_win();
             break;

             case CM_MCAPTURE:
				 CMDLINE = FALSE;

             capture_mode = !capture_mode;

				 if (capture_mode) {
               RECT r;

					GetWindowRect(hWnd, &r);

               center_x = (r.left + r.right) >> 1;
               center_y = (r.top + r.bottom) >> 1;
     				SetCursorPos(center_x, center_y);

					SetCapture(hWnd);
  				 	CheckMenuItem(mainmenu, CM_MCAPTURE, MF_CHECKED);
               SetCursor(remote_cursor);

               SetWindowPos(hWnd, HWND_TOPMOST, 0, 0, 0, 0,
                            SWP_NOMOVE	| SWP_NOSIZE | SWP_SHOWWINDOW);

				 } else {
               ReleaseCapture();
				 	CheckMenuItem(mainmenu, CM_MCAPTURE, MF_UNCHECKED);
               SetCursor(orig_cursor);

               if (topness != HWND_TOPMOST) {
                  SetWindowPos(hWnd, HWND_BOTTOM, 0, 0, 0, 0,
                               SWP_NOMOVE	| SWP_NOSIZE);
                  SetWindowPos(hWnd, topness, 0, 0, 0, 0,
                               SWP_NOMOVE	| SWP_NOSIZE | SWP_SHOWWINDOW);
               }

				 }
             break;

             case CM_HELPWEB:
             help_web_site();
             break;

				 case CM_HELPABOUT:
				 about();
				 break;

				 default:
				 break;
		  }
		  break;

        case WM_LBUTTONDOWN:
        if (capture_mode)
	        SendMessage(hWnd, WM_COMMAND, CM_AUDIOPAUSE, 0);
        break;

        case WM_MOUSEMOVE:
        if (capture_mode) {
        	   POINT p;
        		GetCursorPos(&p);
	         if ((center_x != p.x) || (center_y != p.y)) {
               SetCursorPos(center_x, center_y);
	        		if (center_x < (p.x-REMOTE_THRESH)) {
               	if (ListMode) {
		            	SendMessage(hWnd, WM_COMMAND, CM_LISTNEXT, 0);
                  } else {
                  	SendMessage(hWnd, WM_COMMAND, CM_AUDIOFAST_FORWARD, 0);
                  }
	            } else if (center_x > (p.x+REMOTE_THRESH)) {
               	if (ListMode) {
		            	SendMessage(hWnd, WM_COMMAND, CM_LISTPREV, 0);
                  } else {
                  	SendMessage(hWnd, WM_COMMAND, CM_AUDIOREWIND, 0);
                  }
	            } else if (center_y > (p.y+REMOTE_THRESH)) {
                  if (!playing) {
		            	SendMessage(hWnd, WM_COMMAND, CM_AUDIOPLAY, 0);
                  } else {
							SendMessage(hWnd, WM_COMMAND, CM_AUDIOFAST_FORWARD, 0);
                  }
	            } else if (center_y < (p.y-REMOTE_THRESH)) {
               	SendMessage(hWnd, WM_COMMAND, CM_AUDIOREWIND, 0);
               }
            }
        }
        break;

     	  case WM_CONTEXTMENU:
        if (capture_mode) {
           if (can_stop) {
		        SendMessage(hWnd, WM_COMMAND, CM_AUDIOSTOP, 0);
           } else {
		        SendMessage(hWnd, WM_COMMAND, CM_MCAPTURE, 0);
           }
        } else {
           DisplayContextMenu();
        }
        break;

		  case WM_HSCROLL:
  		  // Process scroll messages by sending notification to
		  // appropriate thread.
		  LRESULT new_pos;

		  switch (LOWORD(wParam)) {
		  		case TB_LINEUP:
		  		case TB_LINEDOWN:
		  		case TB_PAGEDOWN:
		  		case TB_PAGEUP:
		  		CMDLINE = FALSE;
		  		new_pos = SendMessage(tracker, TBM_GETPOS, 0, 0);

		  		if ((new_pos  <= scroll_range) && (new_pos >= 0)) {
		  			scrolling = TRUE;
               WaitForSingleObject(args->mutex, INFINITE);
		  			args->desired_position = new_pos;

		  			if (!args->position_change) {
		  				args->position_change = TRUE;
		  				if (!paused && playing)
		  					SendMessage(status_bar,SB_SETTEXT, 0,
                                 (LPARAM) "Seeking...");
               }
					ReleaseMutex(args->mutex);
            }
				break;

				case TB_THUMBTRACK:
				scrolling = TRUE;		// We are scrolling, don't allow
											// scroll position changes by threads.
				new_pos = SendMessage(tracker, TBM_GETPOS, 0, 0);
				update_timewin((int) new_pos);
				break;

            case TB_THUMBPOSITION:	// Scroll box placed somewhere
            WaitForSingleObject(args->mutex, INFINITE);
 				args->desired_position = SendMessage(tracker, TBM_GETPOS, 0, 0);

				if (!args->position_change) {
					args->position_change = true;
					if (!paused && playing)
						SendMessage(status_bar,SB_SETTEXT, 0,
                    				(LPARAM) "Seeking...");
				}
				ReleaseMutex(args->mutex);
				break;
        }
		  return 0;

		  case SCROLL_POS:	// Process new position data from threads.
		  if (!scrolling && (wParam <=scroll_range) && playing) {
				SendMessage(tracker, TBM_SETPOS, TRUE, wParam);
				update_timewin((int) wParam);
		  }
		  return 0;

		  case SEEK_ACK:		// Seek completed, allow scroll changes.
        if (playing) {
        	  if (save_mode) {
	           SendMessage(status_bar, SB_SETTEXT, 0, (LPARAM) "Saving...");
           } else {
				  SendMessage(status_bar, SB_SETTEXT, 0, (LPARAM) "Playing...");
           }
			  scrolling = FALSE;
        }
		  return 0;

		  case MM_MCINOTIFY:
				switch (wParam) {
					case MCI_NOTIFY_SUCCESSFUL:
					clear_sound();

					if (ListMode) {
						List_Advance(TRUE, FALSE);
						return 0;
					}

					if (!File_Init(TRUE))
               	return 0;

					ok_play();
					no_stop();

					if (Repeat) {
						no_play();
						MCI_play();
						SendMessage(status_bar, SB_SETTEXT, 0, (LPARAM) "Playing...");
					} else {
						if (CMDLINE && cmdline_exit)
							leave();

						SendMessage(status_bar, SB_SETTEXT, 0, (LPARAM) "Stopped");

	               if (prompt_after_done) {
                  	if (!play_after_open)
			               ShowWindow(hWnd, SW_SHOWNORMAL);
	               	if (!openbox_open)
                     	file_open();
	               }
					}

					break;
				}
		  break;

        case WM_DROPFILES:
        CMDLINE = FALSE;
        get_dropped_files(szName, MAX_FILENAME_LENGTH,
                          (HDROP) wParam);

        if (file_load(FALSE))
	        audio_play();
        break;

		  case WM_DESTROY:
		  PostQuitMessage(0);   // this is the end...
		  break;

		  case WM_CLOSE:
		  // Tell windows to destroy our window.
		  leave();
		  break;

		  case WM_THREADEND:
        save_mode = FALSE;

		  clear_sound();

		  if (ListMode) {
			  List_Advance(TRUE, FALSE);
			  return 0;
		  }

    	  ok_play();
        no_stop();

        if (!File_Init(TRUE))
           return 0;

        if (Repeat) {
		    no_play();
		    MPEG_play();
          SendMessage(status_bar,SB_SETTEXT, 0,(LPARAM) "Playing...");
		  } else {
        	 if (CMDLINE && cmdline_exit) {
            	leave();
          } else {

	          SendMessage(status_bar,SB_SETTEXT, 0,(LPARAM) "Stopped");

	          if (prompt_after_done) {
	             if (!play_after_open)
			          ShowWindow(hWnd, SW_SHOWNORMAL);
					 if (!openbox_open)
		             file_open();
		       }
          }
        }
		  break;

        case WM_CMDLINEFILE:

        lstrcpy(szName, (LPSTR) lParam);
        stradd_2n(szName);

	     if (file_load(FALSE))
           audio_play();
        break;

		  case WM_KEYDOWN:
			switch(wParam) {

            case 'C':
            case 'R':
            SendMessage(hWnd, WM_COMMAND, CM_MCAPTURE, 0);
            break;

				case 'O':
  				SendMessage(hWnd, WM_COMMAND, CM_FILEOPEN, 0);
				break;

            case VK_DELETE:
  				SendMessage(hWnd, WM_COMMAND, CM_FILEDELETE, 0);
				break;

				case VK_SPACE:
				SendMessage(hWnd, WM_COMMAND, CM_AUDIOPLAY, 0);
				break;

				case 'S':
            SendMessage(hWnd, WM_COMMAND, CM_AUDIOSTOP, 0);
				break;

				case 'P':
            SendMessage(hWnd, WM_COMMAND, CM_AUDIOPAUSE, 0);
				break;

            case VK_LEFT:
            SendMessage(hWnd, WM_COMMAND, CM_AUDIOREWIND, 0);
            break;

            case VK_RIGHT:
            SendMessage(hWnd, WM_COMMAND, CM_AUDIOFAST_FORWARD, 0);
            break;

            case VK_UP:
            case VK_SUBTRACT:
            SendMessage(hWnd, WM_COMMAND, CM_LISTPREV, 0);
            break;

            case VK_DOWN:
            case VK_ADD:
            SendMessage(hWnd, WM_COMMAND, CM_LISTNEXT, 0);
            break;

            case 'N':
            SendMessage(hWnd, WM_COMMAND, CM_LISTNEW, 0);
            break;

            case 'E':
            SendMessage(hWnd, WM_COMMAND, CM_LISTEDIT, 0);
            break;

            case 'J':
            SendMessage(hWnd, WM_COMMAND, CM_LISTJUMP, 0);
            break;

				case 'X':
				case 'Q':
				leave();
				break;
			}
		  break;


		  case 312:		// CTLCOLOR message for trackbar
		  if ((HWND) lParam == tracker)
			  return (DWORD) GetStockObject(LTGRAY_BRUSH);

		  case MYWM_NOTIFYICON:
        switch (lParam) {

	  	    case WM_LBUTTONDOWN:
  		    ShowWindow(hWnd, SW_SHOWNORMAL);
		    SetForegroundWindow(hWnd);
          hidden = FALSE;
		    return 0;

        	 case WM_RBUTTONDOWN:
          DisplayContextMenu();
          return 0;

          default:
          break;
        }
        break;

        case WM_SIZE:
        if ((wParam == SIZE_MINIMIZED) && systray_mode) {
	        ShowWindow(hWnd, SW_HIDE);
           hidden = TRUE;
        } else if (wParam == SIZE_RESTORED) {
           hidden = FALSE;
        }
        break;

	     default:
		  // Let windows handle all messages we choose to ignore.
		  return DefWindowProc(hWnd, message, wParam, lParam);
	 }
    return 0;
}

BOOL TrayMessage(DWORD dwMessage, HWND hWnd)
{
	BOOL res;
   HICON hIcon = NULL;
	NOTIFYICONDATA tnd;

	tnd.cbSize = sizeof(NOTIFYICONDATA);
	tnd.hWnd	  = hWnd;
	tnd.uID	  = 0;

   if (dwMessage != NIM_DELETE) {

	   hIcon = (HICON) LoadImage(hInst, MAKEINTRESOURCE(ICON_MP), IMAGE_ICON,
						  		        16, 16, 0);

		tnd.uFlags = NIF_MESSAGE | NIF_ICON | NIF_TIP;
		tnd.uCallbackMessage	= MYWM_NOTIFYICON;
		tnd.hIcon  = hIcon;

      LoadString(hInst, IDS_NAME, tnd.szTip, 64);

   } else {
     tnd.uFlags = 0;
   }

	res = Shell_NotifyIcon(dwMessage, &tnd);

	if (hIcon)
	   DestroyIcon(hIcon);

	return res;
}

// id of a timer
enum {IDT_HOLD=1};

BOOL CALLBACK NewPlayButWndProc(HWND hWnd, UINT message,
									     WPARAM wParam, LPARAM lParam)
{
	 static BOOL pushed = FALSE;

	 switch (message) {
    	  case BM_SETSTATE:
        		if (wParam && !pushed) {
            	 // first time it's pushed
            	 SetTimer(hWnd, IDT_HOLD, 500, (TIMERPROC) NULL);
                pushed = TRUE;
            }
            else if (!wParam) {
            	 pushed = FALSE;
            }
            break;

        case WM_TIMER:
        		if ((wParam == IDT_HOLD) && pushed) {
            	 CreatePlayListWnd();
                pushed = FALSE;
            }
            KillTimer(hWnd, IDT_HOLD);
            break;
    }
    return CallWindowProc(OldPlayButWndProc, hWnd, message,
    							  wParam, lParam);
}
// new WndProc for the prevbut
// added functionality:
// - display the previous song alias or name
BOOL CALLBACK NewPrevButWndProc(HWND hWnd, UINT message,
									     WPARAM wParam, LPARAM lParam)
{
    char *tooltip_text;
	 switch (message) {
        case WM_MOUSEMOVE:
        MSG msg;
        msg.hwnd		= hWnd;
        msg.message	= message;
        msg.wParam	= wParam;
        msg.lParam	= lParam;
        msg.time		= GetMessageTime();
        msg.pt.x		= LOWORD(lParam);	// this is supposed to be screen
        msg.pt.y		= HIWORD(lParam); // coordinates; lParam is not
        SendMessage(tooltip_control, TTM_RELAYEVENT,
        				  0, (LPARAM) (LPMSG)&msg);
    	  return CallWindowProc(OldPrevButWndProc, hWnd, message,
    									wParam, lParam);
        // end of "case WM_MOUSEMOVE"

        case WM_NOTIFY:
        switch (((NMHDR FAR *) lParam)->code) {
            case TTN_NEEDTEXT:

            tooltip_text = pl.get_previous_song_alias();
            if (tooltip_text == NULL)
            	tooltip_text = Proper_Filename(pl.get_previous_song_name());
            ((LPTOOLTIPTEXT) lParam)->lpszText = tooltip_text;
            return TRUE; // end of TTN_NEEDTEXT

            default:
				return CallWindowProc(OldPrevButWndProc, hWnd, message,
    													wParam, lParam);
        } // end of "case WM_NOTIFY"

    	  default:
        return CallWindowProc(OldPrevButWndProc, hWnd, message,
    									wParam, lParam);
    }
}

// new WndProc for the nextbut
// added functionality:
// - display the next song alias or name
BOOL CALLBACK NewNextButWndProc(HWND hWnd, UINT message,
									     WPARAM wParam, LPARAM lParam)
{
    char *tooltip_text;
	 switch (message) {
        case WM_MOUSEMOVE:
        MSG msg;
        msg.hwnd		= hWnd;
        msg.message	= message;
        msg.wParam	= wParam;
        msg.lParam	= lParam;
        msg.time		= GetMessageTime();
        msg.pt.x		= LOWORD(lParam);	// this is supposed to be screen
        msg.pt.y		= HIWORD(lParam); // coordinates; lParam is not
        SendMessage(tooltip_control, TTM_RELAYEVENT,
        				  0, (LPARAM) (LPMSG)&msg);
    	  return CallWindowProc(OldNextButWndProc, hWnd, message,
    									wParam, lParam);
        // end of "case WM_MOUSEMOVE"

        case WM_NOTIFY:
        switch (((NMHDR FAR *) lParam)->code) {
            case TTN_NEEDTEXT:
            tooltip_text = pl.get_next_song_alias();

            if (tooltip_text == NULL)
               tooltip_text = Proper_Filename(pl.get_next_song_name());
            ((LPTOOLTIPTEXT) lParam)->lpszText = tooltip_text;
            return TRUE; // end of TTN_NEEDTEXT

            default:
				return CallWindowProc(OldPrevButWndProc, hWnd, message,
    													wParam, lParam);
        } // end of "case WM_NOTIFY"

    	  default:
    	  return CallWindowProc(OldNextButWndProc, hWnd, message,
    									wParam, lParam);
    }
}

void DisplayContextMenu()
{
    HMENU hmenu;            // top-level menu
    HMENU hmenuTrackPopup;  // pop-up menu
    POINT p;
    UINT  enable;

    if ((hmenu = LoadMenu(hInst, MAKEINTRESOURCE(POPUPMENU))) == NULL)
        return;

    // TrackPopupMenu cannot display the top-level menu, so get
    // the handle of the first pop-up menu.

    hmenuTrackPopup = GetSubMenu(hmenu, 0);

    enable = (!openbox_open)  ? MF_ENABLED : MF_GRAYED;
	 EnableMenuItem(hmenuTrackPopup, CM_FILEOPEN,  enable);

    enable = can_play  ? MF_ENABLED : MF_GRAYED;
	 EnableMenuItem(hmenuTrackPopup, CM_AUDIOPLAY,  enable);

    enable = can_pause ? MF_ENABLED : MF_GRAYED;
	 EnableMenuItem(hmenuTrackPopup, CM_AUDIOPAUSE, enable);

	 enable = can_stop  ? MF_ENABLED : MF_GRAYED;
    EnableMenuItem(hmenuTrackPopup, CM_AUDIOSTOP,  enable);

    enable = Repeat ? MF_CHECKED : MF_UNCHECKED;
    CheckMenuItem(hmenuTrackPopup, CM_AUDIOREPEAT, enable);

/*    enable = (can_play || can_pause || can_stop) ? MF_ENABLED : MF_GRAYED;
	 EnableMenuItem(hmenuTrackPopup, CM_AUDIOREWIND, enable);
	 EnableMenuItem(hmenuTrackPopup, CM_AUDIOFAST_FORWARD, enable); */

    enable = ListMode  ? MF_ENABLED : MF_GRAYED;
  	 EnableMenuItem(hmenuTrackPopup, CM_LISTPREV, enable);
	 EnableMenuItem(hmenuTrackPopup, CM_LISTNEXT, enable);
    EnableMenuItem(hmenuTrackPopup, CM_LISTJUMP, enable);

    enable = (!optionsbox_open) ? MF_ENABLED : MF_GRAYED;
	 EnableMenuItem(hmenuTrackPopup, CM_OPTIONS,  enable);

    enable = (!aboutbox_open)   ? MF_ENABLED : MF_GRAYED;
	 EnableMenuItem(hmenuTrackPopup, CM_HELPABOUT,  enable);

	 ModifyMenu(hmenuTrackPopup, 13,
    				MF_BYPOSITION | MF_ENABLED	| MF_POPUP | MF_STRING,
				   (UINT) rfl_file_menu, "Recent &Files");

	 ModifyMenu(hmenuTrackPopup, 14,
    				MF_BYPOSITION | MF_ENABLED	| MF_POPUP | MF_STRING,
				   (UINT) rfl_list_menu, "Recent &Playlists");

    GetCursorPos(&p);

    TrackPopupMenuEx(hmenuTrackPopup, TPM_LEFTALIGN, p.x, p.y, hWnd, NULL);
}

BOOL open_mini_win()
{
	static HWND mini_hWnd = NULL;
   static HWND dumb_mini_hWnd = NULL;

   if (dumb_mini_hWnd == NULL) {
    	 dumb_mini_hWnd = CreateWindowEx(WS_EX_TOOLWINDOW, "BUTTON",
       "", WS_VISIBLE | WS_POPUP, 0, 0, 0, 0, NULL,
       NULL, hInst, NULL);
   }

   if ((mini_hWnd != NULL) && IsWindow(mini_hWnd)) {
   	SetForegroundWindow(mini_hWnd);
      return TRUE;
   }

   mini_hWnd = CreateDialog(hInst, MAKEINTRESOURCE(IDD_TOOLWIN),
                            dumb_mini_hWnd, MiniWndProc);

   return (mini_hWnd != NULL);
}

#define NUM_TOOL_BUTTONS 3
BOOL CALLBACK MiniWndProc(HWND hDlg, UINT message,
                          WPARAM wParam, LPARAM lParam)
{
   static HWND tb_win = NULL;

   switch (message) {

    case WM_INITDIALOG:
    {
/*     TBADDBITMAP tbab;
     TBBUTTON tbb[NUM_TOOL_BUTTONS];
     int iPlayBmp, iPauseBmp, iStopBmp;
     int iPlayStr, iPauseStr, iStopStr;

     tb_win = GetDlgItem(hDlg, IDC_CONTROLTOOL);

     // Send the TB_BUTTONSTRUCTSIZE message, which is required for
     // backward compatibility.
     SendMessage(tb_win, TB_BUTTONSTRUCTSIZE, (WPARAM) sizeof(TBBUTTON), 0);

     tbab.hInst = hInst;
     tbab.nID   = IDB_PLAYBMP;
     iPlayBmp = SendMessage(tb_win, TB_ADDBITMAP, (WPARAM) 1, (WPARAM) &tbab);
     tbab.hInst = hInst;
     tbab.nID   = IDB_PAUSEBMP;
     iPauseBmp = SendMessage(tb_win, TB_ADDBITMAP, (WPARAM) 1, (WPARAM) &tbab);
     tbab.hInst = hInst;
     tbab.nID   = IDB_STOPBMP;
     iStopBmp = SendMessage(tb_win, TB_ADDBITMAP, (WPARAM) 1, (WPARAM) &tbab);

     iPlayStr = SendMessage(tb_win, TB_ADDSTRING, 0,
                            (LPARAM) (LPSTR) "Play\0");
     iPauseStr = SendMessage(tb_win, TB_ADDSTRING, 0,
                             (LPARAM) (LPSTR) "Pause\0");
     iStopStr = SendMessage(tb_win, TB_ADDSTRING, 0,
                         (LPARAM) (LPSTR) "Stop\0");

     tbb[0].iBitmap = iPlayBmp;
     tbb[0].idCommand = CM_AUDIOPLAY;
     tbb[0].fsState = TBSTATE_ENABLED;
     tbb[0].fsStyle = TBSTYLE_BUTTON;
     tbb[0].dwData = 0;
     tbb[0].iString = iPlayStr;

     tbb[1].iBitmap = iPauseBmp;
     tbb[1].idCommand = CM_AUDIOPAUSE;
     tbb[1].fsState = TBSTATE_ENABLED;
     tbb[1].fsStyle = TBSTYLE_BUTTON;
     tbb[1].dwData = 0;
     tbb[1].iString = iPauseStr;

     tbb[2].iBitmap = iStopBmp;
     tbb[2].idCommand = CM_AUDIOSTOP;
     tbb[2].fsState = TBSTATE_ENABLED;
     tbb[2].fsStyle = TBSTYLE_BUTTON;
     tbb[2].dwData = 0;
     tbb[2].iString = iStopStr;

     SendMessage(tb_win, TB_ADDBUTTONS, (WPARAM) NUM_TOOL_BUTTONS,
                 (LPARAM) (LPTBBUTTON) &tbb); */
    }
    return TRUE;

/*    case WM_NOTIFY:

    return TRUE; */
   }
   return FALSE;
}
