/************************* MPEG-2 NBC Audio Decoder **************************
 *                                                                           *
"This software module was originally developed by 
AT&T, Dolby Laboratories, Fraunhofer Gesellschaft IIS 
and edited by
Yoshiaki Oikawa (Sony Corporation),
Mitsuyuki Hatanaka (Sony Corporation)
Mike Coleman (Five Bats Research)
in the course of development of the MPEG-2 NBC/MPEG-4 Audio standard ISO/IEC 13818-7, 
14496-1,2 and 3. This software module is an implementation of a part of one or more 
MPEG-2 NBC/MPEG-4 Audio tools as specified by the MPEG-2 NBC/MPEG-4 
Audio standard. ISO/IEC  gives users of the MPEG-2 NBC/MPEG-4 Audio 
standards free license to this software module or modifications thereof for use in 
hardware or software products claiming conformance to the MPEG-2 NBC/MPEG-4
Audio  standards. Those intending to use this software module in hardware or 
software products are advised that this use may infringe existing patents. 
The original developer of this software module and his/her company, the subsequent 
editors and their companies, and ISO/IEC have no liability for use of this software 
module or modifications thereof in an implementation. Copyright is not released for 
non MPEG-2 NBC/MPEG-4 Audio conforming products.The original developer
retains full right to use the code for his/her  own purpose, assign or donate the 
code to a third party and to inhibit third party from using the code for non 
MPEG-2 NBC/MPEG-4 Audio conforming products. This copyright notice must
be included in all copies or derivative works." 
Copyright(c)1996,1997.
 *                                                                           *
 ****************************************************************************/

#include "all.h"
#include <fcntl.h>
#include <sys/stat.h>
#include <unistd.h>

#include "aiff_support.h"

#ifdef WIN32
#include <io.h>
#endif

#define	PATH_LEN    128

static	FILE*	bin;
static	int	outfd[Chans];
static	char	output_path[PATH_LEN];

static	int            useAIFF         = 0;
static	unsigned short AIFF_channels   = 1;
static	unsigned short AIFF_wordsize   = 16;
static	unsigned long  AIFF_frames     = 0;
static	unsigned long  AIFF_sampleRate = 48000;
static	MC_Info*       AIFF_MC_Info    = 0;

void
usage(char *s)
{
    PRINT(SE, "usage:  %s [options] chan_file pcm_file\n", s);
    PRINT(SE, "	options are:\n");

    PRINT(SE, "	-a  (create AIFF output files)\n");
    PRINT(SE, "	-i  (ADIF header present)\n");
    PRINT(SE, "	-s sampling_frequency, Hz   (48 kHz is default)\n");
    PRINT(SE, "	-D[a-Z]	(enable debugging printouts)\n");
#ifdef  SSR
    PRINT(SE, "	-p profile (Main or LC or SSR, Main is default)\n");
    PRINT(SE, "	-b Band For SSR Profile (1 to 4, 4 is default)\n");
#else
    PRINT(SE, "	-p profile (Main or LC, Main is default)\n");
#endif
    myexit(0);
}

void
initio(int argc, char *argv[])
{
    int i, j, *p;
    char *cmd, *s;

    /* set defaults */
    adif_header_present = 0;
    current_program = -1;
    default_config = 1;
    mc_info.profile = Main_Profile;
    mc_info.sampling_rate_idx = Fs_48;

    /* save cmd 
     * ARGBEGIN is so clever that it throws away argv[0] and 
     * increments the argv pointer
     */
    cmd = argv[0];

    ARGBEGIN {
    default:
	PRINT(SE, "unknown option\n");
	usage(cmd);
    case 'a':
    	useAIFF = 1;
    	break;
    case 'i':
	adif_header_present=1;
	break;
    case 'p':
	s = ARGF();
	p = &mc_info.profile;
	if ((!strncmp(s, "Main", 4)) || (!strncmp(s, "main", 4))) {
	    *p = Main_Profile; 
	    break;
	}
	if ((!strncmp(s, "LC", 2)) || (!strncmp(s, "lc", 2))) {
	    *p = LC_Profile; 
	    break;
	}
#ifdef	SSR
	if ((!strncmp(s, "SSR", 3)) || (!strncmp(s, "ssr", 3))) {
	    *p = SSR_Profile;
	    break;
	}
#endif
	PRINT(SE, "Unsupported profile %s\n", s);
	myexit("profile");
	break;
    case 's':
	j = atoi(ARGF());
	for (i=0; i<(1<<LEN_SAMP_IDX); i++) {
	    if (j == samp_rate_info[i].samp_rate)
		break;
	}
	if (i == (1<<LEN_SAMP_IDX)) {
	    PRINT(SE, "Unsupported sampling frequency %d\n", j);
	    myexit("samling_rate_idx");
	}
	mc_info.sampling_rate_idx = i;
	AIFF_sampleRate = samp_rate_info[i].samp_rate;
	break;
    case 'D':
	for (s = ARGF(); *s; s++) {
	    debug[(int)(*s)] = 1;
	    PRINT(SE, "debug option %c enabled\n", *s);
	}
	break;
#ifdef SSR
  case 'b':
        s = ARGF();
        ssr_decoder_band = atoi(s);
      if(ssr_decoder_band == 1)
          PRINT(SE, "%d-band is decoded in SSR profile\n", ssr_decoder_band);
      else
          PRINT(SE, "%d-bands are decoded in SSR profile\n", ssr_decoder_band);
        break;
#endif
    } 
    ARGEND

    if(argc != 2) {
	usage(cmd);
    }
    
    bin = fopen(argv[0], "rb");
    if(bin == 0) {
	PRINT(SE, "cant open %s\n", argv[0]);
	myexit("open");
    }

    if (strlen(argv[1]) + strlen("_f00.pcm") + 1 > PATH_LEN) {
	PRINT(SE, "output path too long %s\n", argv[1]);
	myexit("create");
    }
    strcpy(output_path, argv[1]);
}

static void
write_fext(Ch_Info *cip, char c, int n)
{
	cip->fext = (char *)mal1(5);
	sprintf(cip->fext, "_%c%02d", c, n);
}


int
open_output_files(MC_Info *mip)
{
    int i, j;
    char file[PATH_LEN];

    if (useAIFF){
	AIFF_MC_Info = mip;
	AIFF_sampleRate = samp_rate_info[mip->sampling_rate_idx].samp_rate; 
    }	
    j=0;
    /* if FCenter == 1 (center speaker present)
     *	position    index   extension
     *	left	    1	    01
     *	center	    0	    00
     *	right	    2	    02
     * if FCenter == 0 (center speaker absent)
     *	position    index   extension
     *	left	    0	    01
     *	right	    1	    02
      */
    for (i=((FCenter==1) ? 0 : 1); i<FChans; i++) {
	write_fext(&mip->ch_info[j], 'f', i);
	j++;
    }
    for (i=0; i<SChans; i++) {
	write_fext(&mip->ch_info[j], 's', i);
	j++;
    }
    /* if BFCenter == 1 (center speaker present)
     *	position    index   extension
     *	left	    3	    00
     *	center	    5	    02
     *	right	    4	    01
     * if BCenter == 0 (center speaker absent)
     *	position    index   extension
     *	left	    3	    00
     *	right	    4	    01
      */
    for (i=0; i<BChans; i++) {
	write_fext(&mip->ch_info[j], 'b', i);
	j++;
    }
    for (i=0; i<LChans; i++) {
	write_fext(&mip->ch_info[j], 'l', i);
	j++;
    }
    
    for (i=0; i<Chans; i++) {
	if (!(mip->ch_info[i].present))
	    continue;
       
	strcpy(file, output_path);
	strcat(file, mip->ch_info[i].fext);
	if (!useAIFF)
	    strcat(file, ".pcm");
	else
	    strcat(file, ".aif");

	if (debug['o'])
	    PRINT(SE, "opening file %s at position %d\n", file, i);
	    
#ifdef WIN32
	outfd[i] = _open (file, _O_CREAT | _O_TRUNC | _O_BINARY | _O_WRONLY, _S_IWRITE);
#else
	outfd[i] = creat(file, 0666);
#endif

	if(outfd[i] < 0) {
	    PRINT(SE, "cant create %s\n", file);
	    myexit("create");
	}
	if ( useAIFF ) {
	    /* write dummy header and position for audio data */
	    if ( WriteAIFFHeader(outfd[i], AIFF_channels, AIFF_frames, AIFF_wordsize, AIFF_sampleRate) != 0 ) {
		PRINT( SE, "cannot write AIFF header for %s\n", file );
		myexit( "aiff-header" );
	    }
	}
    }
    return(1);
}

void
fmtchan(char *p, float *data, int stride)
{
    int i, c;
    float s;

    for(i=0; i<LN2; i++) {
	s = *data++;
	if(s < 0) {
	    s -= .5;
	    if(s < -0x7fff)
		s = (float) -0x7fff;
	} else {
	    s += .5;
	    if(s > 0x7fff)
		s = (float) 0x7fff;
	}
	c = (int) s;

    p[0] = (c >> 8) & 0xff;
    p[1] = c & 0xff;

	p += stride;
    }
}

void
writeout(Float **data, MC_Info *mip)
{
   int i, stride, size;
   char obuf[LN2*2];
   static int opened = 0;
   
   if (!opened) {
      open_output_files(mip);
      opened=1;
   }
   
   stride=2;
   size = sizeof(obuf);
   for (i=0; i<Chans; i++) {
      if (!(mip->ch_info[i].present))
         continue;
      
      fmtchan(obuf, data[i], stride);
      if(write(outfd[i], obuf, size) != size) {
         PRINT(SE, "write error\n");
         myexit("write");
      }
   }
   if ( useAIFF )
      AIFF_frames += LN2;
}

int
getbyte(void)
{
    int c;

    c = getc(bin);
    if(c < 0)
	myexit((debug['n']) ? 
	    "End of Bitstream\n" : 0);
    return c;
}

void
myexit(char *s)
{
   if ( useAIFF )
      closefiles( AIFF_MC_Info );
    if (debug['v'])
	PRINT(SE,"%lx %d\n", cword, nbits);
    if (s)
	PRINT(SE,"\n%s\n", s);
    exit(s ? 1 : 0);
}

void
closefiles( MC_Info *mip )
{
   int i;
   if ( mip && useAIFF )
   {
      for ( i = 0; i < Chans; i++ )
      {
         if ( !(mip->ch_info[i].present) )
            continue;
         WriteAIFFHeader( outfd[i], AIFF_channels, AIFF_frames, AIFF_wordsize, AIFF_sampleRate );
         close( outfd[i] );
      }
   }
}
