////////////////////////////////////////////////////////////
// comp.c - BC computer AI code!
// Copyright Joe Kopena 1998
////////////////////////////////////////////////////////////

#include <stdio.h>
#include <math.h>
#include <allegro.h>
#include <noflick.h>

#include "data.h"
#include "vars.h"
#include "game.h"
#include "colors.h"

#define MAX_NUM_FRONT_COLS    NUM_COLUMNS

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
int recalcAngle(playerBoxPtr p, int dir, int sx, int sy);

void getFirstShot(playerBoxPtr p, fixed *angle, float *velocity, int dir, int sx, int sy);
void getNewShot(playerBoxPtr p, fixed *ang, float *vel, int dir);
void adjBuildBlow(float *velocity, fixed *angle, fixed *ang, int ex, int ey, int dir);

void getDir(playerBoxPtr p, int *dir, fixed *angle);
void getSharpAngle(playerBoxPtr p, fixed *angle, int dir, int sx, int sy);
unsigned long int timeGuess(playerBoxPtr p, fixed ang, int dir);
float getVelocity(playerBoxPtr p, int sx, int sy, unsigned long int oTime, fixed angle);

int cycleDrawing(playerBoxPtr p, fixed angle, float velocity);

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
int lcx, lcy, lcType, lminD;
float lastVelAdj, lastAngleAdj;
int heights[MAX_NUM_FRONT_COLS];
int numFrontCols;

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
int doCompTurn(void) {
int res = 0, dir, sx, sy;
float velocity;
fixed angle;
playerBoxPtr p = &player[currPlayer];

   #ifdef DEBUG
          fprintf(stderr, "\n============== Computer's Turn! ================\n");
   #endif // DEBUG

   switch (game.difficulty) {
          case EASY:
               numFrontCols = 6;
          break;
          case MED:
               numFrontCols = 9;
          break;
          case HARD:
               numFrontCols = 20;
          break;
   }

   ff_eraseScreen();
   drawWindBar();
   drawVelBar();
   drawGuns();
   ff_updateScreen();

   sx = ((p->column+1)*COLUMN_WIDTH) + 2 + ((COLUMN_WIDTH-4)/2);
   sy = map[p->column].h-5;

   getDir(p, &dir, &angle);

   #ifdef DEBUG
          fprintf(stderr, "numTurns: %i\n", numTurns);
   #endif // DEBUG

   if (numTurns < 2) { // first shot
      getFirstShot(p, &angle, &velocity, dir, sx, sy);
   } else {
     if (game.difficulty > EASY) {
        #ifdef DEBUG
           fprintf(stderr, "Medium or hard difficulty!\n");
        #endif // DEBUG
        if (recalcAngle(p, dir, sx, sy)) {
           getFirstShot(p, &angle, &velocity, dir, sx, sy);

           if ((fabs(fixtof(p->angle-angle)) < 5) && (fabs(p->velocity-velocity) < 5))
              getNewShot(p, &angle, &velocity, dir);
        } else
            getNewShot(p, &angle, &velocity, dir);

     } else getNewShot(p, &angle, &velocity, dir);
   } // not first shot

   if ((res = cycleDrawing(p, angle, velocity)))
      return res;

   #ifdef DEBUG
      fprintf(stderr, "p->vel: %f; p->angle: %f\n", p->velocity, fixtof(p->angle));
   #endif // DEBUG
   res = fireShot();
   lcType = cRes; lcx = cx; lcy = cy; lminD = minDistance;

   #ifdef DEBUG
          fprintf(stderr, "*************** Computer's Turn Is Over! ***************\n\n");
   #endif // DEBUG

// end doCompTurn
return res;
}

////////////////////////////////////////////////////////////
int recalcAngle(playerBoxPtr p, int dir, int sx, int sy) {
int res = 0, dx, dy, i=0, c;
fixed a;

   for (c=p->column; c != p->column+(dir*numFrontCols); c+=dir) {
       if (map[c].h != heights[i])
          res = c;
       heights[i] = map[c].h;
       i++;
   }

   if (res) {
      if (res == p->column)
         return 1;

      dy = (map[res].h-5) - (map[p->column].h-5);
      dx = ((res+((dir>0)?1:2))*COLUMN_WIDTH) - sx;
      a = fatan2(itofix(dy), itofix(dx));

      res = 0;

      if (a < itofix(0))
         a += itofix(256);

      if (dir < 0) { // Going left
         if (a < p->angle && a > itofix(LEFT_DIR))
            res = 1;
      } else {
        if (a > p->angle)
           res = 1;
      } // going right
   }

   #ifdef DEBUG
          fprintf(stderr, "recalc? %s\n", (res)?"yes":"no");
   #endif // DEBUG

// end recalcAngle
return res;
}

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
void getFirstShot(playerBoxPtr p, fixed *angle, float *velocity, int dir, int sx, int sy) {
unsigned long int oTime;

      getSharpAngle(p, angle, dir, sx, sy);
      oTime = timeGuess(p, (*angle), dir);
      (*velocity) = getVelocity(p, sx, sy, oTime, (*angle));
      lastVelAdj = 10.0;
      lastAngleAdj = 5.0;

// end getFirstShot
}

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
void getNewShot(playerBoxPtr p, fixed *ang, float *vel, int dir) {
fixed angle = p->angle;
float velocity = p->velocity;
int ex, ey;

   ex = ((p->target->column+1)*COLUMN_WIDTH) + 2 + ((COLUMN_WIDTH-4)/2);
   ey = map[p->target->column].h-5;

   #ifdef DEBUG
      fprintf(stderr, "@@@@@ last col type: %i\n", lcType);
   #endif // DEBUG
   switch (lcType) { // last collision type
//          case OFF_SCREEN:
//               if (lcy < 0)
//                  lastVelAdj = 20;
//               else if (lcy < 100)
//                    lsatVelAdj
//               if (lcx > ex)
//                  velocity -= lastVelAdj;
//               } else if (lcx < ex) {
//                 velocity -= lastVelAdj;
//               }
//               velocity -= lastVelAdj;
//          break;

          case OFF_SCREEN:
          case BUILD_BLOW:
               if (game.difficulty < MED) {
                    lastVelAdj /= 1.12;
               } if (game.difficulty < HARD) {
                  if (lminD < DEATH_DIST * 10) {
                     lastVelAdj /= 2.0;
                  } else {
                    lastVelAdj = 6;
                  } // Not a close shot
               } else {
                  if (lminD < DEATH_DIST * 5) {
                     lastVelAdj /= 2;
                  } else if (lminD < DEATH_DIST * 10) {
                     lastVelAdj = 4;
                  } else if (lminD < DEATH_DIST * 17) {
                     lastVelAdj = 6;
                  } else if (lminD < DEATH_DIST * 26) {
                     lastVelAdj = 7;
                  } else {
                    lastVelAdj = 12;
                  } // Not a close shot
               } // HARD

               #ifdef DEBUG
                      fprintf(stderr, "lastVelAdj: %f\n", lastVelAdj);
               #endif // DEBUG
               adjBuildBlow(&velocity, &angle, ang, ex, ey, dir);
          break;
   } // end swicth lcType

   #ifdef DEBUG
      fprintf(stderr, "@@@@@ angle: %f; vel: %f\n", fixtof(angle), velocity);
   #endif // DEBUG

   (*ang) = angle;
   (*vel) = velocity;

// end getNewShot
}

////////////////////////////////////////////////////////////
void adjBuildBlow(float *velocity, fixed *angle, fixed *ang, int ex, int ey, int dir) {

      #ifdef DEBUG
         fprintf(stderr, "lcx:%i; ex:%i; dir:%i; ", lcx, ex, dir);
      #endif // DEBUG

      if (dir < 0) {    // Shooting to the left
         if (lcx < ex) {
            (*velocity) -= lastVelAdj;
            #ifdef DEBUG
                   fprintf(stderr, "-=\n");
            #endif // DEBUG
         } else {
           (*velocity) += lastVelAdj;
           #ifdef DEBUG
                  fprintf(stderr, "+=\n");
           #endif // DEBUG
         }
      } else { // shooting to the right
        if (lcx < ex) {
           (*velocity) += lastVelAdj;
           #ifdef DEBUG
                  fprintf(stderr, "+=\n");
           #endif // DEBUG
        } else {
          (*velocity) -= lastVelAdj;
          #ifdef DEBUG
                 fprintf(stderr, "-=\n");
          #endif // DEBUG
        }
      }

               if ((*velocity) > MAX_VELOCITY) {
                  (*velocity) = MAX_VELOCITY;
                  if (dir > 0)
                     (*angle) += ftofix(lastAngleAdj);
                  else (*angle) -= ftofix(lastAngleAdj);
               } else if ((*velocity) < 0) {
                 (*velocity) = MAX_VELOCITY/2;
                 (*angle) = (*ang);
               }
// end adjEasyMedBuildBlow
}

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
void getDir(playerBoxPtr p, int *dir, fixed *angle) {

   if (p->target->column < p->column) {
      (*dir) = -1;
      (*angle) = itofix(LEFT_DIR+(64/3)); // 160
   } else {
     (*dir) = 1;
     (*angle) = itofix(DRIGHT_DIR-(64/3)); // 224
   }

// end getDir
}

////////////////////////////////////////////////////////////
void getSharpAngle(playerBoxPtr p, fixed *angle, int dir, int sx, int sy) {
int dx, dy, c, i=1;
fixed a;

   heights[0] = map[p->column].h;
   for (c=p->column+dir; c != p->column+(dir*numFrontCols); c+=dir) {
       heights[i] = map[c].h;
       i++;
       dy = (map[c].h - ((i<5)?(3+i):(i*2))) - sy;
       dx = ((c+((dir>0)?1:2))*COLUMN_WIDTH) - sx; // The corner of the building, not the center

       a = fatan2(itofix(dy), itofix(dx));

       if (a < itofix(0))
          a += itofix(256);

       #ifdef DEBUG
              fprintf(stderr, "col: %i; angle: %f\n", c, fixtof(a));
       #endif // DEBUG

       if (dir < 0) { // Going left
          if (a > (*angle))
             (*angle) = a;
       } else {
         if (a < (*angle) && a > itofix(LEFT_DIR))
            (*angle) = a;
       } // going right
   } // for c

// end getSharpAngle
}

////////////////////////////////////////////////////////////
unsigned long int timeGuess(playerBoxPtr p, fixed ang, int dir) {
unsigned long int guess = 75;
int g=0;

   switch (game.difficulty) {
          case EASY:
               guess = 75;
          break;
          case MED:
               if (fabs(wind) > MAX_WIND/3)
                  guess = 90;
          break;
          case HARD:
               if (wind < 0) {   // Blowing to the left
                  if (dir < 0) // Firing to the left
                     g -= 1;
                  else g += 1;
               } else { // Blowing ot the right
                 if (dir > 0) // Firing to the right
                    g -= 1;
                 else
                     g += 1;
               }

               if (fabs(wind) > MAX_WIND/3)
                  guess += 5*g;
               if (fabs(wind) > 2*(MAX_WIND/3))
                  guess += 5*g;

               if (fabs(fixtof(ang)-UP_DIR) < 30)
                  guess += 7;
          break;
   } // end switch game.difficulty

// end timeGuess
return guess;
}

////////////////////////////////////////////////////////////
float getVelocity(playerBoxPtr p, int sx, int sy, unsigned long int oTime, fixed angle) {
float velocity, xvel, yvel, gxvel, gyvel;
int ex, ey;

   ey = map[p->target->column].h-5;
   yvel = -((((float)oTime*(float)oTime)/10.0) + sy - ey) / (float)oTime;
//   fprintf(stderr, "yvel: %f; ", yvel);

   ex = ((p->target->column+1)*COLUMN_WIDTH) + 2 + ((COLUMN_WIDTH-4)/2);
   xvel = -((((float)oTime*(float)oTime)*(wind/(MAX_WIND*25.0))) + sx - ex)/(float)oTime;
//   fprintf(stderr, "xvel: %f; ", xvel);

   gxvel = (xvel/fixtof(fcos(angle)))*5;
   gyvel = (yvel/fixtof(fsin(angle)))*5;
//   fprintf(stderr, "gxvel: %f; gyvel: %f; ", gxvel, gyvel);

   velocity = (gxvel + gyvel)/2;
//   fprintf(stderr, "vel: %f; ", velocity);

   if (game.difficulty == EASY) {
      velocity += ((rand()/RAND_MAX)*20)-10;
   }

   if (velocity > MAX_VELOCITY)
      velocity = MAX_VELOCITY;
   else if (velocity < 0)
      velocity = 0;

// end getVelocity
return velocity;
}

////////////////////////////////////////////////////////////
int cycleDrawing(playerBoxPtr p, fixed angle, float velocity) {
unsigned long int oTime;

   gameTime = 0;
   do {
      oTime = gameTime;
      ff_eraseScreen();

      if (key[KEY_ESC])
         return PLAYER_QUIT;

      if (key[KEY_S] || key[KEY_F10])
         if (takeScreenShot())
            return 255;

      if (fabs(fixtof(p->angle-angle)) < 0.5)
         p->angle = angle;
      else {
           if (p->angle < angle)
              p->angle += ftofix(0.5);
           else p->angle -= ftofix(0.5);
      } // angles not close

      if (fabs(p->velocity-velocity) < 0.25)
         p->velocity = velocity;
      else {
           if (p->velocity < velocity)
              p->velocity += 0.25;
           else p->velocity -= 0.25;
      } // velocities not close

      drawWindBar();
      drawVelBar();
      drawGuns();
      ff_updateScreen();

      while ((gameTime - oTime) < 1)
            ;

   } while (p->velocity != velocity || p->angle != angle);

// end cycleDrawing
return 0;
}

