////////////////////////////////////////////////////////////
// game.c - BC game code!
// Copyright Joe Kopena 1998
////////////////////////////////////////////////////////////

#include <stdio.h>
#include <math.h>
#include <unistd.h>

#include <allegro.h>
#include <noflick.h>

#include "data.h"
#include "vars.h"
#include "colors.h"
#include "game.h"

////////////////////////////////////////////////////////////
playerBox player[NUM_BOMBERS];
mapCell map[NUM_COLUMNS];
int currPlayer = LEFT;
unsigned char gameDone = 0, matchDone = 0;
int minDistance;
int cx, cy, minX, minY;
char cRes;
char hitType;
float wind;
int numTurns;

int doTurn(void);

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
int doGame(void) {
int err = 0;

   if (!(err=setupGame())) {

      do {

         setupMatch();

         do {

            setupTurn();
            if (!(matchDone = doTurn())) {
               setupTurn();
               matchDone = doTurn();
            } // !matchDone

         } while (!matchDone);

         switch (matchDone) {
                case PLAYER_QUIT:
                     gameDone = PLAYER_QUIT;
                break;
                case SHOT_KILLED:
                     player[currPlayer].roundsWon++;
                     if (((float)player[currPlayer].roundsWon/(float)game.rounds) > 0.5) {
                        gameDone = 2;
                     }
                break;
                default:
                break;
         }

      } while (!gameDone);

      switch (gameDone) {   // Show end game stuff
             case 1:    // Show nothing, they quit!
             break;
             case 2: // Someone won!
                  showVicScreen();
             break;
             default:
                     // Must have been an error!
                     fprintf(stderr, "An error occured!\n");
             break;
      } // switch done

   } // no setup error

   shutupGame();

// end game
return err;
}

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
void showVicScreen(void) {
int wFnt, lFnt, k;
char line[80];
char wName[10];
char lName[10];

   blit(bmp(TITLE_BMP), screen, 0, 0, 0, 0, bmp(TITLE_BMP)->w, bmp(TITLE_BMP)->h);

   switch (player[currPlayer].color) {
          case BRIGHT_BLUE:
               strcpy(wName, "BLUE");
               strcpy(lName, "RED");
               wFnt = BLUESQR_FNT;
               lFnt = REDSQR_FNT;
          break;
          case BRIGHT_RED:
               strcpy(wName, "RED");
               strcpy(lName, "BLUE");
               wFnt = REDSQR_FNT;
               lFnt = BLUESQR_FNT;
          break;
   } // switch color

   rect(screen, 49, 189, 591, 416, DK_GREEN);
   rectfill(screen, 50, 190, 590, 415, BLACK);

   sprintf(line, "%s Wins!", wName);
   textout_centre(screen, fnt(wFnt), line, SCREEN_W/2, 200, -1);

   textout_centre(screen, fnt(LARGE_FONT), "Player", 125, 275, -1);
   textout_centre(screen, fnt(LARGE_FONT), "Matches", 320, 275, -1);
   textout_centre(screen, fnt(LARGE_FONT), "Points", 515, 275, -1);

   hline(screen, 55, 310, 585, GOLD);
   vline(screen, 222, 280, 390, GOLD);
   vline(screen, 417, 280, 390, GOLD);

   textout_centre(screen, fnt(wFnt), wName, 125, 325, -1);
   sprintf(line, "%i", player[currPlayer].roundsWon);
   textout_centre(screen, fnt(wFnt), line, 320, 325, -1);
   sprintf(line, "%i", player[currPlayer].score);
   textout_centre(screen, fnt(wFnt), line, 515, 325, -1);

   textout_centre(screen, fnt(lFnt), lName, 125, 375, -1);
   sprintf(line, "%i", player[!currPlayer].roundsWon);
   textout_centre(screen, fnt(lFnt), line, 320, 375, -1);
   sprintf(line, "%i", player[!currPlayer].score);
   textout_centre(screen, fnt(lFnt), line, 515, 375, -1);

   clear_keybuf();
   k = 0;
   do {
      k = readkey() >> 8;

      if (k == KEY_S || k == KEY_F10) {
         if (takeScreenShot())
            return;
         k = 0;
      }
   } while (!k);
   clear_keybuf();

// end showVicScreen
}

////////////////////////////////////////////////////////////
void setupMatch(void) {

   makeMap();
//   matchDone = 0;

//     currPlayer = LEFT;

   player[LEFT].angle = itofix(UP_DIR); // 223.125);
   player[RIGHT].angle = itofix(UP_DIR); // 159.375);

   player[LEFT].velocity = MAX_VELOCITY/2.0;
   player[RIGHT].velocity = MAX_VELOCITY/2.0;

   wind = (((float)rand()/(float)RAND_MAX)*(MAX_WIND*2))-MAX_WIND; // (rand()%(MAX_WIND*2)) - MAX_WIND;

   numTurns = 0;

//   if (game.drawMode & NO_REDRAW)
//      drawGame();

// end setupMatch
}

////////////////////////////////////////////////////////////
void setupTurn(void) {

//   velocity = MAX_VELOCITY/2;

   currPlayer = !currPlayer;  // It's still gonna be random, damnit!

   drawGame();

// end setupTurn
}

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
int doTurn(void) {
int res;
   if (player[currPlayer].human==HUMAN)
      res = doHumanTurn();
   else res = doCompTurn();

   numTurns++;

// end doTurn
return res;
}

////////////////////////////////////////////////////////////
int doHumanTurn(void) {
char meep = 0;
unsigned long gTime;

//   fprintf(stderr, "currPlayer is #%i (%s)\n", currPlayer, (currPlayer==LEFT)?"left":"right");

   clear_keybuf();

   do {
      gTime = gameTime;

      ff_eraseScreen();

      if (key[KEY_ESC]) {
         return PLAYER_QUIT;
      }

      if (key[KEY_LEFT]) {
         player[currPlayer].angle = (player[currPlayer].angle-ftofix(0.5));
         if (player[currPlayer].angle < itofix(0))
            player[currPlayer].angle += itofix(255);
      }

      if (key[KEY_RIGHT]) {
         player[currPlayer].angle = (player[currPlayer].angle+ftofix(0.5));
         if (player[currPlayer].angle > itofix(255)) // Using a logical and wasn't working
            player[currPlayer].angle -= 255;
      }

      if (key[KEY_UP]) {
         player[currPlayer].velocity+=0.25;
         if (player[currPlayer].velocity > MAX_VELOCITY)
            player[currPlayer].velocity = MAX_VELOCITY;
      }

      if (key[KEY_DOWN]) {
         player[currPlayer].velocity-=0.25;
         if (player[currPlayer].velocity < 0)
            player[currPlayer].velocity = 0;
      }

      if (key[KEY_PGUP])
         player[currPlayer].angle = itofix(224);
      if (key[KEY_INSERT])
         player[currPlayer].angle = itofix(160);
      if (key[KEY_HOME])
         player[currPlayer].angle = itofix(UP_DIR);
      if (key[KEY_DEL])
         player[currPlayer].angle = itofix(LEFT_DIR);
      if (key[KEY_PGDN])
         player[currPlayer].angle = itofix(RIGHT_DIR);
      if (key[KEY_END])
         player[currPlayer].velocity = MAX_VELOCITY/2.0;

      if (key[KEY_S] || key[KEY_F10])
         if (takeScreenShot())
            return 255;

      if (key[KEY_SPACE] || key[KEY_ENTER])
         meep = 1;

      drawWindBar();
      drawVelBar();
      drawGuns();
      ff_updateScreen();

      while ((gameTime - gTime) < 1)
            ;

   } while (!meep);

   clear_keybuf();

   return fireShot();

// end doHumanTurn
return 0;
}

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
int setupGame(void) {

   gameDone = matchDone = 0;

   switch (game.numPlayers) {
          case 0:
               player[LEFT].human = player[RIGHT].human = COMP;
          break;
          case 1:
               player[RIGHT].human = rand()%2;
               player[LEFT].human = !player[RIGHT].human;
          break;
          case 2:
               player[LEFT].human = player[RIGHT].human = HUMAN;
          break;
   } // end humans

   #ifdef DEBUG
      fprintf(stderr, "right=%s;left=%s\n", (player[RIGHT].human==HUMAN)?"Human":"Computer",
         (player[LEFT].human==HUMAN)?"Human":"Computer");
   #endif

   player[LEFT].roundsWon = 0;
   player[LEFT].score = 0;
   player[LEFT].color = BRIGHT_BLUE;
   player[LEFT].target = &player[RIGHT];
   player[LEFT].column = 2;

   player[RIGHT].roundsWon = 0;
   player[RIGHT].score = 0;
   player[RIGHT].color = BRIGHT_RED;
   player[RIGHT].target = &player[LEFT];
   player[RIGHT].column = 27;

//   makeMap();
   currPlayer = (rand()%2)?LEFT:RIGHT;

   clear_keybuf();

// end setupGame
return 0;
}

////////////////////////////////////////////////////////////
void shutupGame(void) {

//   destroy_bitmap(statusBar);

   clear_keybuf();

// end shutupGame
}

////////////////////////////////////////////////////////////
int vertTextHeight(FONT *font, char *string) {
// end vertTextHeight
return text_height(font)*strlen(string);
}

////////////////////////////////////////////////////////////
void vertDrawText(BITMAP *bmp, FONT *font, char *string, int x, int y, int color) {
int i;
char c[2];

   c[1] = 0;

   i = 0;
   while (string[i]) {
      c[0] = string[i];

      textout_centre(bmp, font, c, x, y, color);

      y += text_height(font);

      i++;
   } // string[i]

// end vertDrawText
}

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
void drawStatus(void) {
int i;
char line[80];
// char l2[10];

   for (i = 0; i < NUM_BOMBERS; i++) { // It's only two bombers, but what the hell????
      sprintf(line, "%i Rounds Won  %i Points    ", player[i].roundsWon, player[i].score);
      vertDrawText(double_buffer, font, line, ((SCREEN_W-COLUMN_WIDTH)*i)+(COLUMN_WIDTH/2),
         5, player[i].color);
   } // for i

//   if (player[currPlayer].human)
//      strcpy(l2, "Your Turn!");
//   else strcpy(l2, "My Turn!");
   vertDrawText(double_buffer, font, (player[currPlayer].human==HUMAN)?"Your Turn!":"My Turn!", ((SCREEN_W-COLUMN_WIDTH)*currPlayer)+(COLUMN_WIDTH/2),
      5+vertTextHeight(font, line), BRIGHT_WHITE);

// end drawStatus
}

////////////////////////////////////////////////////////////
void drawWindBar(void) {
int x, x1, y, ex;
char dir = 1;

   y = 50;
   if (wind < 0) {
      ex = x = (SCREEN_W/2)+(wind*5);
      x1 = (SCREEN_W/2);
   } else {
      dir = -1;
      ex = x1 = (SCREEN_W/2)+(wind*5);
      x = (SCREEN_W/2);
   }

   ff_pushDrawn(x-1, y-2, (x1-x)+3, 7);

//   rect(double_buffer, x, y, x1, y+2, DK_WHITE);

   hline(double_buffer, x, y, x1, DK_WHITE);
   hline(double_buffer, x, y+2, x1, DK_WHITE);

   putpixel(double_buffer, ex+dir, y-1, DK_WHITE);
   putpixel(double_buffer, ex+(dir)*2, y-2, DK_WHITE);

   putpixel(double_buffer, ex+dir, y+3, DK_WHITE);
   putpixel(double_buffer, ex+(dir)*2, y+4, DK_WHITE);

   putpixel(double_buffer, ex-dir, y+1, DK_WHITE);

// end drawWindBar
}

////////////////////////////////////////////////////////////
void drawVelBar(void) {
int x, y, i, x1;

   x = ((SCREEN_W-COLUMN_WIDTH)*currPlayer)+(COLUMN_WIDTH/4);
   y = SCREEN_H - 9 - MAX_VELBAR_H;

   ff_pushDrawn(x, y, (COLUMN_WIDTH/2)+1, MAX_VELBAR_H+5);

   rect(double_buffer, x, y, x+(COLUMN_WIDTH/2), y+MAX_VELBAR_H+4, DK_GREEN);
   rect(double_buffer, x+1, y+1, (x-1)+(COLUMN_WIDTH/2), y+MAX_VELBAR_H+3, BLACK);
   y += 2 + (MAX_VELBAR_H - (((float)player[currPlayer].velocity/(float)MAX_VELOCITY)*MAX_VELBAR_H));
   rectfill(double_buffer, x+2, y, (x-2)+(COLUMN_WIDTH/2), (SCREEN_H - 7), ORANGE);

   y = SCREEN_H - 7;
   if (currPlayer == LEFT) {
      x1 = x + (COLUMN_WIDTH/4);
      x += 2;
   } else {
     x1 = (x-2)+(COLUMN_WIDTH/2);
     x += COLUMN_WIDTH/4;
   }

//   x += 2 + ((currPlayer==LEFT) ? 2: ((COLUMN_WIDTH/4)-2));
//   x1 = (x) + (COLUMN_WIDTH/4); // + ((currPlayer==LEFT)?0:-2);

   for (i = 1; i < (MAX_VELBAR_H/10); i++) {
      hline(double_buffer, x, y-(i*(MAX_VELBAR_H/10)), x1, BLACK);
   }

// end drawBar
}

////////////////////////////////////////////////////////////////////
void drawGun(playerBoxPtr p) {
int x, y, i, nx, ny;
int verts[6] = {10, 0, -1, 3, -1, -3};
float cs, sn;

   // This doesn't call drawShot and drawGun mostly for the ff_pushDrawn redundancies
   // acquired by doing that! Grr!

   y = map[p->column].h-1;
   x = ((p->column+1)*COLUMN_WIDTH) + 2;

//   rectfill(double_buffer, x, y, x+(COLUMN_WIDTH-4), y-15, BLACK);

   ff_pushDrawn(x-3, y-15, (COLUMN_WIDTH+2), 25);

   rect(double_buffer, x, y, x+(COLUMN_WIDTH-4), y-5, p->color);

   x = x + ((COLUMN_WIDTH-4)/2);
   y -= 4;

   cs = fixtof(fcos(p->angle));
   sn = fixtof(fsin(p->angle));

//   fprintf(stderr, "cs, sn: %f, %f; angle: %f\n", cs, sn, fixtof(p->angle));

   // Rotate the gun, and translate to correct spot!
   for (i = 0; i < 6; i+=2) {
//       fprintf(stderr, "pre-vert (%i, %i); ", verts[i], verts[i+1]);
      nx = (verts[i] * cs) - (verts[i+1] * sn);
      ny = (verts[i+1] * cs) + (verts[i] * sn);
//      fprintf(stderr, "post-vert (%i, %i)\n", verts[i], verts[i+1]);
      verts[i] = x + nx;
      verts[i+1] = y + ny;
   }

   polygon(double_buffer, 3, verts, BLACK);
   line(double_buffer, verts[0], verts[1], verts[2], verts[3], p->color);
   line(double_buffer, verts[2], verts[3], verts[4], verts[5], p->color);
   line(double_buffer, verts[4], verts[5], verts[0], verts[1], p->color);

// end drawGun
}

////////////////////////////////////////////////////////////
void drawGuns(void) {

   drawGun(&player[LEFT]);
   drawGun(&player[RIGHT]);

// end drawGuns
}

////////////////////////////////////////////////////////////
void drawGame(void) {

   clear(double_buffer);

   blit(bmp(TITLE_BMP), double_buffer, bmp(TITLE_BMP)->w-COLUMN_WIDTH, 0, 0, 0, COLUMN_WIDTH, SCREEN_H);
   blit(bmp(TITLE_BMP), double_buffer, 0, 0, (SCREEN_W+1)-COLUMN_WIDTH, 0, COLUMN_WIDTH, SCREEN_H);

   drawStatus();

   drawMap();

   drawGuns();

   ff_drawDB();

// end drawGame
}

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////
int buildRand(int range) {
int res;

   if ((rand()%4)<1) {
      res = range-(rand()%(range/5));
   } else {
     res = rand()%(range-(range/5));
   }

   if (rand()%2)
      res = -res;

// end buildRand
return res;
}

////////////////////////////////////////////////////////////
void makeMap(void) {
int i;

   // Generate cityscape!

   map[0].h = MAX_HEIGHT + (rand()%(MIN_HEIGHT-MAX_HEIGHT));
   map[0].destroyed = 0; // !(rand()%10);

   for (i = 1; i < NUM_COLUMNS; i++) {

      if (!(rand()%9)) {
         map[i].h = MAX_HEIGHT + (rand()%(MIN_HEIGHT-MAX_HEIGHT));
      } else {
        map[i].h = map[i-1].h + buildRand(50);
        if (map[i].h > MIN_HEIGHT)
           map[i].h = MIN_HEIGHT;
        else if (map[i].h < MAX_HEIGHT)
             map[i].h = MAX_HEIGHT;
      }

//      map[i].destroyed = !(rand()%10);
        map[i].destroyed = 0;
   } // for i

// end makeMap
}

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
void drawMap(void) {
int i, x, y;

//   hline(double_buffer, COLUMN_WIDTH, MIN_HEIGHT, SCREEN_W-COLUMN_WIDTH, BRIGHT_WHITE);
//   hline(double_buffer, COLUMN_WIDTH, MAX_HEIGHT, SCREEN_W-COLUMN_WIDTH, BRIGHT_RED);

   for (i=0; i < NUM_COLUMNS; i++)
       if (!map[i].destroyed)
          rect(double_buffer, (i+1)*COLUMN_WIDTH, map[i].h, (i+2)*COLUMN_WIDTH, SCREEN_H-1, COLUMN_COLOR);
       else {
            // Draw some bent lines!
            // Draw left side
            x = ((i+1)*COLUMN_WIDTH)+(rand()%5) + 2;
            y = map[i].h+((map[i].h/2)+((rand()%((int)(map[i].h/1.5)))-(map[i].h/3)));
            line(double_buffer, (i+1)*COLUMN_WIDTH, map[i].h, x, y, COLUMN_COLOR);
            line(double_buffer, x, y, (i+1)*COLUMN_WIDTH, SCREEN_H-1, COLUMN_COLOR);

            // Draw right side
            x = ((i+2)*COLUMN_WIDTH)-(rand()%5) - 2;
            y = map[i].h+((map[i].h/2)+((rand()%((int)(map[i].h/1.5)))-(map[i].h/3)));
            line(double_buffer, (i+2)*COLUMN_WIDTH, map[i].h, x, y, COLUMN_COLOR);
            line(double_buffer, x, y, (i+2)*COLUMN_WIDTH, SCREEN_H-1, COLUMN_COLOR);

            // Draw top
            x = ((i+1)*COLUMN_WIDTH)+(COLUMN_WIDTH/2)+((rand()%14)-7);
            y = map[i].h+((rand()%7))+2;
            line(double_buffer, (i+1)*COLUMN_WIDTH, map[i].h, x, y, COLUMN_COLOR);
            line(double_buffer, x, y, (i+2)*COLUMN_WIDTH, map[i].h, COLUMN_COLOR);

            // Draw Bottom line
            hline(double_buffer, (i+1)*COLUMN_WIDTH, SCREEN_H-1, (i+2)*COLUMN_WIDTH, COLUMN_COLOR);
       }

// end drawMap
}

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
int fireShot(void) {
float x, y, xvel, yvel, sx, sy, ox, oy;
fixed angle;
playerBoxPtr p;
unsigned long oTime;

   cRes = 0;

   p = &player[currPlayer];
   minDistance = 32767;

   // return 1 for quit game!

   sy = y = oy = map[p->column].h-5;
   sx = x = ox = ((p->column+1)*COLUMN_WIDTH) + 2 + ((COLUMN_WIDTH-4)/2);

   p->baseHeight = sy;

//   cs = fixtof(fcos(angle));
//   sn = fixtof(fsin(angle));

   angle = p->angle;
   xvel = fixtof(fcos(angle)) * (p->velocity/5.0);
   yvel = fixtof(fsin(angle)) * (p->velocity/5.0);

   gameTime = oTime = 0;  // Start the clock over!

//   fprintf(stderr, "sx, sy (%f, %f); angle: %f; vel: %f\n", sx, sy, fixtof(p->angle), velocity);
   do {

      ff_eraseScreen();

      oTime = gameTime; // I think gameTime was being inc'd in between these equations:
      x = (xvel*oTime) + ((oTime*oTime)*(wind/(MAX_WIND*25.0))) + sx;
      y = (yvel*oTime) + ((oTime*oTime)/10) + sy;

      if (x != ox || y != oy) {
         cRes = collisionCheck(p, ox, oy, x, y);
         drawLine(ox, oy, x, y);
         angle = fatan2(ftofix(y-oy), ftofix(x-ox));
//         #ifdef DEBUG
//                fprintf(stderr, "x, y: %f, %f; ox, oy: %f, %f; angle: %f\n", x, y, ox, oy, fixtof(angle));
//         #endif
         ox = x; oy = y;
      }

      if (key[KEY_ESC]) {
         return PLAYER_QUIT;
      }

      if (key[KEY_S] || key[KEY_F10]) {
         oTime = gameTime;
         if (takeScreenShot())
            return 255;
         gameTime = oTime;
      }

      drawGun(p->target);  // Draw the other guy!
      drawGunBox(p);
      if (!hitType) // Don't draw it if it blew up! sheesh! :)
         drawShot(x, y, angle, p->color);
      ff_updateScreen();

   } while (!cRes);

   #ifdef DEBUG
          fprintf(stderr, "##### gameTime: %i\n", (int)gameTime);
   #endif // DEBUG
   doPoints(); // It's here because you can get pretty close and still go off screen

   if (cRes != OFF_SCREEN) {

      if (game.sound) {
         if (minDistance <= DEATH_DIST)
            play_sample(data[LARGE_EXPLO_SND].dat, 255, (x/(float)SCREEN_W)*255,1000, 0);
         else
            play_sample(data[SMALL_EXPLO_SND].dat, 255, (x/(float)SCREEN_W)*255,1000, 0);
      }

      drawExplosion(p, cx, cy);

      x = cx;
      y = cy;

      x -= COLUMN_WIDTH;
      x /= (float)(COLUMN_WIDTH);

      map[(int)x].h += MIN_DAMAGE+rand()%DAMAGE;
      if (map[(int)x].h > ABS_MIN_HEIGHT)
         map[(int)x].h = ABS_MIN_HEIGHT;
      map[(int)x].destroyed = 1;

      if (minDistance <= DEATH_DIST) {
         /*
         textout(screen, fnt(LARGE_FONT), "You Win!",
            (currPlayer==LEFT)?COLUMN_WIDTH+10:(SCREEN_W-COLUMN_WIDTH-10-
               text_length(fnt(LARGE_FONT), "You Win!")),
            100, player[currPlayer].color);
         tTime = gameTime;
         while ((gameTime-tTime) < 150)
               ;
         */
         return SHOT_KILLED;
      }
   }

// end fireShot
return 0;
}

////////////////////////////////////////////////////////////
void doPoints(void) {

   if (minDistance <= DEATH_DIST) {
      if (numTurns <= 1) {
         player[currPlayer].score += 100;
      } else player[currPlayer].score += 50;
   } else {
     player[currPlayer].score += (((float)DEATH_DIST/(float)minDistance)*50);
     #ifdef DEBUG
        fprintf(stderr, "minDistance: %i; DD: %i; minD/DD: %f; DD/minD: %f ", minDistance, DEATH_DIST, ((float)minDistance/(float)DEATH_DIST), ((float)DEATH_DIST/(float)minDistance));
        fprintf(stderr, "newScore: %i\n", player[currPlayer].score);
     #endif // DEBUG
   } // did not kill

// end doPoints
}

////////////////////////////////////////////////////////////
void drawLine(int ox, int oy, int x, int y) {
int minx, miny, maxx, maxy;

   if (game.drawMode == DRAW_LINE) {
      if (hitType != PLAY_ON) {
         if (ox < cx) {
            minx = ox;
            maxx = cx;
         } else {
            minx = cx;
            maxx = ox;
         }
         if (oy < cy) {
            miny = oy;
            maxy = cy;
         } else {
            miny = cy;
            maxy = oy;
         }
         line(double_buffer, ox, oy, cx, cy, player[currPlayer].color);
      } else {
         if (ox < x) {
            minx = ox;
            maxx = x;
         } else {
            minx = x;
            maxx = ox;
         }
         if (oy < y) {
            miny = oy;
            maxy = y;
         } else {
            miny = y;
            maxy = oy;
         }
         line(double_buffer, ox, oy, x, y, player[currPlayer].color);
      } // cRes

//      fprintf(stderr, "o %i, %i; true %i, %i; min %i, %i; max %i, %i\n", ox, oy, x, y, minx, miny, maxx, maxy);
      ff_pushDrawn(minx, miny, (maxx-minx)+1, (maxy-miny)+1); // Make sure it gets drawn
   } // draw line

// end drawLine
}

////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////
void checkHit(BITMAP *bmp, int x, int y, int col) {
int d;

   if (hitType != PLAY_ON)
      return;

   if ((x >= (SCREEN_W-COLUMN_WIDTH)) || (x <= COLUMN_WIDTH)) {
      cx = x;
      cy = y;
      hitType = OFF_SCREEN;
      return;
   }

   if ((d = getDistance(col, x, y)) < minDistance) {
      minDistance = d;
      minX = x;
      minY = y;
      if (minDistance <= DEATH_DIST) {
         hitType = PLAYER_KILL;
         cx = x;
         cy = y;
         return;
      }
   }

   if ((y >= map[(int)((x-COLUMN_WIDTH)/COLUMN_WIDTH)].h)) {
      hitType = BUILD_BLOW;
      cx = x;
      cy = y;
      return;
   }

// end checkHit
}

////////////////////////////////////////////////////////////////////
int collisionCheck(playerBoxPtr p, float ox, float oy, float x, float y) {
// int d;

   hitType = PLAY_ON;
   do_line(double_buffer, ox, oy, x, y, p->target->column, checkHit);
   return hitType;

// end collisionCheck
return PLAY_ON;
}

////////////////////////////////////////////////////////////////////
int getDistance(int col, int x, int y) {
int d, ex, ey;

   ey = map[col].h-3;
   ex = ((col+1)*COLUMN_WIDTH) + (COLUMN_WIDTH/2);

   d = ((ex-x)*(ex-x))+((ey-y)*(ey-y));

// end getDistance
return d;
}

////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////
void drawShot(int x, int y, fixed angle, int color) {
int i, nx, ny, maxx=0, maxy=0, minx=640, miny=480;
int verts[6] = {10, 0, -1, 3, -1, -3};
float cs, sn;

   cs = fixtof(fcos(angle));
   sn = fixtof(fsin(angle));

   // Rotate the gun, and translate to correct spot!
   for (i = 0; i < 6; i+=2) {
      nx = (verts[i] * cs) - (verts[i+1] * sn);
      ny = (verts[i+1] * cs) + (verts[i] * sn);

      verts[i] = x + nx;
      verts[i+1] = y + ny;

      if (verts[i] < minx)
         minx = verts[i];
      if (verts[i] > maxx)
           maxx = verts[i];

      if (verts[i+1] < miny)
         miny = verts[i+1];
      if (verts[i+1] > maxy)
           maxy = verts[i+1];

//           fprintf(stderr, "\tvert! (%i, %i)\n", verts[i], verts[i+1]);
   }
   maxx++;
   maxy++;

//   fprintf(stderr, "grr (%i, %i); min (%i, %i); max (%i, %i)\n", x, y, minx, miny, maxx, maxy);

   if (game.drawMode == DRAW_DOT)
      putpixel(double_buffer, x, y, color);

   ff_pushDrawn(minx, miny, maxx-minx, maxy-miny);

   polygon(double_buffer, 3, verts, BLACK);
   line(double_buffer, verts[0], verts[1], verts[2], verts[3], color);
   line(double_buffer, verts[2], verts[3], verts[4], verts[5], color);
   line(double_buffer, verts[4], verts[5], verts[0], verts[1], color);

// end drawShot
}

////////////////////////////////////////////////////////////////////
void drawGunBox(playerBoxPtr p) {
int x, y;

   y = map[p->column].h-1;
   x = ((p->column+1)*COLUMN_WIDTH) + 2;

   ff_pushDrawn(x, y-5, (COLUMN_WIDTH-3), 6);

   rect(double_buffer, x, y, x+(COLUMN_WIDTH-4), y-5, p->color);

// end drawGunBox
}

////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////
void drawExplosion(playerBoxPtr p, float x, float y) {
RLE_SPRITE *nasty;
int dx, dy;
unsigned long meep, // This was made cause of probs with gameTime changing unexpectedly
         tTime;

   meep = gameTime = 0;
   do {
      ff_eraseScreen();

      drawGun(p->target);  // Draw the other guy!
      drawGunBox(p);

      nasty = explosion[meep];

      dx = x - (nasty->w/2);
      dy = y - (nasty->h/2);

      if (key[KEY_S] || key[KEY_F10]) {
         tTime = gameTime;
         takeScreenShot();
         gameTime = tTime;
      }

      ff_pushDrawn(dx, dy, nasty->w, nasty->h);
      draw_rle_sprite(double_buffer, nasty, dx, dy);

      ff_updateScreen();

      if (gameTime > meep)
         meep++;

   } while (meep < NUM_EXPLO_FRAMES);

//end drawExplosion
}

////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////
int takeScreenShot(void) {
int err = 0, i;
char filename[80];
BITMAP *bmp;

   for (i = 0; i < 100; i++) { // No one's ever gonna take that many pictures :)
       sprintf(filename, "%sbomber%i.bmp", game.mainPath, i);
//       strcpy(filename, game.mainPath);
//       strcat(filename, "bomber");
//       itoa(i, &filename[4], 10);
//       strcat(filename, ".bmp");

       if (!__file_exists(filename))
          break;

   } // for i

   #ifdef DEBUG
          fprintf(stderr, "pic: '%s'\n", filename);
   #endif

   bmp = create_sub_bitmap(screen, 0, 0, SCREEN_W, SCREEN_H);
   err = save_bitmap(filename, bmp, data[GAME_PAL].dat);
   destroy_bitmap(bmp);

// end takeScreenShot
return err;
}

